/*
 * Copyright (C) 2014 The Android Open Source Project
 * Copyright (c) 1997, 2014, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package javax.crypto;

import java.util.*;
import java.util.regex.*;
import java.security.*;
import javax.crypto.spec.*;
import java.nio.ReadOnlyBufferException;
import java.nio.ByteBuffer;
import java.security.spec.AlgorithmParameterSpec;
import java.security.Provider.Service;

/**
 * This class provides the functionality of a cryptographic cipher for
 * encryption and decryption. It forms the core of the Java Cryptographic
 * Extension (JCE) framework.
 *
 * <p>In order to create a Cipher object, the application calls the
 * Cipher's <code>getInstance</code> method, and passes the name of the
 * requested <i>transformation</i> to it. Optionally, the name of a provider
 * may be specified.
 *
 * <p>A <i>transformation</i> is a string that describes the operation (or
 * set of operations) to be performed on the given input, to produce some
 * output. A transformation always includes the name of a cryptographic
 * algorithm (e.g., <i>DES</i>), and may be followed by a feedback mode and
 * padding scheme.
 *
 * <p> A transformation is of the form:
 *
 * <ul>
 * <li>"<i>algorithm/mode/padding</i>" or
 *
 * <li>"<i>algorithm</i>"
 * </ul>
 *
 * <P> (in the latter case,
 * provider-specific default values for the mode and padding scheme are used).
 * For example, the following is a valid transformation:
 *
 * <pre>
 *     Cipher c = Cipher.getInstance("<i>DES/CBC/PKCS5Padding</i>");
 * </pre>
 *
 * Using modes such as <code>CFB</code> and <code>OFB</code>, block
 * ciphers can encrypt data in units smaller than the cipher's actual
 * block size.  When requesting such a mode, you may optionally specify
 * the number of bits to be processed at a time by appending this number
 * to the mode name as shown in the "<code>DES/CFB8/NoPadding</code>" and
 * "<code>DES/OFB32/PKCS5Padding</code>" transformations. If no such
 * number is specified, a provider-specific default is used. (For
 * example, the SunJCE provider uses a default of 64 bits for DES.)
 * Thus, block ciphers can be turned into byte-oriented stream ciphers by
 * using an 8 bit mode such as CFB8 or OFB8.
 * <p>
 * Modes such as Authenticated Encryption with Associated Data (AEAD)
 * provide authenticity assurances for both confidential data and
 * Additional Associated Data (AAD) that is not encrypted.  (Please see
 * <a href="http://www.ietf.org/rfc/rfc5116.txt"> RFC 5116 </a> for more
 * information on AEAD and AEAD algorithms such as GCM/CCM.) Both
 * confidential and AAD data can be used when calculating the
 * authentication tag (similar to a {@link Mac}).  This tag is appended
 * to the ciphertext during encryption, and is verified on decryption.
 * <p>
 * AEAD modes such as GCM/CCM perform all AAD authenticity calculations
 * before starting the ciphertext authenticity calculations.  To avoid
 * implementations having to internally buffer ciphertext, all AAD data
 * must be supplied to GCM/CCM implementations (via the {@code
 * updateAAD} methods) <b>before</b> the ciphertext is processed (via
 * the {@code update} and {@code doFinal} methods).
 * <p>
 * Note that GCM mode has a uniqueness requirement on IVs used in
 * encryption with a given key. When IVs are repeated for GCM
 * encryption, such usages are subject to forgery attacks. Thus, after
 * each encryption operation using GCM mode, callers should re-initialize
 * the cipher objects with GCM parameters which has a different IV value.
 * <pre>
 *     GCMParameterSpec s = ...;
 *     cipher.init(..., s);
 *
 *     // If the GCM parameters were generated by the provider, it can
 *     // be retrieved by:
 *     // cipher.getParameters().getParameterSpec(GCMParameterSpec.class);
 *
 *     cipher.updateAAD(...);  // AAD
 *     cipher.update(...);     // Multi-part update
 *     cipher.doFinal(...);    // conclusion of operation
 *
 *     // Use a different IV value for every encryption
 *     byte[] newIv = ...;
 *     s = new GCMParameterSpec(s.getTLen(), newIv);
 *     cipher.init(..., s);
 *     ...
 *
 * </pre>
 * <p> Android provides the following <code>Cipher</code> transformations:
 * <table>
 *   <thead>
 *     <tr>
 *       <th>Algorithm</th>
 *       <th>Modes</th>
 *       <th>Paddings</th>
 *       <th>Supported API Levels</th>
 *       <th>Notes</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td rowspan="2"><span style="white-space: nowrap">AES</span></td>
 *       <td><span style="white-space: nowrap">CBC</span><br><span style="white-space: nowrap">CFB</span><br><span style="white-space: nowrap">CTR</span><br><span style="white-space: nowrap">CTS</span><br><span style="white-space: nowrap">ECB</span><br><span style="white-space: nowrap">OFB</span></td>
 *       <td><span style="white-space: nowrap">ISO10126Padding</span><br><span style="white-space: nowrap">NoPadding</span><br><span style="white-space: nowrap">PKCS5Padding</span></td>
 *       <td><span style="white-space: nowrap">1+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">GCM</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span></td>
 *       <td><span style="white-space: nowrap">10+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td rowspan="2"><span style="white-space: nowrap">AES_128</span></td>
 *       <td><span style="white-space: nowrap">CBC</span><br><span style="white-space: nowrap">ECB</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span><br><span style="white-space: nowrap">PKCS5Padding</span></td>
 *       <td><span style="white-space: nowrap">26+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">GCM</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span></td>
 *       <td><span style="white-space: nowrap">26+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td rowspan="2"><span style="white-space: nowrap">AES_256</span></td>
 *       <td><span style="white-space: nowrap">CBC</span><br><span style="white-space: nowrap">ECB</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span><br><span style="white-space: nowrap">PKCS5Padding</span></td>
 *       <td><span style="white-space: nowrap">26+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">GCM</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span></td>
 *       <td><span style="white-space: nowrap">26+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td rowspan="2"><span style="white-space: nowrap">ARC4</span></td>
 *       <td><span style="white-space: nowrap">ECB</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span></td>
 *       <td><span style="white-space: nowrap">10+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">NONE</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span></td>
 *       <td><span style="white-space: nowrap">28+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">BLOWFISH</span></td>
 *       <td><span style="white-space: nowrap">CBC</span><br><span style="white-space: nowrap">CFB</span><br><span style="white-space: nowrap">CTR</span><br><span style="white-space: nowrap">CTS</span><br><span style="white-space: nowrap">ECB</span><br><span style="white-space: nowrap">OFB</span></td>
 *       <td><span style="white-space: nowrap">ISO10126Padding</span><br><span style="white-space: nowrap">NoPadding</span><br><span style="white-space: nowrap">PKCS5Padding</span></td>
 *       <td><span style="white-space: nowrap">10+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">ChaCha20</span></td>
 *       <td><span style="white-space: nowrap">NONE</span><br><span style="white-space: nowrap">Poly1305</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span></td>
 *       <td><span style="white-space: nowrap">28+</span></td>
 *       <td>ChaCha with 20 rounds, 96-bit nonce, and 32-bit counter as described in RFC 7539.</td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">DES</span></td>
 *       <td><span style="white-space: nowrap">CBC</span><br><span style="white-space: nowrap">CFB</span><br><span style="white-space: nowrap">CTR</span><br><span style="white-space: nowrap">CTS</span><br><span style="white-space: nowrap">ECB</span><br><span style="white-space: nowrap">OFB</span></td>
 *       <td><span style="white-space: nowrap">ISO10126Padding</span><br><span style="white-space: nowrap">NoPadding</span><br><span style="white-space: nowrap">PKCS5Padding</span></td>
 *       <td><span style="white-space: nowrap">1+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">DESede</span></td>
 *       <td><span style="white-space: nowrap">CBC</span><br><span style="white-space: nowrap">CFB</span><br><span style="white-space: nowrap">CTR</span><br><span style="white-space: nowrap">CTS</span><br><span style="white-space: nowrap">ECB</span><br><span style="white-space: nowrap">OFB</span></td>
 *       <td><span style="white-space: nowrap">ISO10126Padding</span><br><span style="white-space: nowrap">NoPadding</span><br><span style="white-space: nowrap">PKCS5Padding</span></td>
 *       <td><span style="white-space: nowrap">1+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td rowspan="3"><span style="white-space: nowrap">RSA</span></td>
 *       <td rowspan="3"><span style="white-space: nowrap">ECB</span><br><span style="white-space: nowrap">NONE</span></td>
 *       <td><span style="white-space: nowrap">NoPadding</span><br><span style="white-space: nowrap">OAEPPadding</span><br><span style="white-space: nowrap">PKCS1Padding</span></td>
 *       <td><span style="white-space: nowrap">1+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">OAEPwithSHA-1andMGF1Padding</span><br><span style="white-space: nowrap">OAEPwithSHA-256andMGF1Padding</span></td>
 *       <td><span style="white-space: nowrap">10+</span></td>
 *       <td></td>
 *     </tr>
 *     <tr>
 *       <td><span style="white-space: nowrap">OAEPwithSHA-224andMGF1Padding</span><br><span style="white-space: nowrap">OAEPwithSHA-384andMGF1Padding</span><br><span style="white-space: nowrap">OAEPwithSHA-512andMGF1Padding</span></td>
 *       <td><span style="white-space: nowrap">23+</span></td>
 *       <td></td>
 *     </tr>
 *   </tbody>
 * </table>
 *
 * These transformations are described in the
 * <a href="https://docs.oracle.com/javase/8/docs/technotes/guides/security/StandardNames.html#Cipher">
 * Cipher section</a> of the
 * Java Cryptography Architecture Standard Algorithm Name Documentation.
 *
 * @author Jan Luehe
 * @see KeyGenerator
 * @see SecretKey
 * @since 1.4
 * @apiSince 1
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class Cipher {

/**
 * Creates a Cipher object.
 *
 * @param cipherSpi the delegate
 * @param provider the provider
 * @param transformation the transformation
 * @apiSince 1
 */

protected Cipher(javax.crypto.CipherSpi cipherSpi, java.security.Provider provider, java.lang.String transformation) { throw new RuntimeException("Stub!"); }

/**
 * Returns a <code>Cipher</code> object that implements the specified
 * transformation.
 *
 * <p> This method traverses the list of registered security Providers,
 * starting with the most preferred Provider.
 * A new Cipher object encapsulating the
 * CipherSpi implementation from the first
 * Provider that supports the specified algorithm is returned.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link Security#getProviders() Security.getProviders()} method.
 *
 * @param transformation the name of the transformation, e.g.,
 * <i>DES/CBC/PKCS5Padding</i>.
 * See the Cipher section in the <a href=
 *   "https://docs.oracle.com/javase/8/docs/technotes/guides/security/StandardNames.html#Cipher">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard transformation names.
 *
 * @return a cipher that implements the requested transformation.
 *
 * @exception NoSuchAlgorithmException if <code>transformation</code>
 *          is null, empty, in an invalid format,
 *          or if no Provider supports a CipherSpi implementation for the
 *          specified algorithm.
 *
 * @exception NoSuchPaddingException if <code>transformation</code>
 *          contains a padding scheme that is not available.
 *
 * @see java.security.Provider
 * @apiSince 1
 */

public static final javax.crypto.Cipher getInstance(java.lang.String transformation) throws java.security.NoSuchAlgorithmException, javax.crypto.NoSuchPaddingException { throw new RuntimeException("Stub!"); }

/**
 * Returns a <code>Cipher</code> object that implements the specified
 * transformation.
 *
 * <p> A new Cipher object encapsulating the
 * CipherSpi implementation from the specified provider
 * is returned.  The specified provider must be registered
 * in the security provider list.
 *
 * <p> Note that the list of registered providers may be retrieved via
 * the {@link Security#getProviders() Security.getProviders()} method.
 *
 * @param transformation the name of the transformation,
 * e.g., <i>DES/CBC/PKCS5Padding</i>.
 * See the Cipher section in the <a href=
 *   "https://docs.oracle.com/javase/8/docs/technotes/guides/security/StandardNames.html#Cipher">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard transformation names.
 *
 * @param provider the name of the provider.
 *
 * @return a cipher that implements the requested transformation.
 *
 * @exception NoSuchAlgorithmException if <code>transformation</code>
 *          is null, empty, in an invalid format,
 *          or if a CipherSpi implementation for the specified algorithm
 *          is not available from the specified provider.
 *
 * @exception NoSuchProviderException if the specified provider is not
 *          registered in the security provider list.
 *
 * @exception NoSuchPaddingException if <code>transformation</code>
 *          contains a padding scheme that is not available.
 *
 * @exception IllegalArgumentException if the <code>provider</code>
 *          is null or empty.
 *
 * @see java.security.Provider
 * @apiSince 1
 */

public static final javax.crypto.Cipher getInstance(java.lang.String transformation, java.lang.String provider) throws java.security.NoSuchAlgorithmException, javax.crypto.NoSuchPaddingException, java.security.NoSuchProviderException { throw new RuntimeException("Stub!"); }

/**
 * Returns a <code>Cipher</code> object that implements the specified
 * transformation.
 *
 * <p> A new Cipher object encapsulating the
 * CipherSpi implementation from the specified Provider
 * object is returned.  Note that the specified Provider object
 * does not have to be registered in the provider list.
 *
 * @param transformation the name of the transformation,
 * e.g., <i>DES/CBC/PKCS5Padding</i>.
 * See the Cipher section in the <a href=
 *   "https://docs.oracle.com/javase/8/docs/technotes/guides/security/StandardNames.html#Cipher">
 * Java Cryptography Architecture Standard Algorithm Name Documentation</a>
 * for information about standard transformation names.
 *
 * @param provider the provider.
 *
 * @return a cipher that implements the requested transformation.
 *
 * @exception NoSuchAlgorithmException if <code>transformation</code>
 *          is null, empty, in an invalid format,
 *          or if a CipherSpi implementation for the specified algorithm
 *          is not available from the specified Provider object.
 *
 * @exception NoSuchPaddingException if <code>transformation</code>
 *          contains a padding scheme that is not available.
 *
 * @exception IllegalArgumentException if the <code>provider</code>
 *          is null.
 *
 * @see java.security.Provider
 * @apiSince 1
 */

public static final javax.crypto.Cipher getInstance(java.lang.String transformation, java.security.Provider provider) throws java.security.NoSuchAlgorithmException, javax.crypto.NoSuchPaddingException { throw new RuntimeException("Stub!"); }

/**
 * Returns the provider of this <code>Cipher</code> object.
 *
 * @return the provider of this <code>Cipher</code> object
 * @apiSince 1
 */

public final java.security.Provider getProvider() { throw new RuntimeException("Stub!"); }

/**
 * Returns the algorithm name of this <code>Cipher</code> object.
 *
 * <p>This is the same name that was specified in one of the
 * <code>getInstance</code> calls that created this <code>Cipher</code>
 * object..
 *
 * @return the algorithm name of this <code>Cipher</code> object.
 * @apiSince 1
 */

public final java.lang.String getAlgorithm() { throw new RuntimeException("Stub!"); }

/**
 * Returns the block size (in bytes).
 *
 * @return the block size (in bytes), or 0 if the underlying algorithm is
 * not a block cipher
 * @apiSince 1
 */

public final int getBlockSize() { throw new RuntimeException("Stub!"); }

/**
 * Returns the length in bytes that an output buffer would need to be in
 * order to hold the result of the next <code>update</code> or
 * <code>doFinal</code> operation, given the input length
 * <code>inputLen</code> (in bytes).
 *
 * <p>This call takes into account any unprocessed (buffered) data from a
 * previous <code>update</code> call, padding, and AEAD tagging.
 *
 * <p>The actual output length of the next <code>update</code> or
 * <code>doFinal</code> call may be smaller than the length returned by
 * this method.
 *
 * @param inputLen the input length (in bytes)
 *
 * @return the required output buffer size (in bytes)
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not yet been initialized)
 * @apiSince 1
 */

public final int getOutputSize(int inputLen) { throw new RuntimeException("Stub!"); }

/**
 * Returns the initialization vector (IV) in a new buffer.
 *
 * <p>This is useful in the case where a random IV was created,
 * or in the context of password-based encryption or
 * decryption, where the IV is derived from a user-supplied password.
 *
 * @return the initialization vector in a new buffer, or null if the
 * underlying algorithm does not use an IV, or if the IV has not yet
 * been set.
 * @apiSince 1
 */

public final byte[] getIV() { throw new RuntimeException("Stub!"); }

/**
 * Returns the parameters used with this cipher.
 *
 * <p>The returned parameters may be the same that were used to initialize
 * this cipher, or may contain a combination of default and random
 * parameter values used by the underlying cipher implementation if this
 * cipher requires algorithm parameters but was not initialized with any.
 *
 * @return the parameters used with this cipher, or null if this cipher
 * does not use any parameters.
 * @apiSince 1
 */

public final java.security.AlgorithmParameters getParameters() { throw new RuntimeException("Stub!"); }

/**
 * Returns the exemption mechanism object used with this cipher.
 *
 * @return the exemption mechanism object used with this cipher, or
 * null if this cipher does not use any exemption mechanism.
 * @apiSince 1
 */

public final javax.crypto.ExemptionMechanism getExemptionMechanism() { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with a key.
 *
 * <p>The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping or key unwrapping, depending
 * on the value of <code>opmode</code>.
 *
 * <p>If this cipher requires any algorithm parameters that cannot be
 * derived from the given <code>key</code>, the underlying cipher
 * implementation is supposed to generate the required parameters itself
 * (using provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an
 * <code>InvalidKeyException</code> if it is being
 * initialized for decryption or key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them using the {@link java.security.SecureRandom}
 * implementation of the highest-priority
 * installed provider as the source of randomness.
 * (If none of the installed providers supply an implementation of
 * SecureRandom, a system-provided source of randomness will be used.)
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of
 * the following:
 * <code>ENCRYPT_MODE</code>, <code>DECRYPT_MODE</code>,
 * <code>WRAP_MODE</code> or <code>UNWRAP_MODE</code>)
 * @param key the key
 *
 * @exception InvalidKeyException if the given key is inappropriate for
 * initializing this cipher, or requires
 * algorithm parameters that cannot be
 * determined from the given key, or if the given key has a keysize that
 * exceeds the maximum allowable keysize (as determined from the
 * configured jurisdiction policy files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.Key key) throws java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with a key and a source of randomness.
 *
 * <p>The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping or  key unwrapping, depending
 * on the value of <code>opmode</code>.
 *
 * <p>If this cipher requires any algorithm parameters that cannot be
 * derived from the given <code>key</code>, the underlying cipher
 * implementation is supposed to generate the required parameters itself
 * (using provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an
 * <code>InvalidKeyException</code> if it is being
 * initialized for decryption or key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them from <code>random</code>.
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of the
 * following:
 * <code>ENCRYPT_MODE</code>, <code>DECRYPT_MODE</code>,
 * <code>WRAP_MODE</code> or <code>UNWRAP_MODE</code>)
 * @param key the encryption key
 * @param random the source of randomness
 *
 * @exception InvalidKeyException if the given key is inappropriate for
 * initializing this cipher, or requires
 * algorithm parameters that cannot be
 * determined from the given key, or if the given key has a keysize that
 * exceeds the maximum allowable keysize (as determined from the
 * configured jurisdiction policy files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.Key key, java.security.SecureRandom random) throws java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with a key and a set of algorithm
 * parameters.
 *
 * <p>The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping or  key unwrapping, depending
 * on the value of <code>opmode</code>.
 *
 * <p>If this cipher requires any algorithm parameters and
 * <code>params</code> is null, the underlying cipher implementation is
 * supposed to generate the required parameters itself (using
 * provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an
 * <code>InvalidAlgorithmParameterException</code> if it is being
 * initialized for decryption or key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them using the {@link java.security.SecureRandom}
 * implementation of the highest-priority
 * installed provider as the source of randomness.
 * (If none of the installed providers supply an implementation of
 * SecureRandom, a system-provided source of randomness will be used.)
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of the
 * following:
 * <code>ENCRYPT_MODE</code>, <code>DECRYPT_MODE</code>,
 * <code>WRAP_MODE</code> or <code>UNWRAP_MODE</code>)
 * @param key the encryption key
 * @param params the algorithm parameters
 *
 * @exception InvalidKeyException if the given key is inappropriate for
 * initializing this cipher, or its keysize exceeds the maximum allowable
 * keysize (as determined from the configured jurisdiction policy files).
 * @exception InvalidAlgorithmParameterException if the given algorithm
 * parameters are inappropriate for this cipher,
 * or this cipher requires
 * algorithm parameters and <code>params</code> is null, or the given
 * algorithm parameters imply a cryptographic strength that would exceed
 * the legal limits (as determined from the configured jurisdiction
 * policy files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.Key key, java.security.spec.AlgorithmParameterSpec params) throws java.security.InvalidAlgorithmParameterException, java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with a key, a set of algorithm
 * parameters, and a source of randomness.
 *
 * <p>The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping or  key unwrapping, depending
 * on the value of <code>opmode</code>.
 *
 * <p>If this cipher requires any algorithm parameters and
 * <code>params</code> is null, the underlying cipher implementation is
 * supposed to generate the required parameters itself (using
 * provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an
 * <code>InvalidAlgorithmParameterException</code> if it is being
 * initialized for decryption or key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them from <code>random</code>.
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of the
 * following:
 * <code>ENCRYPT_MODE</code>, <code>DECRYPT_MODE</code>,
 * <code>WRAP_MODE</code> or <code>UNWRAP_MODE</code>)
 * @param key the encryption key
 * @param params the algorithm parameters
 * @param random the source of randomness
 *
 * @exception InvalidKeyException if the given key is inappropriate for
 * initializing this cipher, or its keysize exceeds the maximum allowable
 * keysize (as determined from the configured jurisdiction policy files).
 * @exception InvalidAlgorithmParameterException if the given algorithm
 * parameters are inappropriate for this cipher,
 * or this cipher requires
 * algorithm parameters and <code>params</code> is null, or the given
 * algorithm parameters imply a cryptographic strength that would exceed
 * the legal limits (as determined from the configured jurisdiction
 * policy files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.Key key, java.security.spec.AlgorithmParameterSpec params, java.security.SecureRandom random) throws java.security.InvalidAlgorithmParameterException, java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with a key and a set of algorithm
 * parameters.
 *
 * <p>The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping or  key unwrapping, depending
 * on the value of <code>opmode</code>.
 *
 * <p>If this cipher requires any algorithm parameters and
 * <code>params</code> is null, the underlying cipher implementation is
 * supposed to generate the required parameters itself (using
 * provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an
 * <code>InvalidAlgorithmParameterException</code> if it is being
 * initialized for decryption or key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them using the {@link java.security.SecureRandom}
 * implementation of the highest-priority
 * installed provider as the source of randomness.
 * (If none of the installed providers supply an implementation of
 * SecureRandom, a system-provided source of randomness will be used.)
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of the
 * following: <code>ENCRYPT_MODE</code>,
 * <code>DECRYPT_MODE</code>, <code>WRAP_MODE</code>
 * or <code>UNWRAP_MODE</code>)
 * @param key the encryption key
 * @param params the algorithm parameters
 *
 * @exception InvalidKeyException if the given key is inappropriate for
 * initializing this cipher, or its keysize exceeds the maximum allowable
 * keysize (as determined from the configured jurisdiction policy files).
 * @exception InvalidAlgorithmParameterException if the given algorithm
 * parameters are inappropriate for this cipher,
 * or this cipher requires
 * algorithm parameters and <code>params</code> is null, or the given
 * algorithm parameters imply a cryptographic strength that would exceed
 * the legal limits (as determined from the configured jurisdiction
 * policy files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.Key key, java.security.AlgorithmParameters params) throws java.security.InvalidAlgorithmParameterException, java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with a key, a set of algorithm
 * parameters, and a source of randomness.
 *
 * <p>The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping or  key unwrapping, depending
 * on the value of <code>opmode</code>.
 *
 * <p>If this cipher requires any algorithm parameters and
 * <code>params</code> is null, the underlying cipher implementation is
 * supposed to generate the required parameters itself (using
 * provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an
 * <code>InvalidAlgorithmParameterException</code> if it is being
 * initialized for decryption or key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them from <code>random</code>.
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of the
 * following: <code>ENCRYPT_MODE</code>,
 * <code>DECRYPT_MODE</code>, <code>WRAP_MODE</code>
 * or <code>UNWRAP_MODE</code>)
 * @param key the encryption key
 * @param params the algorithm parameters
 * @param random the source of randomness
 *
 * @exception InvalidKeyException if the given key is inappropriate for
 * initializing this cipher, or its keysize exceeds the maximum allowable
 * keysize (as determined from the configured jurisdiction policy files).
 * @exception InvalidAlgorithmParameterException if the given algorithm
 * parameters are inappropriate for this cipher,
 * or this cipher requires
 * algorithm parameters and <code>params</code> is null, or the given
 * algorithm parameters imply a cryptographic strength that would exceed
 * the legal limits (as determined from the configured jurisdiction
 * policy files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.Key key, java.security.AlgorithmParameters params, java.security.SecureRandom random) throws java.security.InvalidAlgorithmParameterException, java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with the public key from the given certificate.
 * <p> The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping or  key unwrapping, depending
 * on the value of <code>opmode</code>.
 *
 * <p>If the certificate is of type X.509 and has a <i>key usage</i>
 * extension field marked as critical, and the value of the <i>key usage</i>
 * extension field implies that the public key in
 * the certificate and its corresponding private key are not
 * supposed to be used for the operation represented by the value
 * of <code>opmode</code>,
 * an <code>InvalidKeyException</code>
 * is thrown.
 *
 * <p> If this cipher requires any algorithm parameters that cannot be
 * derived from the public key in the given certificate, the underlying
 * cipher
 * implementation is supposed to generate the required parameters itself
 * (using provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an <code>
 * InvalidKeyException</code> if it is being initialized for decryption or
 * key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them using the
 * <code>SecureRandom</code>
 * implementation of the highest-priority
 * installed provider as the source of randomness.
 * (If none of the installed providers supply an implementation of
 * SecureRandom, a system-provided source of randomness will be used.)
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of the
 * following:
 * <code>ENCRYPT_MODE</code>, <code>DECRYPT_MODE</code>,
 * <code>WRAP_MODE</code> or <code>UNWRAP_MODE</code>)
 * @param certificate the certificate
 *
 * @exception InvalidKeyException if the public key in the given
 * certificate is inappropriate for initializing this cipher, or this
 * cipher requires algorithm parameters that cannot be determined from the
 * public key in the given certificate, or the keysize of the public key
 * in the given certificate has a keysize that exceeds the maximum
 * allowable keysize (as determined by the configured jurisdiction policy
 * files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.cert.Certificate certificate) throws java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Initializes this cipher with the public key from the given certificate
 * and
 * a source of randomness.
 *
 * <p>The cipher is initialized for one of the following four operations:
 * encryption, decryption, key wrapping
 * or key unwrapping, depending on
 * the value of <code>opmode</code>.
 *
 * <p>If the certificate is of type X.509 and has a <i>key usage</i>
 * extension field marked as critical, and the value of the <i>key usage</i>
 * extension field implies that the public key in
 * the certificate and its corresponding private key are not
 * supposed to be used for the operation represented by the value of
 * <code>opmode</code>,
 * an <code>InvalidKeyException</code>
 * is thrown.
 *
 * <p>If this cipher requires any algorithm parameters that cannot be
 * derived from the public key in the given <code>certificate</code>,
 * the underlying cipher
 * implementation is supposed to generate the required parameters itself
 * (using provider-specific default or random values) if it is being
 * initialized for encryption or key wrapping, and raise an
 * <code>InvalidKeyException</code> if it is being
 * initialized for decryption or key unwrapping.
 * The generated parameters can be retrieved using
 * {@link #getParameters() getParameters} or
 * {@link #getIV() getIV} (if the parameter is an IV).
 *
 * <p>If this cipher requires algorithm parameters that cannot be
 * derived from the input parameters, and there are no reasonable
 * provider-specific default values, initialization will
 * necessarily fail.
 *
 * <p>If this cipher (including its underlying feedback or padding scheme)
 * requires any random bytes (e.g., for parameter generation), it will get
 * them from <code>random</code>.
 *
 * <p>Note that when a Cipher object is initialized, it loses all
 * previously-acquired state. In other words, initializing a Cipher is
 * equivalent to creating a new instance of that Cipher and initializing
 * it.
 *
 * @param opmode the operation mode of this cipher (this is one of the
 * following:
 * <code>ENCRYPT_MODE</code>, <code>DECRYPT_MODE</code>,
 * <code>WRAP_MODE</code> or <code>UNWRAP_MODE</code>)
 * @param certificate the certificate
 * @param random the source of randomness
 *
 * @exception InvalidKeyException if the public key in the given
 * certificate is inappropriate for initializing this cipher, or this
 * cipher
 * requires algorithm parameters that cannot be determined from the
 * public key in the given certificate, or the keysize of the public key
 * in the given certificate has a keysize that exceeds the maximum
 * allowable keysize (as determined by the configured jurisdiction policy
 * files).
 * @throws UnsupportedOperationException if (@code opmode} is
 * {@code WRAP_MODE} or {@code UNWRAP_MODE} but the mode is not implemented
 * by the underlying {@code CipherSpi}.
 * @apiSince 1
 */

public final void init(int opmode, java.security.cert.Certificate certificate, java.security.SecureRandom random) throws java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Continues a multiple-part encryption or decryption operation
 * (depending on how this cipher was initialized), processing another data
 * part.
 *
 * <p>The bytes in the <code>input</code> buffer are processed, and the
 * result is stored in a new buffer.
 *
 * <p>If <code>input</code> has a length of zero, this method returns
 * <code>null</code>.
 *
 * @param input the input buffer
 *
 * @return the new buffer with the result, or null if the underlying
 * cipher is a block cipher and the input data is too short to result in a
 * new block.
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @apiSince 1
 */

public final byte[] update(byte[] input) { throw new RuntimeException("Stub!"); }

/**
 * Continues a multiple-part encryption or decryption operation
 * (depending on how this cipher was initialized), processing another data
 * part.
 *
 * <p>The first <code>inputLen</code> bytes in the <code>input</code>
 * buffer, starting at <code>inputOffset</code> inclusive, are processed,
 * and the result is stored in a new buffer.
 *
 * <p>If <code>inputLen</code> is zero, this method returns
 * <code>null</code>.
 *
 * @param input the input buffer
 * @param inputOffset the offset in <code>input</code> where the input
 * starts
 * @param inputLen the input length
 *
 * @return the new buffer with the result, or null if the underlying
 * cipher is a block cipher and the input data is too short to result in a
 * new block.
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @apiSince 1
 */

public final byte[] update(byte[] input, int inputOffset, int inputLen) { throw new RuntimeException("Stub!"); }

/**
 * Continues a multiple-part encryption or decryption operation
 * (depending on how this cipher was initialized), processing another data
 * part.
 *
 * <p>The first <code>inputLen</code> bytes in the <code>input</code>
 * buffer, starting at <code>inputOffset</code> inclusive, are processed,
 * and the result is stored in the <code>output</code> buffer.
 *
 * <p>If the <code>output</code> buffer is too small to hold the result,
 * a <code>ShortBufferException</code> is thrown. In this case, repeat this
 * call with a larger output buffer. Use
 * {@link #getOutputSize(int) getOutputSize} to determine how big
 * the output buffer should be.
 *
 * <p>If <code>inputLen</code> is zero, this method returns
 * a length of zero.
 *
 * <p>Note: this method should be copy-safe, which means the
 * <code>input</code> and <code>output</code> buffers can reference
 * the same byte array and no unprocessed input data is overwritten
 * when the result is copied into the output buffer.
 *
 * @param input the input buffer
 * @param inputOffset the offset in <code>input</code> where the input
 * starts
 * @param inputLen the input length
 * @param output the buffer for the result
 *
 * @return the number of bytes stored in <code>output</code>
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception ShortBufferException if the given output buffer is too small
 * to hold the result
 * @apiSince 1
 */

public final int update(byte[] input, int inputOffset, int inputLen, byte[] output) throws javax.crypto.ShortBufferException { throw new RuntimeException("Stub!"); }

/**
 * Continues a multiple-part encryption or decryption operation
 * (depending on how this cipher was initialized), processing another data
 * part.
 *
 * <p>The first <code>inputLen</code> bytes in the <code>input</code>
 * buffer, starting at <code>inputOffset</code> inclusive, are processed,
 * and the result is stored in the <code>output</code> buffer, starting at
 * <code>outputOffset</code> inclusive.
 *
 * <p>If the <code>output</code> buffer is too small to hold the result,
 * a <code>ShortBufferException</code> is thrown. In this case, repeat this
 * call with a larger output buffer. Use
 * {@link #getOutputSize(int) getOutputSize} to determine how big
 * the output buffer should be.
 *
 * <p>If <code>inputLen</code> is zero, this method returns
 * a length of zero.
 *
 * <p>Note: this method should be copy-safe, which means the
 * <code>input</code> and <code>output</code> buffers can reference
 * the same byte array and no unprocessed input data is overwritten
 * when the result is copied into the output buffer.
 *
 * @param input the input buffer
 * @param inputOffset the offset in <code>input</code> where the input
 * starts
 * @param inputLen the input length
 * @param output the buffer for the result
 * @param outputOffset the offset in <code>output</code> where the result
 * is stored
 *
 * @return the number of bytes stored in <code>output</code>
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception ShortBufferException if the given output buffer is too small
 * to hold the result
 * @apiSince 1
 */

public final int update(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws javax.crypto.ShortBufferException { throw new RuntimeException("Stub!"); }

/**
 * Continues a multiple-part encryption or decryption operation
 * (depending on how this cipher was initialized), processing another data
 * part.
 *
 * <p>All <code>input.remaining()</code> bytes starting at
 * <code>input.position()</code> are processed. The result is stored
 * in the output buffer.
 * Upon return, the input buffer's position will be equal
 * to its limit; its limit will not have changed. The output buffer's
 * position will have advanced by n, where n is the value returned
 * by this method; the output buffer's limit will not have changed.
 *
 * <p>If <code>output.remaining()</code> bytes are insufficient to
 * hold the result, a <code>ShortBufferException</code> is thrown.
 * In this case, repeat this call with a larger output buffer. Use
 * {@link #getOutputSize(int) getOutputSize} to determine how big
 * the output buffer should be.
 *
 * <p>Note: this method should be copy-safe, which means the
 * <code>input</code> and <code>output</code> buffers can reference
 * the same block of memory and no unprocessed input data is overwritten
 * when the result is copied into the output buffer.
 *
 * @param input the input ByteBuffer
 * @param output the output ByteByffer
 *
 * @return the number of bytes stored in <code>output</code>
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalArgumentException if input and output are the
 *   same object
 * @exception ReadOnlyBufferException if the output buffer is read-only
 * @exception ShortBufferException if there is insufficient space in the
 * output buffer
 * @since 1.5
 * @apiSince 1
 */

public final int update(java.nio.ByteBuffer input, java.nio.ByteBuffer output) throws javax.crypto.ShortBufferException { throw new RuntimeException("Stub!"); }

/**
 * Finishes a multiple-part encryption or decryption operation, depending
 * on how this cipher was initialized.
 *
 * <p>Input data that may have been buffered during a previous
 * <code>update</code> operation is processed, with padding (if requested)
 * being applied.
 * If an AEAD mode such as GCM/CCM is being used, the authentication
 * tag is appended in the case of encryption, or verified in the
 * case of decryption.
 * The result is stored in a new buffer.
 *
 * <p>Upon finishing, this method resets this cipher object to the state
 * it was in when previously initialized via a call to <code>init</code>.
 * That is, the object is reset and available to encrypt or decrypt
 * (depending on the operation mode that was specified in the call to
 * <code>init</code>) more data.
 *
 * <p>Note: if any exception is thrown, this cipher object may need to
 * be reset before it can be used again.
 *
 * @return the new buffer with the result
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalBlockSizeException if this cipher is a block cipher,
 * no padding has been requested (only in encryption mode), and the total
 * input length of the data processed by this cipher is not a multiple of
 * block size; or if this encryption algorithm is unable to
 * process the input data provided.
 * @exception BadPaddingException if this cipher is in decryption mode,
 * and (un)padding has been requested, but the decrypted data is not
 * bounded by the appropriate padding bytes
 * @exception AEADBadTagException if this cipher is decrypting in an
 * AEAD mode (such as GCM/CCM), and the received authentication tag
 * does not match the calculated value
 * @apiSince 1
 */

public final byte[] doFinal() throws javax.crypto.BadPaddingException, javax.crypto.IllegalBlockSizeException { throw new RuntimeException("Stub!"); }

/**
 * Finishes a multiple-part encryption or decryption operation, depending
 * on how this cipher was initialized.
 *
 * <p>Input data that may have been buffered during a previous
 * <code>update</code> operation is processed, with padding (if requested)
 * being applied.
 * If an AEAD mode such as GCM/CCM is being used, the authentication
 * tag is appended in the case of encryption, or verified in the
 * case of decryption.
 * The result is stored in the <code>output</code> buffer, starting at
 * <code>outputOffset</code> inclusive.
 *
 * <p>If the <code>output</code> buffer is too small to hold the result,
 * a <code>ShortBufferException</code> is thrown. In this case, repeat this
 * call with a larger output buffer. Use
 * {@link #getOutputSize(int) getOutputSize} to determine how big
 * the output buffer should be.
 *
 * <p>Upon finishing, this method resets this cipher object to the state
 * it was in when previously initialized via a call to <code>init</code>.
 * That is, the object is reset and available to encrypt or decrypt
 * (depending on the operation mode that was specified in the call to
 * <code>init</code>) more data.
 *
 * <p>Note: if any exception is thrown, this cipher object may need to
 * be reset before it can be used again.
 *
 * @param output the buffer for the result
 * @param outputOffset the offset in <code>output</code> where the result
 * is stored
 *
 * @return the number of bytes stored in <code>output</code>
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalBlockSizeException if this cipher is a block cipher,
 * no padding has been requested (only in encryption mode), and the total
 * input length of the data processed by this cipher is not a multiple of
 * block size; or if this encryption algorithm is unable to
 * process the input data provided.
 * @exception ShortBufferException if the given output buffer is too small
 * to hold the result
 * @exception BadPaddingException if this cipher is in decryption mode,
 * and (un)padding has been requested, but the decrypted data is not
 * bounded by the appropriate padding bytes
 * @exception AEADBadTagException if this cipher is decrypting in an
 * AEAD mode (such as GCM/CCM), and the received authentication tag
 * does not match the calculated value
 * @apiSince 1
 */

public final int doFinal(byte[] output, int outputOffset) throws javax.crypto.BadPaddingException, javax.crypto.IllegalBlockSizeException, javax.crypto.ShortBufferException { throw new RuntimeException("Stub!"); }

/**
 * Encrypts or decrypts data in a single-part operation, or finishes a
 * multiple-part operation. The data is encrypted or decrypted,
 * depending on how this cipher was initialized.
 *
 * <p>The bytes in the <code>input</code> buffer, and any input bytes that
 * may have been buffered during a previous <code>update</code> operation,
 * are processed, with padding (if requested) being applied.
 * If an AEAD mode such as GCM/CCM is being used, the authentication
 * tag is appended in the case of encryption, or verified in the
 * case of decryption.
 * The result is stored in a new buffer.
 *
 * <p>Upon finishing, this method resets this cipher object to the state
 * it was in when previously initialized via a call to <code>init</code>.
 * That is, the object is reset and available to encrypt or decrypt
 * (depending on the operation mode that was specified in the call to
 * <code>init</code>) more data.
 *
 * <p>Note: if any exception is thrown, this cipher object may need to
 * be reset before it can be used again.
 *
 * @param input the input buffer
 *
 * @return the new buffer with the result
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalBlockSizeException if this cipher is a block cipher,
 * no padding has been requested (only in encryption mode), and the total
 * input length of the data processed by this cipher is not a multiple of
 * block size; or if this encryption algorithm is unable to
 * process the input data provided.
 * @exception BadPaddingException if this cipher is in decryption mode,
 * and (un)padding has been requested, but the decrypted data is not
 * bounded by the appropriate padding bytes
 * @exception AEADBadTagException if this cipher is decrypting in an
 * AEAD mode (such as GCM/CCM), and the received authentication tag
 * does not match the calculated value
 * @apiSince 1
 */

public final byte[] doFinal(byte[] input) throws javax.crypto.BadPaddingException, javax.crypto.IllegalBlockSizeException { throw new RuntimeException("Stub!"); }

/**
 * Encrypts or decrypts data in a single-part operation, or finishes a
 * multiple-part operation. The data is encrypted or decrypted,
 * depending on how this cipher was initialized.
 *
 * <p>The first <code>inputLen</code> bytes in the <code>input</code>
 * buffer, starting at <code>inputOffset</code> inclusive, and any input
 * bytes that may have been buffered during a previous <code>update</code>
 * operation, are processed, with padding (if requested) being applied.
 * If an AEAD mode such as GCM/CCM is being used, the authentication
 * tag is appended in the case of encryption, or verified in the
 * case of decryption.
 * The result is stored in a new buffer.
 *
 * <p>Upon finishing, this method resets this cipher object to the state
 * it was in when previously initialized via a call to <code>init</code>.
 * That is, the object is reset and available to encrypt or decrypt
 * (depending on the operation mode that was specified in the call to
 * <code>init</code>) more data.
 *
 * <p>Note: if any exception is thrown, this cipher object may need to
 * be reset before it can be used again.
 *
 * @param input the input buffer
 * @param inputOffset the offset in <code>input</code> where the input
 * starts
 * @param inputLen the input length
 *
 * @return the new buffer with the result
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalBlockSizeException if this cipher is a block cipher,
 * no padding has been requested (only in encryption mode), and the total
 * input length of the data processed by this cipher is not a multiple of
 * block size; or if this encryption algorithm is unable to
 * process the input data provided.
 * @exception BadPaddingException if this cipher is in decryption mode,
 * and (un)padding has been requested, but the decrypted data is not
 * bounded by the appropriate padding bytes
 * @exception AEADBadTagException if this cipher is decrypting in an
 * AEAD mode (such as GCM/CCM), and the received authentication tag
 * does not match the calculated value
 * @apiSince 1
 */

public final byte[] doFinal(byte[] input, int inputOffset, int inputLen) throws javax.crypto.BadPaddingException, javax.crypto.IllegalBlockSizeException { throw new RuntimeException("Stub!"); }

/**
 * Encrypts or decrypts data in a single-part operation, or finishes a
 * multiple-part operation. The data is encrypted or decrypted,
 * depending on how this cipher was initialized.
 *
 * <p>The first <code>inputLen</code> bytes in the <code>input</code>
 * buffer, starting at <code>inputOffset</code> inclusive, and any input
 * bytes that may have been buffered during a previous <code>update</code>
 * operation, are processed, with padding (if requested) being applied.
 * If an AEAD mode such as GCM/CCM is being used, the authentication
 * tag is appended in the case of encryption, or verified in the
 * case of decryption.
 * The result is stored in the <code>output</code> buffer.
 *
 * <p>If the <code>output</code> buffer is too small to hold the result,
 * a <code>ShortBufferException</code> is thrown. In this case, repeat this
 * call with a larger output buffer. Use
 * {@link #getOutputSize(int) getOutputSize} to determine how big
 * the output buffer should be.
 *
 * <p>Upon finishing, this method resets this cipher object to the state
 * it was in when previously initialized via a call to <code>init</code>.
 * That is, the object is reset and available to encrypt or decrypt
 * (depending on the operation mode that was specified in the call to
 * <code>init</code>) more data.
 *
 * <p>Note: if any exception is thrown, this cipher object may need to
 * be reset before it can be used again.
 *
 * <p>Note: this method should be copy-safe, which means the
 * <code>input</code> and <code>output</code> buffers can reference
 * the same byte array and no unprocessed input data is overwritten
 * when the result is copied into the output buffer.
 *
 * @param input the input buffer
 * @param inputOffset the offset in <code>input</code> where the input
 * starts
 * @param inputLen the input length
 * @param output the buffer for the result
 *
 * @return the number of bytes stored in <code>output</code>
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalBlockSizeException if this cipher is a block cipher,
 * no padding has been requested (only in encryption mode), and the total
 * input length of the data processed by this cipher is not a multiple of
 * block size; or if this encryption algorithm is unable to
 * process the input data provided.
 * @exception ShortBufferException if the given output buffer is too small
 * to hold the result
 * @exception BadPaddingException if this cipher is in decryption mode,
 * and (un)padding has been requested, but the decrypted data is not
 * bounded by the appropriate padding bytes
 * @exception AEADBadTagException if this cipher is decrypting in an
 * AEAD mode (such as GCM/CCM), and the received authentication tag
 * does not match the calculated value
 * @apiSince 1
 */

public final int doFinal(byte[] input, int inputOffset, int inputLen, byte[] output) throws javax.crypto.BadPaddingException, javax.crypto.IllegalBlockSizeException, javax.crypto.ShortBufferException { throw new RuntimeException("Stub!"); }

/**
 * Encrypts or decrypts data in a single-part operation, or finishes a
 * multiple-part operation. The data is encrypted or decrypted,
 * depending on how this cipher was initialized.
 *
 * <p>The first <code>inputLen</code> bytes in the <code>input</code>
 * buffer, starting at <code>inputOffset</code> inclusive, and any input
 * bytes that may have been buffered during a previous
 * <code>update</code> operation, are processed, with padding
 * (if requested) being applied.
 * If an AEAD mode such as GCM/CCM is being used, the authentication
 * tag is appended in the case of encryption, or verified in the
 * case of decryption.
 * The result is stored in the <code>output</code> buffer, starting at
 * <code>outputOffset</code> inclusive.
 *
 * <p>If the <code>output</code> buffer is too small to hold the result,
 * a <code>ShortBufferException</code> is thrown. In this case, repeat this
 * call with a larger output buffer. Use
 * {@link #getOutputSize(int) getOutputSize} to determine how big
 * the output buffer should be.
 *
 * <p>Upon finishing, this method resets this cipher object to the state
 * it was in when previously initialized via a call to <code>init</code>.
 * That is, the object is reset and available to encrypt or decrypt
 * (depending on the operation mode that was specified in the call to
 * <code>init</code>) more data.
 *
 * <p>Note: if any exception is thrown, this cipher object may need to
 * be reset before it can be used again.
 *
 * <p>Note: this method should be copy-safe, which means the
 * <code>input</code> and <code>output</code> buffers can reference
 * the same byte array and no unprocessed input data is overwritten
 * when the result is copied into the output buffer.
 *
 * @param input the input buffer
 * @param inputOffset the offset in <code>input</code> where the input
 * starts
 * @param inputLen the input length
 * @param output the buffer for the result
 * @param outputOffset the offset in <code>output</code> where the result
 * is stored
 *
 * @return the number of bytes stored in <code>output</code>
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalBlockSizeException if this cipher is a block cipher,
 * no padding has been requested (only in encryption mode), and the total
 * input length of the data processed by this cipher is not a multiple of
 * block size; or if this encryption algorithm is unable to
 * process the input data provided.
 * @exception ShortBufferException if the given output buffer is too small
 * to hold the result
 * @exception BadPaddingException if this cipher is in decryption mode,
 * and (un)padding has been requested, but the decrypted data is not
 * bounded by the appropriate padding bytes
 * @exception AEADBadTagException if this cipher is decrypting in an
 * AEAD mode (such as GCM/CCM), and the received authentication tag
 * does not match the calculated value
 * @apiSince 1
 */

public final int doFinal(byte[] input, int inputOffset, int inputLen, byte[] output, int outputOffset) throws javax.crypto.BadPaddingException, javax.crypto.IllegalBlockSizeException, javax.crypto.ShortBufferException { throw new RuntimeException("Stub!"); }

/**
 * Encrypts or decrypts data in a single-part operation, or finishes a
 * multiple-part operation. The data is encrypted or decrypted,
 * depending on how this cipher was initialized.
 *
 * <p>All <code>input.remaining()</code> bytes starting at
 * <code>input.position()</code> are processed.
 * If an AEAD mode such as GCM/CCM is being used, the authentication
 * tag is appended in the case of encryption, or verified in the
 * case of decryption.
 * The result is stored in the output buffer.
 * Upon return, the input buffer's position will be equal
 * to its limit; its limit will not have changed. The output buffer's
 * position will have advanced by n, where n is the value returned
 * by this method; the output buffer's limit will not have changed.
 *
 * <p>If <code>output.remaining()</code> bytes are insufficient to
 * hold the result, a <code>ShortBufferException</code> is thrown.
 * In this case, repeat this call with a larger output buffer. Use
 * {@link #getOutputSize(int) getOutputSize} to determine how big
 * the output buffer should be.
 *
 * <p>Upon finishing, this method resets this cipher object to the state
 * it was in when previously initialized via a call to <code>init</code>.
 * That is, the object is reset and available to encrypt or decrypt
 * (depending on the operation mode that was specified in the call to
 * <code>init</code>) more data.
 *
 * <p>Note: if any exception is thrown, this cipher object may need to
 * be reset before it can be used again.
 *
 * <p>Note: this method should be copy-safe, which means the
 * <code>input</code> and <code>output</code> buffers can reference
 * the same byte array and no unprocessed input data is overwritten
 * when the result is copied into the output buffer.
 *
 * @param input the input ByteBuffer
 * @param output the output ByteBuffer
 *
 * @return the number of bytes stored in <code>output</code>
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized)
 * @exception IllegalArgumentException if input and output are the
 *   same object
 * @exception ReadOnlyBufferException if the output buffer is read-only
 * @exception IllegalBlockSizeException if this cipher is a block cipher,
 * no padding has been requested (only in encryption mode), and the total
 * input length of the data processed by this cipher is not a multiple of
 * block size; or if this encryption algorithm is unable to
 * process the input data provided.
 * @exception ShortBufferException if there is insufficient space in the
 * output buffer
 * @exception BadPaddingException if this cipher is in decryption mode,
 * and (un)padding has been requested, but the decrypted data is not
 * bounded by the appropriate padding bytes
 * @exception AEADBadTagException if this cipher is decrypting in an
 * AEAD mode (such as GCM/CCM), and the received authentication tag
 * does not match the calculated value
 *
 * @since 1.5
 * @apiSince 1
 */

public final int doFinal(java.nio.ByteBuffer input, java.nio.ByteBuffer output) throws javax.crypto.BadPaddingException, javax.crypto.IllegalBlockSizeException, javax.crypto.ShortBufferException { throw new RuntimeException("Stub!"); }

/**
 * Wrap a key.
 *
 * @param key the key to be wrapped.
 *
 * @return the wrapped key.
 *
 * @exception IllegalStateException if this cipher is in a wrong
 * state (e.g., has not been initialized).
 *
 * @exception IllegalBlockSizeException if this cipher is a block
 * cipher, no padding has been requested, and the length of the
 * encoding of the key to be wrapped is not a
 * multiple of the block size.
 *
 * @exception InvalidKeyException if it is impossible or unsafe to
 * wrap the key with this cipher (e.g., a hardware protected key is
 * being passed to a software-only cipher).
 *
 * @throws UnsupportedOperationException if the corresponding method in the
 * {@code CipherSpi} is not supported.
 * @apiSince 1
 */

public final byte[] wrap(java.security.Key key) throws javax.crypto.IllegalBlockSizeException, java.security.InvalidKeyException { throw new RuntimeException("Stub!"); }

/**
 * Unwrap a previously wrapped key.
 *
 * @param wrappedKey the key to be unwrapped.
 *
 * @param wrappedKeyAlgorithm the algorithm associated with the wrapped
 * key.
 *
 * @param wrappedKeyType the type of the wrapped key. This must be one of
 * <code>SECRET_KEY</code>, <code>PRIVATE_KEY</code>, or
 * <code>PUBLIC_KEY</code>.
 *
 * @return the unwrapped key.
 *
 * @exception IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized).
 *
 * @exception NoSuchAlgorithmException if no installed providers
 * can create keys of type <code>wrappedKeyType</code> for the
 * <code>wrappedKeyAlgorithm</code>.
 *
 * @exception InvalidKeyException if <code>wrappedKey</code> does not
 * represent a wrapped key of type <code>wrappedKeyType</code> for
 * the <code>wrappedKeyAlgorithm</code>.
 *
 * @throws UnsupportedOperationException if the corresponding method in the
 * {@code CipherSpi} is not supported.
 * @apiSince 1
 */

public final java.security.Key unwrap(byte[] wrappedKey, java.lang.String wrappedKeyAlgorithm, int wrappedKeyType) throws java.security.InvalidKeyException, java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Returns the maximum key length for the specified transformation
 * according to the installed JCE jurisdiction policy files. If
 * JCE unlimited strength jurisdiction policy files are installed,
 * Integer.MAX_VALUE will be returned.
 * For more information on default key size in JCE jurisdiction
 * policy files, please see Appendix E in the
 * <a href=
 *   "https://docs.oracle.com/javase/8/docs/technotes/guides/security/crypto/CryptoSpec.html#AppC">
 * Java Cryptography Architecture Reference Guide</a>.
 *
 * @param transformation the cipher transformation.
 * @return the maximum key length in bits or Integer.MAX_VALUE.
 * @exception NullPointerException if <code>transformation</code> is null.
 * @exception NoSuchAlgorithmException if <code>transformation</code>
 * is not a valid transformation, i.e. in the form of "algorithm" or
 * "algorithm/mode/padding".
 * @since 1.5
 * @apiSince 1
 */

public static final int getMaxAllowedKeyLength(java.lang.String transformation) throws java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Returns an AlgorithmParameterSpec object which contains
 * the maximum cipher parameter value according to the
 * jurisdiction policy file. If JCE unlimited strength jurisdiction
 * policy files are installed or there is no maximum limit on the
 * parameters for the specified transformation in the policy file,
 * null will be returned.
 *
 * @param transformation the cipher transformation.
 * @return an AlgorithmParameterSpec which holds the maximum
 * value or null.
 * @exception NullPointerException if <code>transformation</code>
 * is null.
 * @exception NoSuchAlgorithmException if <code>transformation</code>
 * is not a valid transformation, i.e. in the form of "algorithm" or
 * "algorithm/mode/padding".
 * @since 1.5
 * @apiSince 1
 */

public static final java.security.spec.AlgorithmParameterSpec getMaxAllowedParameterSpec(java.lang.String transformation) throws java.security.NoSuchAlgorithmException { throw new RuntimeException("Stub!"); }

/**
 * Continues a multi-part update of the Additional Authentication
 * Data (AAD).
 * <p>
 * Calls to this method provide AAD to the cipher when operating in
 * modes such as AEAD (GCM/CCM).  If this cipher is operating in
 * either GCM or CCM mode, all AAD must be supplied before beginning
 * operations on the ciphertext (via the {@code update} and {@code
 * doFinal} methods).
 *
 * @param src the buffer containing the Additional Authentication Data
 *
 * @throws IllegalArgumentException if the {@code src}
 * byte array is null
 * @throws IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized), does not accept AAD, or if
 * operating in either GCM or CCM mode and one of the {@code update}
 * methods has already been called for the active
 * encryption/decryption operation
 * @throws UnsupportedOperationException if the corresponding method
 * in the {@code CipherSpi} has not been overridden by an
 * implementation
 *
 * @since 1.7
 * @apiSince 19
 */

public final void updateAAD(byte[] src) { throw new RuntimeException("Stub!"); }

/**
 * Continues a multi-part update of the Additional Authentication
 * Data (AAD), using a subset of the provided buffer.
 * <p>
 * Calls to this method provide AAD to the cipher when operating in
 * modes such as AEAD (GCM/CCM).  If this cipher is operating in
 * either GCM or CCM mode, all AAD must be supplied before beginning
 * operations on the ciphertext (via the {@code update} and {@code
 * doFinal} methods).
 *
 * @param src the buffer containing the AAD
 * @param offset the offset in {@code src} where the AAD input starts
 * @param len the number of AAD bytes
 *
 * @throws IllegalArgumentException if the {@code src}
 * byte array is null, or the {@code offset} or {@code length}
 * is less than 0, or the sum of the {@code offset} and
 * {@code len} is greater than the length of the
 * {@code src} byte array
 * @throws IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized), does not accept AAD, or if
 * operating in either GCM or CCM mode and one of the {@code update}
 * methods has already been called for the active
 * encryption/decryption operation
 * @throws UnsupportedOperationException if the corresponding method
 * in the {@code CipherSpi} has not been overridden by an
 * implementation
 *
 * @since 1.7
 * @apiSince 19
 */

public final void updateAAD(byte[] src, int offset, int len) { throw new RuntimeException("Stub!"); }

/**
 * Continues a multi-part update of the Additional Authentication
 * Data (AAD).
 * <p>
 * Calls to this method provide AAD to the cipher when operating in
 * modes such as AEAD (GCM/CCM).  If this cipher is operating in
 * either GCM or CCM mode, all AAD must be supplied before beginning
 * operations on the ciphertext (via the {@code update} and {@code
 * doFinal} methods).
 * <p>
 * All {@code src.remaining()} bytes starting at
 * {@code src.position()} are processed.
 * Upon return, the input buffer's position will be equal
 * to its limit; its limit will not have changed.
 *
 * @param src the buffer containing the AAD
 *
 * @throws IllegalArgumentException if the {@code src ByteBuffer}
 * is null
 * @throws IllegalStateException if this cipher is in a wrong state
 * (e.g., has not been initialized), does not accept AAD, or if
 * operating in either GCM or CCM mode and one of the {@code update}
 * methods has already been called for the active
 * encryption/decryption operation
 * @throws UnsupportedOperationException if the corresponding method
 * in the {@code CipherSpi} has not been overridden by an
 * implementation
 *
 * @since 1.7
 * @apiSince 19
 */

public final void updateAAD(java.nio.ByteBuffer src) { throw new RuntimeException("Stub!"); }

/**
 * Constant used to initialize cipher to decryption mode.
 * @apiSince 1
 */

public static final int DECRYPT_MODE = 2; // 0x2

/**
 * Constant used to initialize cipher to encryption mode.
 * @apiSince 1
 */

public static final int ENCRYPT_MODE = 1; // 0x1

/**
 * Constant used to indicate the to-be-unwrapped key is a "private key".
 * @apiSince 1
 */

public static final int PRIVATE_KEY = 2; // 0x2

/**
 * Constant used to indicate the to-be-unwrapped key is a "public key".
 * @apiSince 1
 */

public static final int PUBLIC_KEY = 1; // 0x1

/**
 * Constant used to indicate the to-be-unwrapped key is a "secret key".
 * @apiSince 1
 */

public static final int SECRET_KEY = 3; // 0x3

/**
 * Constant used to initialize cipher to key-unwrapping mode.
 * @apiSince 1
 */

public static final int UNWRAP_MODE = 4; // 0x4

/**
 * Constant used to initialize cipher to key-wrapping mode.
 * @apiSince 1
 */

public static final int WRAP_MODE = 3; // 0x3
}

