/*
 * Copyright 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.hardware.camera2;

import android.graphics.ImageFormat;
import android.media.ExifInterface;
import android.graphics.Bitmap;
import android.media.Image;
import android.location.Location;
import android.util.Size;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import android.graphics.Color;

/**
 * The {@link DngCreator} class provides functions to write raw pixel data as a DNG file.
 *
 * <p>
 * This class is designed to be used with the {@link android.graphics.ImageFormat#RAW_SENSOR}
 * buffers available from {@link android.hardware.camera2.CameraDevice}, or with Bayer-type raw
 * pixel data that is otherwise generated by an application.  The DNG metadata tags will be
 * generated from a {@link android.hardware.camera2.CaptureResult} object or set directly.
 * </p>
 *
 * <p>
 * The DNG file format is a cross-platform file format that is used to store pixel data from
 * camera sensors with minimal pre-processing applied.  DNG files allow for pixel data to be
 * defined in a user-defined colorspace, and have associated metadata that allow for this
 * pixel data to be converted to the standard CIE XYZ colorspace during post-processing.
 * </p>
 *
 * <p>
 * For more information on the DNG file format and associated metadata, please refer to the
 * <a href=
 * "https://wwwimages2.adobe.com/content/dam/Adobe/en/products/photoshop/pdfs/dng_spec_1.4.0.0.pdf">
 * Adobe DNG 1.4.0.0 specification</a>.
 * </p>
 * @apiSince 21
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class DngCreator implements java.lang.AutoCloseable {

/**
 * Create a new DNG object.
 *
 * <p>
 * It is not necessary to call any set methods to write a well-formatted DNG file.
 * </p>
 * <p>
 * DNG metadata tags will be generated from the corresponding parameters in the
 * {@link android.hardware.camera2.CaptureResult} object.
 * </p>
 * <p>
 * For best quality DNG files, it is strongly recommended that lens shading map output is
 * enabled if supported. See {@link CaptureRequest#STATISTICS_LENS_SHADING_MAP_MODE}.
 * </p>
 * @param characteristics an object containing the static
 *          {@link android.hardware.camera2.CameraCharacteristics}.
 * This value must never be {@code null}.
 * @param metadata a metadata object to generate tags from.
 
 * This value must never be {@code null}.
 * @apiSince 21
 */

public DngCreator(@android.annotation.NonNull android.hardware.camera2.CameraCharacteristics characteristics, @android.annotation.NonNull android.hardware.camera2.CaptureResult metadata) { throw new RuntimeException("Stub!"); }

/**
 * Set the orientation value to write.
 *
 * <p>
 * This will be written as the TIFF "Orientation" tag {@code (0x0112)}.
 * Calling this will override any prior settings for this tag.
 * </p>
 *
 * @param orientation the orientation value to set, one of:
 *                    <ul>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_NORMAL}</li>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_FLIP_HORIZONTAL}</li>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_ROTATE_180}</li>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_FLIP_VERTICAL}</li>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_TRANSPOSE}</li>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_ROTATE_90}</li>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_TRANSVERSE}</li>
 *                      <li>{@link android.media.ExifInterface#ORIENTATION_ROTATE_270}</li>
 *                    </ul>
 * @return this {@link #DngCreator} object.
 
 * This value will never be {@code null}.
 * @apiSince 21
 */

@android.annotation.NonNull
public android.hardware.camera2.DngCreator setOrientation(int orientation) { throw new RuntimeException("Stub!"); }

/**
 * Set the thumbnail image.
 *
 * <p>
 * Pixel data will be converted to a Baseline TIFF RGB image, with 8 bits per color channel.
 * The alpha channel will be discarded.  Thumbnail images with a dimension larger than
 * {@link #MAX_THUMBNAIL_DIMENSION} will be rejected.
 * </p>
 *
 * @param pixels a {@link android.graphics.Bitmap} of pixel data.
 * This value must never be {@code null}.
 * @return this {@link #DngCreator} object.
 * This value will never be {@code null}.
 * @throws java.lang.IllegalArgumentException if the given thumbnail image has a dimension
 *      larger than {@link #MAX_THUMBNAIL_DIMENSION}.
 * @apiSince 21
 */

@android.annotation.NonNull
public android.hardware.camera2.DngCreator setThumbnail(@android.annotation.NonNull android.graphics.Bitmap pixels) { throw new RuntimeException("Stub!"); }

/**
 * Set the thumbnail image.
 *
 * <p>
 * Pixel data is interpreted as a {@link android.graphics.ImageFormat#YUV_420_888} image.
 * Thumbnail images with a dimension larger than {@link #MAX_THUMBNAIL_DIMENSION} will be
 * rejected.
 * </p>
 *
 * @param pixels an {@link android.media.Image} object with the format
 *               {@link android.graphics.ImageFormat#YUV_420_888}.
 * This value must never be {@code null}.
 * @return this {@link #DngCreator} object.
 * This value will never be {@code null}.
 * @throws java.lang.IllegalArgumentException if the given thumbnail image has a dimension
 *      larger than {@link #MAX_THUMBNAIL_DIMENSION}.
 * @apiSince 21
 */

@android.annotation.NonNull
public android.hardware.camera2.DngCreator setThumbnail(@android.annotation.NonNull android.media.Image pixels) { throw new RuntimeException("Stub!"); }

/**
 * Set image location metadata.
 *
 * <p>
 * The given location object must contain at least a valid time, latitude, and longitude
 * (equivalent to the values returned by {@link android.location.Location#getTime()},
 * {@link android.location.Location#getLatitude()}, and
 * {@link android.location.Location#getLongitude()} methods).
 * </p>
 *
 * @param location an {@link android.location.Location} object to set.
 * This value must never be {@code null}.
 * @return this {@link #DngCreator} object.
 *
 * This value will never be {@code null}.
 * @throws java.lang.IllegalArgumentException if the given location object doesn't
 *          contain enough information to set location metadata.
 * @apiSince 21
 */

@android.annotation.NonNull
public android.hardware.camera2.DngCreator setLocation(@android.annotation.NonNull android.location.Location location) { throw new RuntimeException("Stub!"); }

/**
 * Set the user description string to write.
 *
 * <p>
 * This is equivalent to setting the TIFF "ImageDescription" tag {@code (0x010E)}.
 * </p>
 *
 * @param description the user description string.
 * This value must never be {@code null}.
 * @return this {@link #DngCreator} object.
 
 * This value will never be {@code null}.
 * @apiSince 21
 */

@android.annotation.NonNull
public android.hardware.camera2.DngCreator setDescription(@android.annotation.NonNull java.lang.String description) { throw new RuntimeException("Stub!"); }

/**
 * Write the {@link android.graphics.ImageFormat#RAW_SENSOR} pixel data to a DNG file with
 * the currently configured metadata.
 *
 * <p>
 * Raw pixel data must have 16 bits per pixel, and the input must contain at least
 * {@code offset + 2 * width * height)} bytes.  The width and height of
 * the input are taken from the width and height set in the {@link DngCreator} metadata tags,
 * and will typically be equal to the width and height of
 * {@link CameraCharacteristics#SENSOR_INFO_PRE_CORRECTION_ACTIVE_ARRAY_SIZE}.  Prior to
 * API level 23, this was always the same as
 * {@link CameraCharacteristics#SENSOR_INFO_ACTIVE_ARRAY_SIZE}.
 * The pixel layout in the input is determined from the reported color filter arrangement (CFA)
 * set in {@link CameraCharacteristics#SENSOR_INFO_COLOR_FILTER_ARRANGEMENT}.  If insufficient
 * metadata is available to write a well-formatted DNG file, an
 * {@link java.lang.IllegalStateException} will be thrown.
 * </p>
 *
 * @param dngOutput an {@link java.io.OutputStream} to write the DNG file to.
 * This value must never be {@code null}.
 * @param size the {@link Size} of the image to write, in pixels.
 * This value must never be {@code null}.
 * @param pixels an {@link java.io.InputStream} of pixel data to write.
 * This value must never be {@code null}.
 * @param offset the offset of the raw image in bytes.  This indicates how many bytes will
 *               be skipped in the input before any pixel data is read.
 *
 * Value is 0 or greater
 * @throws IOException if an error was encountered in the input or output stream.
 * @throws java.lang.IllegalStateException if not enough metadata information has been
 *          set to write a well-formatted DNG file.
 * @throws java.lang.IllegalArgumentException if the size passed in does not match the
 * @apiSince 21
 */

public void writeInputStream(@android.annotation.NonNull java.io.OutputStream dngOutput, @android.annotation.NonNull android.util.Size size, @android.annotation.NonNull java.io.InputStream pixels, long offset) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Write the {@link android.graphics.ImageFormat#RAW_SENSOR} pixel data to a DNG file with
 * the currently configured metadata.
 *
 * <p>
 * Raw pixel data must have 16 bits per pixel, and the input must contain at least
 * {@code offset + 2 * width * height)} bytes.  The width and height of
 * the input are taken from the width and height set in the {@link DngCreator} metadata tags,
 * and will typically be equal to the width and height of
 * {@link CameraCharacteristics#SENSOR_INFO_PRE_CORRECTION_ACTIVE_ARRAY_SIZE}.  Prior to
 * API level 23, this was always the same as
 * {@link CameraCharacteristics#SENSOR_INFO_ACTIVE_ARRAY_SIZE}.
 * The pixel layout in the input is determined from the reported color filter arrangement (CFA)
 * set in {@link CameraCharacteristics#SENSOR_INFO_COLOR_FILTER_ARRANGEMENT}.  If insufficient
 * metadata is available to write a well-formatted DNG file, an
 * {@link java.lang.IllegalStateException} will be thrown.
 * </p>
 *
 * <p>
 * Any mark or limit set on this {@link ByteBuffer} is ignored, and will be cleared by this
 * method.
 * </p>
 *
 * @param dngOutput an {@link java.io.OutputStream} to write the DNG file to.
 * This value must never be {@code null}.
 * @param size the {@link Size} of the image to write, in pixels.
 * This value must never be {@code null}.
 * @param pixels an {@link java.nio.ByteBuffer} of pixel data to write.
 * This value must never be {@code null}.
 * @param offset the offset of the raw image in bytes.  This indicates how many bytes will
 *               be skipped in the input before any pixel data is read.
 *
 * Value is 0 or greater
 * @throws IOException if an error was encountered in the input or output stream.
 * @throws java.lang.IllegalStateException if not enough metadata information has been
 *          set to write a well-formatted DNG file.
 * @apiSince 21
 */

public void writeByteBuffer(@android.annotation.NonNull java.io.OutputStream dngOutput, @android.annotation.NonNull android.util.Size size, @android.annotation.NonNull java.nio.ByteBuffer pixels, long offset) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/**
 * Write the pixel data to a DNG file with the currently configured metadata.
 *
 * <p>
 * For this method to succeed, the {@link android.media.Image} input must contain
 * {@link android.graphics.ImageFormat#RAW_SENSOR} pixel data, otherwise an
 * {@link java.lang.IllegalArgumentException} will be thrown.
 * </p>
 *
 * @param dngOutput an {@link java.io.OutputStream} to write the DNG file to.
 * This value must never be {@code null}.
 * @param pixels an {@link android.media.Image} to write.
 *
 * This value must never be {@code null}.
 * @throws java.io.IOException if an error was encountered in the output stream.
 * @throws java.lang.IllegalArgumentException if an image with an unsupported format was used.
 * @throws java.lang.IllegalStateException if not enough metadata information has been
 *          set to write a well-formatted DNG file.
 * @apiSince 21
 */

public void writeImage(@android.annotation.NonNull java.io.OutputStream dngOutput, @android.annotation.NonNull android.media.Image pixels) throws java.io.IOException { throw new RuntimeException("Stub!"); }

/** @apiSince 21 */

public void close() { throw new RuntimeException("Stub!"); }

/** @apiSince 21 */

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }

/**
 * Max width or height dimension for thumbnails.
 * @apiSince 21
 */

public static final int MAX_THUMBNAIL_DIMENSION = 256; // 0x100
}

