
#include "Assembly.h"

#include <cstring>
#include <iostream>
#include <vector>

#include "os/File.h"
#include "utils/MemoryMappedFile.h"
#include "vm/Assembly.h"
#include "vm/Image.h"
#include "vm/Class.h"
#include "vm/String.h"
#include "vm/MetadataLock.h"

#include "Image.h"
#include "MetadataModule.h"
#include "MetadataUtil.h"
#include "DifferentialHybridImage.h"
#include "ConsistentAOTHomologousImage.h"
#include "SuperSetAOTHomologousImage.h"

namespace hybridclr
{
namespace metadata
{

    std::vector<Il2CppAssembly*> s_placeHolderAssembies;

#if ENABLE_PLACEHOLDER_DLL == 1

    static const char* CreateAssemblyNameWithoutExt(const char* assemblyName)
    {
        const char* extStr = std::strstr(assemblyName, ".dll");
        if (extStr)
        {
            size_t nameLen = extStr - assemblyName;
            char* name = (char*)HYBRIDCLR_MALLOC(nameLen + 1);
            std::strncpy(name, assemblyName, nameLen);
            name[nameLen] = '\0';
            return name;
        }
        else
        {
            return CopyString(assemblyName);
        }
    }

    static Il2CppAssembly* FindPlaceHolderAssemblyLocked(const char* assemblyNameNoExt, il2cpp::os::FastAutoLock& lock)
    {
        for (Il2CppAssembly* ass : s_placeHolderAssembies)
        {
            if (std::strcmp(ass->image->nameNoExt, assemblyNameNoExt) == 0)
            {
                return ass;
            }
        }
        return nullptr;
    }
#else
    static Il2CppAssembly* FindPlaceHolderAssemblyLocked(const char* assemblyNameNoExt, il2cpp::os::FastAutoLock& lock)
    {
        return nullptr;
    }
#endif


    static Il2CppAssembly* CreatePlaceHolderAssembly(const char* assemblyName)
    {
        auto ass = new (HYBRIDCLR_MALLOC_ZERO(sizeof(Il2CppAssembly))) Il2CppAssembly;
        auto image2 = new (HYBRIDCLR_MALLOC_ZERO(sizeof(Il2CppImage))) Il2CppImage;
        ass->image = image2;
        ass->image->name = CopyString(assemblyName);
        ass->image->nameNoExt = ass->aname.name = CreateAssemblyNameWithoutExt(assemblyName);
        image2->assembly = ass;
        return ass;
    }

    void Assembly::InitializePlaceHolderAssemblies()
    {
        for (const char** ptrPlaceHolderName = g_placeHolderAssemblies; *ptrPlaceHolderName; ++ptrPlaceHolderName)
        {
            const char* nameWithExtension = ConcatNewString(*ptrPlaceHolderName, ".dll");
            Il2CppAssembly* placeHolderAss = CreatePlaceHolderAssembly(nameWithExtension);
            s_placeHolderAssembies.push_back(placeHolderAss);
            HYBRIDCLR_FREE((void*)nameWithExtension);
            il2cpp::vm::MetadataCache::RegisterInterpreterAssembly(placeHolderAss);
        }
    }

    static void RunModuleInitializer(Il2CppImage* image)
    {
        Il2CppClass* moduleKlass = il2cpp::vm::Image::ClassFromName(image, "", "<Module>");
        if (!moduleKlass)
        {
            return;
        }
        il2cpp::vm::Runtime::ClassInit(moduleKlass);
    }

    Il2CppAssembly* Assembly::LoadFromBytes(const void* assemblyData, uint64_t length, bool copyData)
    {
        Il2CppAssembly* ass = Create((const byte*)assemblyData, length, copyData);
        if (ass)
        {
            RunModuleInitializer(ass->image);
        }
        return ass;
    }

    Il2CppAssembly* Assembly::Create(const byte* assemblyData, uint64_t length, bool copyData)
    {
        il2cpp::os::FastAutoLock lock(&il2cpp::vm::g_MetadataLock);
        if (!assemblyData)
        {
            il2cpp::vm::Exception::Raise(il2cpp::vm::Exception::GetArgumentNullException("rawAssembly is null"));
        }

        uint32_t imageId = InterpreterImage::AllocImageIndex((uint32_t)length);
        if (imageId == kInvalidImageIndex)
        {
            il2cpp::vm::Exception::Raise(il2cpp::vm::Exception::GetExecutionEngineException("InterpreterImage::AllocImageIndex failed"));
        }
        InterpreterImage* image = new InterpreterImage(imageId);
        
        if (copyData)
        {
            assemblyData = (const byte*)CopyBytes(assemblyData, length);
        }
        LoadImageErrorCode err = image->Load(assemblyData, (size_t)length);


        if (err != LoadImageErrorCode::OK)
        {
            if (copyData)
            {
                HYBRIDCLR_FREE((void*)assemblyData);
            }
            TEMP_FORMAT(errMsg, "LoadImageErrorCode:%d", (int)err);
            il2cpp::vm::Exception::Raise(il2cpp::vm::Exception::GetBadImageFormatException(errMsg));
            // when load a bad image, mean a fatal error. we don't clean image on purpose.
        }

        TbAssembly data = image->GetRawImage().ReadAssembly(1);
        const char* nameNoExt = image->GetStringFromRawIndex(data.name);

        Il2CppAssembly* ass;
        Il2CppImage* image2;
        if ((ass = FindPlaceHolderAssemblyLocked(nameNoExt, lock)) != nullptr)
        {
            if (ass->token)
            {
                RaiseExecutionEngineException("reloading placeholder assembly is not supported!");
            }
            image2 = ass->image;
            HYBRIDCLR_FREE((void*)ass->image->name);
            HYBRIDCLR_FREE((void*)ass->image->nameNoExt);
        }
        else
        {
            ass = new (HYBRIDCLR_MALLOC_ZERO(sizeof(Il2CppAssembly))) Il2CppAssembly;
            image2 = new (HYBRIDCLR_MALLOC_ZERO(sizeof(Il2CppImage))) Il2CppImage;
        }

        image->InitBasic(image2);
        image->BuildIl2CppAssembly(ass);
        ass->image = image2;

        image->BuildIl2CppImage(image2);
        image2->name = ConcatNewString(ass->aname.name, ".dll");
        image2->nameNoExt = ass->aname.name;
        image2->assembly = ass;

        image->InitRuntimeMetadatas();

        il2cpp::vm::Assembly::InvalidateAssemblyList();
        return ass;
    }

    LoadImageErrorCode Assembly::LoadMetadataForAOTAssembly(const void* dllBytes, uint32_t dllSize, HomologousImageMode mode)
    {
        il2cpp::os::FastAutoLock lock(&il2cpp::vm::g_MetadataLock);

        AOTHomologousImage* image = nullptr;
        switch (mode)
        {
        case HomologousImageMode::CONSISTENT: image = new ConsistentAOTHomologousImage(); break;
        case HomologousImageMode::SUPERSET: image = new SuperSetAOTHomologousImage(); break;
        default: return LoadImageErrorCode::INVALID_HOMOLOGOUS_MODE;
        }

        LoadImageErrorCode err = image->Load((byte*)CopyBytes(dllBytes, dllSize), dllSize);
        if (err != LoadImageErrorCode::OK)
        {
            return err;
        }
        if (AOTHomologousImage::FindImageByAssemblyLocked(image->GetAOTAssembly(), lock))
        {
            return LoadImageErrorCode::HOMOLOGOUS_ASSEMBLY_HAS_BEEN_LOADED;
        }
        image->InitRuntimeMetadatas();
        AOTHomologousImage::RegisterLocked(image, lock);
        return LoadImageErrorCode::OK;
    }

    static bool IsDifferentialHybridAssembly(const char* assemblyName)
    {

        for (const char** ptrAotAss = g_differentialHybridAssemblies; *ptrAotAss; ptrAotAss++)
        {
            if (std::strcmp(*ptrAotAss, assemblyName) == 0)
            {
                return true;
            }
        }
        return false;
    }


    void Assembly::InitializeDifferentialHybridAssembles()
    {
        il2cpp::os::FastAutoLock lock(&il2cpp::vm::g_MetadataLock);

        std::vector<Il2CppAssembly*> dheAssemblies;
        for (const char** ptrAotAss = g_differentialHybridAssemblies; *ptrAotAss; ptrAotAss++)
        {
            Il2CppAssembly* aotAss = const_cast<Il2CppAssembly*>(il2cpp::vm::Assembly::GetOriginDifferentialHybridAssembly(*ptrAotAss));
            if (!aotAss)
            {
                continue;
            }
            Il2CppAssembly* dheAss = FindPlaceHolderAssemblyLocked(*ptrAotAss, lock);
            dheAssemblies.push_back(dheAss);
            aotAss->dheAssembly = dheAss;
            aotAss->originAssembly = nullptr;
            dheAss->originAssembly = aotAss;
            dheAss->dheAssembly = nullptr;
        }
        //for (Il2CppAssembly* ass : dheAssemblies)
        //{
        //    il2cpp::vm::Assembly::Register(ass);
        //}
    }

    LoadImageErrorCode Assembly::LoadDifferentialHybridAssembly(const void* assemblyData, uint32_t length, const void* optionData, uint32_t optionCount)
    {
        il2cpp::os::FastAutoLock lock(&il2cpp::vm::g_MetadataLock);
        
        DifferentialHybridOption options = {};

        BlobReader reader((byte*)optionData, optionCount);
        if (!options.Unmarshal(reader))
        {
            return LoadImageErrorCode::DHE_BAD_OPTION_DATA;
        }

        uint32_t imageId = InterpreterImage::AllocImageIndex(length);
        if (imageId > kMaxMetadataImageCount)
        {
            il2cpp::vm::Exception::Raise(il2cpp::vm::Exception::GetArgumentException("[Assembly::LoadDifferentialHybridAssebmly] exceed max image index", ""));
        }

        assemblyData = (const byte*)CopyBytes(assemblyData, length);

        DifferentialHybridImage* image = new DifferentialHybridImage(imageId, options);
        LoadImageErrorCode err = image->Load(assemblyData, (size_t)length);

        if (err != LoadImageErrorCode::OK)
        {
            HYBRIDCLR_FREE((void*)assemblyData);
            // when load a bad image, mean a fatal error. we don't clean image on purpose.
            return err;
        }

        TbAssembly data = image->GetRawImage().ReadAssembly(1);
        const char* nameNoExt = image->GetStringFromRawIndex(data.name);

        if (!IsDifferentialHybridAssembly(nameNoExt))
        {
            return LoadImageErrorCode::DHE_NOT_DIFFERENTIAL_HYBRID_ASSEMBLY;
        }

        Il2CppAssembly* ass = FindPlaceHolderAssemblyLocked(nameNoExt, lock);
        IL2CPP_ASSERT(ass);
        if (ass->token)
        {
            return LoadImageErrorCode::DHE_HAS_BEEN_LOADED;
        }

        Il2CppImage* image2 = ass->image;
        image->InitBasic(image2);
        image->BuildIl2CppAssembly(ass);
        image->BuildIl2CppImage(image2, ass->originAssembly);
        image2->name = ConcatNewString(ass->aname.name, ".dll");
        image2->nameNoExt = ass->aname.name;

        image->InitRuntimeMetadatas();
        il2cpp::vm::Assembly::InvalidateAssemblyList();

        RunModuleInitializer(ass->image);
        return LoadImageErrorCode::OK;
    }
}
}

