// 检查是否在Electron环境中
const isElectron = (function() {
    try {
        return typeof require !== 'undefined' && 
               typeof window !== 'undefined' && 
               window.process && 
               window.process.type;
    } catch (error) {
        return false;
    }
})();

let ipcRenderer, fs, path;
if (isElectron) {
    ({ ipcRenderer } = require('electron'));
    fs = require('fs');
    path = require('path');
}

class BossTimer {
    constructor() {
        this.bossInfo = null; // 存储sBossInfo.json的数据
        this.config = null;
        this.langsEn = null; // 存储langs_en.json的数据
        this.activeTimers = new Map(); // 存储活动的计时器
        this.timerIdCounter = 0;
        this.timezone = localStorage.getItem('timezone') || '0'; // 从本地存储加载时区设置
        this.selectedFilePath = localStorage.getItem('selectedBossFile') || null; // 存储选择的文件路径
        this.selectedFileContent = null; // 浏览器环境中存储文件内容
        this.selectedFileName = null; // 浏览器环境中存储文件名
        this.fileReadInterval = null; // 定时读取文件的间隔
        
        // 通知相关变量
        this.notifiedBosses = new Set(); // 记录已经发送过通知的Boss
        this.lastNotificationTime = {}; // 记录每个Boss上次发送通知的时间
        
        // Debug模式
        this.debugMode = false;
        this.originalBossInfo = null; // 保存原始Boss数据
        
        // 收藏功能
        this.favoriteBosses = new Set(JSON.parse(localStorage.getItem('favoriteBosses') || '[]')); // 收藏的Boss ID列表
        
        // 筛选功能
        this.currentFilter = 'all'; // 当前筛选类型: 'all', 'MVP', 'Mini'
        
        // 顶部计时器
        this.topTimers = {
            '1min': {
                duration: 60,
                remaining: 60,
                isRunning: false,
                interval: null
            },
            '2min': {
                duration: 120,
                remaining: 120,
                isRunning: false,
                interval: null
            }
        };
        
        this.initializeElements();
        this.initialize();
        this.bindEvents();
        this.startMainLoop();
        this.initializeTimezone();
        this.startClock();
        this.initializeFileSelector();
        this.startFileWatcher();
        this.initializeTopTimers();
    }
    
    async initialize() {
        try {
            // 按顺序加载配置和语言文件
            await this.loadConfig();
            await this.loadLangsEn();
            await this.loadBossInfo();
        } catch (error) {
            console.error('初始化失败:', error);
        }
    }
    
    initializeElements() {
        this.mvpList = document.getElementById('mvp-list');
        this.miniList = document.getElementById('mini-list');
        this.activeTimersList = document.getElementById('active-timers-list');
        
        // 时区设置元素
        this.timezoneInput = document.getElementById('timezone');
        this.saveTimezoneBtn = document.getElementById('save-timezone');
        this.currentTimeDisplay = document.getElementById('current-time-display');
        
        // Debug模式元素
        this.debugModeCheckbox = document.getElementById('debug-mode');
        this.testNotificationBtn = document.getElementById('test-notification');
        
        // 文件选择元素
        this.selectFileBtn = document.getElementById('select-file');
        this.bossFileInput = document.getElementById('boss-file');
        this.fileStatus = document.getElementById('file-status');
        this.filePathDiv = document.getElementById('file-path');
        this.filePathText = document.getElementById('file-path-text');
        
        // 弹窗元素
        this.modal = document.getElementById('timer-modal');
        this.modalBossName = document.getElementById('modal-boss-name');
        this.modalMinutes = document.getElementById('modal-minutes');
        this.modalSeconds = document.getElementById('modal-seconds');
        this.modalTimerText = document.getElementById('modal-timer-text');
        this.modalStartBtn = document.getElementById('modal-start-btn');
        this.modalCancelBtn = document.getElementById('modal-cancel-btn');
        this.closeBtn = document.querySelector('.close');
        
        // 顶部计时器元素
        this.timer1minDisplay = document.getElementById('timer-1min');
        this.timer2minDisplay = document.getElementById('timer-2min');
        this.start1minBtn = document.getElementById('start-1min');
        this.reset1minBtn = document.getElementById('reset-1min');
        this.start2minBtn = document.getElementById('start-2min');
        this.reset2minBtn = document.getElementById('reset-2min');
        
        // 筛选按钮元素
        this.filterAllBtn = document.getElementById('filter-all');
        this.filterMvpBtn = document.getElementById('filter-mvp');
        this.filterMiniBtn = document.getElementById('filter-mini');
        
        this.currentEditingBoss = null;
    }
    


    // 发送Boss刷新提醒通知
    async sendBossNotification(bossName, remainingTime) {
        console.log(`尝试发送通知: ${bossName}, 剩余时间: ${remainingTime}秒`);
        
        const minutes = Math.floor(remainingTime / 60);
        const seconds = remainingTime % 60;
        
        let title, body;
        if (remainingTime <= 0) {
            title = `${bossName} 已刷新！`;
            body = `${bossName} 现在可以挑战了！`;
        } else {
            title = `${bossName} 即将刷新`;
            body = `还有 ${minutes > 0 ? `${minutes}分` : ''}${seconds}秒 刷新`;
        }
        
        console.log(`发送通知: ${title} - ${body}`);
        
        try {
            if (isElectron && ipcRenderer) {
                // 使用Electron的IPC发送系统通知
                const result = await ipcRenderer.invoke('send-notification', title, body);
                if (result.success) {
                    console.log('系统通知发送成功');
                } else {
                    console.error('系统通知发送失败:', result.message);
                }
                return result;
            } else {
                // 浏览器环境使用Web Notification API
                if ('Notification' in window) {
                    if (Notification.permission === 'granted') {
                        new Notification(title, { body, icon: 'data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><text y="0.9em" font-size="90">⏰</text></svg>' });
                        return { success: true };
                    } else if (Notification.permission !== 'denied') {
                        const permission = await Notification.requestPermission();
                        if (permission === 'granted') {
                            new Notification(title, { body, icon: 'data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><text y="0.9em" font-size="90">⏰</text></svg>' });
                            return { success: true };
                        }
                    }
                }
                console.log('浏览器通知不可用，仅控制台显示');
                return { success: false, message: '浏览器通知不可用' };
            }
        } catch (error) {
            console.error('发送通知时出错:', error);
            return { success: false, message: error.message };
        }
    }

    // 创建Debug模式的测试Boss数据
    createDebugBossData() {
        if (!this.bossInfo || !this.config) return;
        
        console.log('开始创建Debug测试数据');
        
        // 获取当前时间戳
        const now = Math.floor(Date.now() / 1000);
        
        // 按照配置文件中Boss的顺序，设置测试时间
        let debugTimeIndex = 0;
        
        this.config.BOSS.forEach(bossConfig => {
            const bossInfoItem = this.bossInfo.find(item => item.id === bossConfig.id);
            if (bossInfoItem) {
                // 从55秒开始，每个Boss递增20秒
                // 这样第一个Boss是55秒（可以测试通知），第二个75秒，第三个95秒...
                const testRemainingTime = 55 + (debugTimeIndex * 20);
                
                // 计算新的deadTime：当前时间 - (刷新时间 - 测试剩余时间)
                const newDeadTime = now - (bossConfig.fixedRefreshTime - testRemainingTime);
                
                bossInfoItem.deadTime = newDeadTime;
                bossInfoItem.alive = false; // 确保Boss未活着，这样才会有倒计时
                
                console.log(`Debug: Boss ${bossConfig.name} (ID: ${bossConfig.id}) 设置为 ${testRemainingTime} 秒后刷新`);
                debugTimeIndex++;
            }
        });
        
        console.log('Debug测试数据创建完成');
    }

    // 测试通知功能
    async testNotification() {
        console.log('开始测试通知功能');
        
        try {
            await this.sendTestNotification();
        } catch (error) {
            console.error('测试通知失败:', error);
            alert('测试通知发送失败，请检查控制台错误信息');
        }
    }
    
    // 发送测试通知
    async sendTestNotification() {
        const result = await this.sendBossNotification('测试Boss', 30);
        console.log('测试通知已发送');
        
        // 根据结果显示相应的消息
        if (result.success) {
            const message = isElectron ? 
                '测试系统通知已发送成功！请查看系统通知。' : 
                '测试浏览器通知已发送成功！请查看浏览器通知。';
            alert(message);
        } else {
            alert(`测试通知发送失败: ${result.message}`);
        }
    }
    
    initializeTimezone() {
        // 设置输入框的初始值
        this.timezoneInput.value = this.timezone;
    }
    
    // 初始化文件选择器
    initializeFileSelector() {
        if (this.selectedFilePath) {
            const fileName = this.selectedFilePath.split('\\').pop() || this.selectedFilePath.split('/').pop();
            this.fileStatus.textContent = `已选择: ${fileName}`;
            this.fileStatus.className = 'selected';
            
            // 显示完整文件路径
            this.filePathText.textContent = this.selectedFilePath;
            this.filePathDiv.style.display = 'block';
        }
    }
    
    // 启动文件监控
    startFileWatcher() {
        // 清除之前的间隔
        if (this.fileReadInterval) {
            clearInterval(this.fileReadInterval);
        }
        
        // 如果有选择的文件且不在debug模式，每5秒读取一次
        if (this.selectedFilePath && !this.debugMode) {
            this.fileReadInterval = setInterval(() => {
                this.loadBossInfoFromFile();
            }, 5000);
        }
    }
    
    // 从指定文件加载Boss信息
    async loadBossInfoFromFile() {
        if (!this.selectedFilePath && !this.selectedFileContent) {
            return this.loadBossInfo(); // 回退到默认方法
        }
        
        try {
            let data;
            if (isElectron && ipcRenderer && this.selectedFilePath) {
                // Electron环境：使用IPC读取文件
                data = await ipcRenderer.invoke('read-file-content', this.selectedFilePath);
            } else if (this.selectedFileContent) {
                // 浏览器环境：使用已读取的文件内容
                data = this.selectedFileContent;
            } else {
                throw new Error('无法读取文件内容');
            }
            
            this.bossInfo = JSON.parse(data);
            this.updateBossRemainingTime();
            
            // 更新状态显示
            const fileName = this.selectedFilePath ? 
                (this.selectedFilePath.split('\\').pop() || this.selectedFilePath.split('/').pop()) :
                this.selectedFileName || '已选择文件';
            this.fileStatus.textContent = `已选择: ${fileName} (最后更新: ${new Date().toLocaleTimeString()})`;
            this.fileStatus.className = 'selected';
            
            // 显示完整文件路径
            if (this.selectedFilePath) {
                this.filePathText.textContent = this.selectedFilePath;
                this.filePathDiv.style.display = 'block';
            }
        } catch (error) {
            console.error('读取指定Boss信息文件失败:', error);
            this.fileStatus.textContent = `读取失败: ${error.message}`;
            this.fileStatus.className = 'error';
        }
    }
    
    // 启动时钟
    startClock() {
        this.updateClock(); // 立即更新一次
        setInterval(() => this.updateClock(), 1000); // 每秒更新一次
    }
    
    // 更新时钟显示
    updateClock() {
        const now = new Date();
        const utc = now.getTime() + (now.getTimezoneOffset() * 60000);
        const timezoneTime = new Date(utc + (3600000 * this.timezone));
        
        const timeString = timezoneTime.toLocaleTimeString('zh-CN', {
            hour: '2-digit',
            minute: '2-digit',
            second: '2-digit',
            hour12: false
        });
        
        const dateString = timezoneTime.toLocaleDateString('zh-CN', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit'
        });
        
        this.currentTimeDisplay.textContent = `${dateString} ${timeString}`;
        
        // 更新Boss剩余时间
        if (this.bossInfo) {
            this.updateBossRemainingTime();
            
            // 检查是否需要重新加载Boss信息（每分钟检查一次，debug模式下跳过）
            if (!this.debugMode) {
                const currentSeconds = Math.floor(Date.now() / 1000);
                if (!this.lastCheckTime || currentSeconds - this.lastCheckTime >= 60) {
                    this.lastCheckTime = currentSeconds;
                    this.loadBossInfo();
                }
            }
        }
    }
    
    // 加载sBossInfo.json数据
    async loadBossInfo() {
        // Debug模式下跳过文件读取
        if (this.debugMode) {
            console.log('Debug模式下跳过文件读取');
            return;
        }
        
        try {
            if (this.selectedFilePath) {
                // 如果有选择的文件，使用指定文件
                await this.loadBossInfoFromFile();
            } else {
                if (isElectron && ipcRenderer) {
                    // Electron环境：使用IPC
                    this.bossInfo = await ipcRenderer.invoke('get-boss-info');
                } else {
                    // 浏览器环境：从网络获取
                    const response = await fetch('./sBossInfo.json');
                    this.bossInfo = await response.json();
                }
                this.updateBossRemainingTime();
            }
        } catch (error) {
            console.error('加载Boss信息失败:', error);
            // 浏览器环境显示提示信息
            if (!isElectron) {
                this.mvpList.innerHTML = '<div class="no-timers">无法加载Boss信息，请确保sBossInfo.json文件存在</div>';
                this.miniList.innerHTML = '<div class="no-timers">无法加载Boss信息，请确保sBossInfo.json文件存在</div>';
            }
        }
    }
    
    // 更新Boss剩余时间
    updateBossRemainingTime() {
        if (!this.bossInfo || !this.config) return;
        
        // 获取当前时间戳（秒）
        const now = Math.floor(Date.now() / 1000);
        
        // 创建Boss信息数组，用于排序
        const bossTimers = [];
        
        // 遍历所有配置中的Boss
        this.config.BOSS.forEach(bossConfig => {
            const bossId = bossConfig.id;
            const bossType = bossConfig.type;
            const bossName = bossConfig.name;
            const bossInfoItem = this.bossInfo.find(item => item.id === bossId);
            
            if (bossInfoItem) {
                // 计算时间差（秒）
                const deltaTime = now - bossInfoItem.deadTime;
                
                // 从配置文件中获取该Boss的固定刷新时间
                const bossConfig = this.config.BOSS.find(boss => boss.id === bossId);
                let fixedRefreshTime = 7200; // 默认2小时
                
                if (bossConfig && bossConfig.fixedRefreshTime) {
                    fixedRefreshTime = bossConfig.fixedRefreshTime;
                } else {
                    // 如果配置中没有具体设置，使用类型默认值
                    if (bossType === 'MVP') {
                        fixedRefreshTime = 10800; // MVP Boss 3小时
                    } else if (bossType === 'Mini') {
                        fixedRefreshTime = 7200; // Mini Boss 2小时
                    }
                }
                
                // 计算下次刷新剩余时间
                const nextRefreshRemainingTime = fixedRefreshTime - deltaTime;
                
                // 计算下次刷新的具体时间点
                const nextRefreshTime = new Date((bossInfoItem.deadTime + fixedRefreshTime) * 1000);
                
                // 考虑时区
                const utc = nextRefreshTime.getTime() + (nextRefreshTime.getTimezoneOffset() * 60000);
                const timezoneNextRefreshTime = new Date(utc + (3600000 * this.timezone));
                
                // 格式化下次刷新时间
                const nextRefreshTimeString = timezoneNextRefreshTime.toLocaleTimeString('zh-CN', {
                    hour: '2-digit',
                    minute: '2-digit',
                    second: '2-digit',
                    hour12: false
                });
                
                // 计算+8时区（北京时间）的刷新时间
                const utcBeijing = nextRefreshTime.getTime() + (nextRefreshTime.getTimezoneOffset() * 60000);
                const beijingTime = new Date(utcBeijing + (3600000 * 8));
                const beijingTimeString = beijingTime.toLocaleTimeString('zh-CN', {
                    hour: '2-digit',
                    minute: '2-digit',
                    second: '2-digit',
                    hour12: false
                });
                
                // 添加到Boss信息数组
                bossTimers.push({
                    id: bossId,
                    name: bossName,
                    type: bossType,
                    remainingTime: nextRefreshRemainingTime,
                    refreshTimeString: nextRefreshTimeString,
                    beijingTimeString: beijingTimeString,
                    alive: bossInfoItem.alive,
                    isManualTimer: false
                });
                
                // 检查是否需要发送通知（小于1分钟且未活着）
                if (!bossInfoItem.alive && nextRefreshRemainingTime > 0 && nextRefreshRemainingTime <= 60) {
                    const notificationKey = `${bossId}_under_1min`; // 每个Boss只有一个"小于1分钟"的通知
                    
                    console.log(`Boss ${bossName} 剩余时间: ${nextRefreshRemainingTime}秒, 通知密钥: ${notificationKey}`);
                    console.log(`已通知列表:`, Array.from(this.notifiedBosses));
                    
                    // 检查是否已经发送过"小于1分钟"的通知
                    if (!this.notifiedBosses.has(notificationKey)) {
                        console.log(`发送通知给 ${bossName}`);
                        this.sendBossNotification(bossName, nextRefreshRemainingTime);
                        this.notifiedBosses.add(notificationKey);
                        
                        // 2分钟后清理通知记录（确保Boss刷新后能重新通知）
                        setTimeout(() => {
                            this.notifiedBosses.delete(notificationKey);
                            console.log(`清理通知记录: ${notificationKey}`);
                        }, 120000); // 2分钟后清理
                    } else {
                        console.log(`Boss ${bossName} 已经发送过小于1分钟通知: ${notificationKey}`);
                    }
                }
            }
        });
        
        // 添加手动创建的计时器
        this.activeTimers.forEach(timer => {
            // 计算目标时间（考虑时区）
            const now = new Date();
            const utc = now.getTime() + (now.getTimezoneOffset() * 60000);
            const currentTime = new Date(utc + (3600000 * this.timezone));
            const targetTime = new Date(currentTime.getTime() + (timer.remainingSeconds * 1000));
            
            const targetTimeString = targetTime.toLocaleTimeString('zh-CN', {
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
            });
            
            // 计算+8时区（北京时间）的结束时间
            const utcBeijingManual = targetTime.getTime() + (targetTime.getTimezoneOffset() * 60000);
            const beijingTimeManual = new Date(utcBeijingManual + (3600000 * 8));
            const beijingTimeStringManual = beijingTimeManual.toLocaleTimeString('zh-CN', {
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit',
                hour12: false
            });
            
            bossTimers.push({
                id: `manual_${timer.id}`,
                name: `${timer.bossName} (手动)`,
                type: 'Manual',
                remainingTime: timer.remainingSeconds,
                refreshTimeString: targetTimeString,
                beijingTimeString: beijingTimeStringManual,
                alive: false,
                isManualTimer: true,
                timer: timer
            });
            
            // 检查手动计时器是否需要发送通知（小于1分钟）
            if (timer.remainingSeconds > 0 && timer.remainingSeconds <= 60) {
                const notificationKey = `manual_${timer.id}_under_1min`; // 每个手动计时器只有一个"小于1分钟"的通知
                
                if (!this.notifiedBosses.has(notificationKey)) {
                    this.sendBossNotification(`${timer.bossName} (手动)`, timer.remainingSeconds);
                    this.notifiedBosses.add(notificationKey);
                    
                    // 2分钟后清理通知记录
                    setTimeout(() => {
                        this.notifiedBosses.delete(notificationKey);
                    }, 120000);
                }
            }
        });
        
        // 更新右侧倒计时列表
        this.updateCountdownList(bossTimers);
    }

    // 更新右侧倒计时列表
    updateCountdownList(bossTimers) {
        // 获取倒计时列表容器
        const countdownList = document.getElementById('active-timers-list');
        if (!countdownList) return;
        
        // 根据当前筛选类型过滤boss列表
        let filteredBossTimers = bossTimers;
        if (this.currentFilter !== 'all') {
            filteredBossTimers = bossTimers.filter(boss => {
                // 手动计时器通过关联的boss类型筛选
                if (boss.isManualTimer) {
                    const relatedBossConfig = this.config && this.config.BOSS ? 
                        this.config.BOSS.find(b => b.id === boss.timer.bossId) : null;
                    return relatedBossConfig && relatedBossConfig.type === this.currentFilter;
                }
                // 普通boss通过type字段筛选
                return boss.type === this.currentFilter;
            });
        }
        
        // 如果没有Boss信息且当前没有任何倒计时项，显示提示
        if (filteredBossTimers.length === 0) {
            const existingCountdownItems = countdownList.querySelectorAll('.countdown-item');
            if (existingCountdownItems.length === 0 && !countdownList.querySelector('.no-timers')) {
                let message = '暂无Boss信息';
                if (this.currentFilter !== 'all') {
                    message = `暂无${this.currentFilter}类型的Boss`;
                }
                countdownList.innerHTML = `<div class="no-timers">${message}</div>`;
            }
            return;
        }
        
        // 如果有Boss信息，移除"暂无Boss信息"提示（如果存在）
        const noTimers = countdownList.querySelector('.no-timers');
        if (noTimers) {
            noTimers.remove();
        }
        
        // 格式化剩余时间为HH:MM:SS
        const formatRemainingTime = (seconds) => {
            if (seconds <= 0) return '00:00:00';
            const hours = Math.floor(seconds / 3600);
            const minutes = Math.floor((seconds % 3600) / 60);
            const secs = seconds % 60;
            return `${String(hours).padStart(2, '0')}:${String(minutes).padStart(2, '0')}:${String(secs).padStart(2, '0')}`;
        };
        
        // 按收藏状态和剩余时间排序
        filteredBossTimers.sort((a, b) => {
            // 提取Boss ID
            const aBossId = a.isManualTimer ? a.timer.bossId : a.id;
            const bBossId = b.isManualTimer ? b.timer.bossId : b.id;
            
            const aIsFavorite = this.favoriteBosses.has(aBossId);
            const bIsFavorite = this.favoriteBosses.has(bBossId);
            
            // 收藏的Boss排在前面
            if (aIsFavorite && !bIsFavorite) return -1;
            if (!aIsFavorite && bIsFavorite) return 1;
            
            // 已刷新的Boss排在最前面（在收藏状态相同的情况下）
            if (a.alive && !b.alive) return -1;
            if (!a.alive && b.alive) return 1;
            
            // 然后按剩余时间排序
            return a.remainingTime - b.remainingTime;
        });
        
        // 获取当前所有的倒计时项
        const currentItems = Array.from(countdownList.querySelectorAll('.countdown-item'));
        const existingItems = {};
        
        currentItems.forEach(item => {
            const bossId = item.dataset.bossId;
            if (bossId) {
                existingItems[bossId] = item;
            }
        });
        
        // 只更新发生变化的元素，避免重新排序
        const updatedElements = [];
        
        filteredBossTimers.forEach((boss, index) => {
            let element = existingItems[boss.id];
            
            if (!element) {
                // 创建新元素
                element = document.createElement('div');
                element.className = 'countdown-item';
                element.dataset.bossId = boss.id;
                
                // 为非手动计时器添加点击事件
                if (!boss.isManualTimer) {
                    element.addEventListener('click', () => {
                        this.openTimerModal(boss.name, boss.id);
                    });
                    element.style.cursor = 'pointer';
                }
            }
            
            // 更新元素类名
            const newClassList = ['countdown-item'];
            
            // 获取实际的Boss ID（手动计时器和普通Boss的处理方式不同）
            const actualBossId = boss.isManualTimer ? boss.timer.bossId : boss.id;
            
            // 检查是否为收藏的Boss
            if (this.favoriteBosses.has(actualBossId)) {
                newClassList.push('favorited');
            }
            
            if (boss.remainingTime <= 0) {
                if (boss.alive) {
                    newClassList.push('alive');
                } else {
                    newClassList.push('overtime');
                }
            } else if (boss.remainingTime < 300) {
                newClassList.push('soon');
            }
            
            const newClassName = newClassList.join(' ');
            if (element.className !== newClassName) {
                element.className = newClassName;
            }
            
            // 生成新内容
            let newContent = '';
            if (boss.remainingTime <= 0) {
                if (boss.alive) {
                    newContent = `
                        <div class="countdown-name">${this.getBossNameDisplay(boss.name)}</div>
                        <div class="countdown-time">
                            <div class="boss-status alive">已刷新</div>
                        </div>
                        <button class="favorite-btn" onclick="event.stopPropagation(); bossTimer.toggleFavorite(${actualBossId}, this)">
                            ${this.favoriteBosses.has(actualBossId) ? '★' : '☆'}
                        </button>
                    `;
                } else {
                    const overtimeSeconds = Math.abs(boss.remainingTime);
                    const overtimeFormatted = formatRemainingTime(overtimeSeconds);
                    newContent = `
                        <div class="countdown-name">${this.getBossNameDisplay(boss.name)}</div>
                        <div class="countdown-time">
                            <div class="boss-status overtime">已超时</div>
                            <div class="countdown-value">${overtimeFormatted}</div>
                        </div>
                        <button class="favorite-btn" onclick="event.stopPropagation(); bossTimer.toggleFavorite(${actualBossId}, this)">
                            ${this.favoriteBosses.has(actualBossId) ? '★' : '☆'}
                        </button>
                    `;
                }
            } else {
                const remainingFormatted = formatRemainingTime(boss.remainingTime);
                newContent = `
                    <div class="countdown-name">${this.getBossNameDisplay(boss.name)}</div>
                    <div class="countdown-time">
                        <div class="countdown-refresh">刷新: ${boss.refreshTimeString} (${boss.beijingTimeString})</div>
                        <div class="countdown-value ${boss.remainingTime < 300 ? 'soon' : ''}">${remainingFormatted}</div>
                    </div>
                    <button class="favorite-btn" onclick="event.stopPropagation(); bossTimer.toggleFavorite(${actualBossId}, this)">
                        ${this.favoriteBosses.has(actualBossId) ? '★' : '☆'}
                    </button>
                `;
                
                // 如果是手动计时器，添加控制按钮
                if (boss.isManualTimer) {
                    // 从手动计时器名称中提取原始boss名称 (移除"(手动)"后缀)
                    const originalBossName = boss.timer.bossName;
                    newContent = `
                        <div class="countdown-name">${this.getBossNameDisplay(originalBossName)} <span class="manual-timer-label">(手动)</span></div>
                        <div class="countdown-time">
                            <div class="countdown-refresh">结束: ${boss.refreshTimeString} (${boss.beijingTimeString})</div>
                            <div class="countdown-value ${boss.remainingTime < 300 ? 'soon' : ''}">${remainingFormatted}</div>
                        </div>
                        <div class="countdown-controls">
                            <button class="btn btn-small btn-secondary" onclick="bossTimer.pauseTimer(${boss.timer.id})">
                                ${boss.timer.isPaused ? '继续' : '暂停'}
                            </button>
                            <button class="btn btn-small btn-danger" onclick="bossTimer.removeTimer(${boss.timer.id})">
                                删除
                            </button>
                            <button class="favorite-btn" onclick="event.stopPropagation(); bossTimer.toggleFavorite(${actualBossId}, this)">
                                ${this.favoriteBosses.has(actualBossId) ? '★' : '☆'}
                            </button>
                        </div>
                    `;
                }
            }
            
            // 只有内容真正改变时才更新innerHTML
            if (element.innerHTML.trim() !== newContent.trim()) {
                element.innerHTML = newContent;
            }
            
            updatedElements.push(element);
            
            // 标记已处理
            delete existingItems[boss.id];
        });
        
        // 移除不再需要的元素（包括被筛选掉的元素）
        Object.values(existingItems).forEach(item => {
            if (item.parentNode) {
                item.parentNode.removeChild(item);
            }
        });
        
        // 如果筛选改变了，还需要移除不符合当前筛选条件的元素
        if (this.currentFilter !== 'all') {
            const allItems = Array.from(countdownList.querySelectorAll('.countdown-item'));
            allItems.forEach(item => {
                const bossId = item.dataset.bossId;
                const shouldKeep = filteredBossTimers.some(boss => boss.id === bossId);
                if (!shouldKeep && item.parentNode) {
                    item.parentNode.removeChild(item);
                }
            });
        }
        
        // 检查是否需要重新排序
        const currentOrder = Array.from(countdownList.querySelectorAll('.countdown-item'));
        let needsReorder = false;
        
        if (currentOrder.length !== updatedElements.length) {
            needsReorder = true;
        } else {
            for (let i = 0; i < updatedElements.length; i++) {
                if (currentOrder[i] !== updatedElements[i]) {
                    needsReorder = true;
                    break;
                }
            }
        }
        
        // 只在需要时才重新排序或添加新元素
        if (needsReorder) {
            // 记录滚动位置
            const scrollTop = countdownList.scrollTop;
            
            // 使用文档片段来减少重排
            const fragment = document.createDocumentFragment();
            updatedElements.forEach(element => {
                // 如果元素不在DOM中，或者需要重新排序
                if (!element.parentNode || element.parentNode !== countdownList) {
                    fragment.appendChild(element);
                }
            });
            
            // 如果有新元素需要添加，一次性添加
            if (fragment.children.length > 0) {
                countdownList.appendChild(fragment);
            }
            
            // 重新排序现有元素
            updatedElements.forEach(element => {
                if (element.parentNode === countdownList) {
                    countdownList.appendChild(element);
                }
            });
            
            // 恢复滚动位置
            countdownList.scrollTop = scrollTop;
        }
    }
    
    async loadConfig() {
        try {
            if (isElectron && ipcRenderer) {
                // Electron环境：使用IPC
                this.config = await ipcRenderer.invoke('get-config');
            } else {
                // 浏览器环境：从网络获取
                const response = await fetch('./config.json');
                this.config = await response.json();
            }
            this.renderBossLists();
        } catch (error) {
            console.error('加载配置失败:', error);
        }
    }
    
    async loadLangsEn() {
        try {
            if (isElectron && ipcRenderer) {
                // Electron环境：使用IPC
                this.langsEn = await ipcRenderer.invoke('get-langs-en');
                console.log('Electron环境下加载英文名称成功:', this.langsEn);
            } else {
                // 浏览器环境：从网络获取
                const response = await fetch('./langs_en.json');
                if (response.ok) {
                    this.langsEn = await response.json();
                    console.log('浏览器环境下加载英文名称成功:', this.langsEn);
                } else {
                    console.warn('langs_en.json文件不存在，将不显示英文名称');
                    this.langsEn = {};
                }
            }
        } catch (error) {
            console.warn('加载英文名称失败:', error);
            this.langsEn = {};
        }
    }
    
    // 获取Boss名称的显示HTML（包括中文和英文名称）
    getBossNameDisplay(chineseName) {
        // 添加调试日志
        if (!this.langsEn) {
            console.log('langsEn 未加载，显示纯中文名称:', chineseName);
            return `<div class="boss-name-zh">${chineseName}</div>`;
        }
        
        const englishName = this.langsEn[chineseName];
        console.log(`Boss名称映射: ${chineseName} -> ${englishName}`);
        
        if (englishName) {
            return `
                <div class="boss-name-zh">${chineseName}</div>
                <div class="boss-name-en">${englishName}</div>
            `;
        } else {
            console.log(`未找到 ${chineseName} 的英文名称`);
            return `<div class="boss-name-zh">${chineseName}</div>`;
        }
    }
    
    renderBossLists() {
        // 过滤并渲染MVP Boss列表
        this.mvpList.innerHTML = '';
        const mvpBosses = this.config.BOSS.filter(boss => boss.type === 'MVP');
        mvpBosses.forEach(boss => {
            const bossElement = this.createBossElement(boss.name, boss.id, 'mvp');
            this.mvpList.appendChild(bossElement);
        });
        
        // 过滤并渲染Mini Boss列表
        this.miniList.innerHTML = '';
        const miniBosses = this.config.BOSS.filter(boss => boss.type === 'Mini');
        miniBosses.forEach(boss => {
            const bossElement = this.createBossElement(boss.name, boss.id, 'mini');
            this.miniList.appendChild(bossElement);
        });
        
        // 如果已加载Boss信息，立即更新剩余时间
        if (this.bossInfo) {
            this.updateBossRemainingTime();
        }
    }
    
    createBossElement(bossName, bossId, type) {
        const element = document.createElement('div');
        element.className = 'boss-item';
        
        const leftContent = document.createElement('div');
        leftContent.className = 'boss-left-content';
        
        const nameElement = document.createElement('span');
        nameElement.className = 'boss-name';
        nameElement.textContent = bossName;
        
        const timeElement = document.createElement('span');
        timeElement.className = 'boss-time';
        timeElement.textContent = '加载中...';
        
        leftContent.appendChild(nameElement);
        leftContent.appendChild(timeElement);
        
        // 创建收藏按钮
        const favoriteBtn = document.createElement('button');
        favoriteBtn.className = 'favorite-btn';
        favoriteBtn.innerHTML = this.favoriteBosses.has(bossId) ? '★' : '☆';
        favoriteBtn.title = this.favoriteBosses.has(bossId) ? '取消收藏' : '收藏';
        
        // 收藏按钮点击事件
        favoriteBtn.addEventListener('click', (e) => {
            e.stopPropagation(); // 阻止事件冒泡
            this.toggleFavorite(bossId, favoriteBtn);
        });
        
        element.appendChild(leftContent);
        element.appendChild(favoriteBtn);
        
        element.dataset.boss = bossName;
        element.dataset.bossId = bossId;
        element.dataset.type = type;
        
        // 如果是收藏的Boss，添加收藏样式
        if (this.favoriteBosses.has(bossId)) {
            element.classList.add('favorited');
        }
        
        element.addEventListener('click', () => {
            this.openTimerModal(bossName, bossId);
        });
        
        return element;
    }
    
    // 切换收藏状态
    toggleFavorite(bossId, favoriteBtn) {
        if (this.favoriteBosses.has(bossId)) {
            // 取消收藏
            this.favoriteBosses.delete(bossId);
            favoriteBtn.innerHTML = '☆';
            favoriteBtn.title = '收藏';
            
            // 更新所有相关元素的样式
            document.querySelectorAll(`.boss-item[data-boss-id="${bossId}"]`).forEach(el => {
                el.classList.remove('favorited');
                const btn = el.querySelector('.favorite-btn');
                if (btn) {
                    btn.innerHTML = '☆';
                    btn.title = '收藏';
                }
            });
            
            document.querySelectorAll(`.countdown-item[data-boss-id="${bossId}"]`).forEach(el => {
                el.classList.remove('favorited');
                const btn = el.querySelector('.favorite-btn');
                if (btn) {
                    btn.innerHTML = '☆';
                    btn.title = '收藏';
                }
            });
        } else {
            // 添加收藏
            this.favoriteBosses.add(bossId);
            favoriteBtn.innerHTML = '★';
            favoriteBtn.title = '取消收藏';
            
            // 更新所有相关元素的样式
            document.querySelectorAll(`.boss-item[data-boss-id="${bossId}"]`).forEach(el => {
                el.classList.add('favorited');
                const btn = el.querySelector('.favorite-btn');
                if (btn) {
                    btn.innerHTML = '★';
                    btn.title = '取消收藏';
                }
            });
            
            document.querySelectorAll(`.countdown-item[data-boss-id="${bossId}"]`).forEach(el => {
                el.classList.add('favorited');
                const btn = el.querySelector('.favorite-btn');
                if (btn) {
                    btn.innerHTML = '★';
                    btn.title = '取消收藏';
                }
            });
        }
        
        // 保存到本地存储
        localStorage.setItem('favoriteBosses', JSON.stringify(Array.from(this.favoriteBosses)));
        
        // 立即更新右侧倒计时列表的排序
        if (this.bossInfo) {
            this.updateBossRemainingTime();
        }
    }
    
    openTimerModal(bossName, bossId) {
        this.currentEditingBoss = { name: bossName, id: bossId };
        this.modalBossName.textContent = bossName;
        this.modalMinutes.value = 5;
        this.modalSeconds.value = 0;
        this.updateModalDisplay();
        this.modal.style.display = 'block';
    }
    
    closeTimerModal() {
        this.modal.style.display = 'none';
        this.currentEditingBoss = null;
    }
    
    updateModalDisplay() {
        const minutes = parseInt(this.modalMinutes.value) || 0;
        const seconds = parseInt(this.modalSeconds.value) || 0;
        const totalSeconds = minutes * 60 + seconds;
        this.modalTimerText.textContent = this.formatTime(totalSeconds);
    }
    
    bindEvents() {
        // 时区设置事件
        this.saveTimezoneBtn.addEventListener('click', () => {
            const newTimezone = this.timezoneInput.value;
            if (newTimezone >= -12 && newTimezone <= 14) {
                this.timezone = newTimezone;
                localStorage.setItem('timezone', newTimezone);
                this.updateClock(); // 立即更新时钟显示
                alert('时区设置已保存！');
            } else {
                alert('请输入有效的时区值（-12 到 14）！');
            }
        });
        
        // 窗口置顶功能 - 仅在Electron环境中有效
        if (isElectron && ipcRenderer) {
            // 从本地存储加载置顶状态
            const isAlwaysOnTop = localStorage.getItem('alwaysOnTop') === 'true';
            if (isAlwaysOnTop) {
                // 应用启动时设置窗口置顶
                ipcRenderer.invoke('toggle-always-on-top', true);
            }
            
            // 监听菜单栏置顶状态变化
            ipcRenderer.on('always-on-top-changed', (event, isOnTop) => {
                localStorage.setItem('alwaysOnTop', isOnTop ? 'true' : 'false');
            });
        }
        
        // Debug模式事件
        this.debugModeCheckbox.addEventListener('change', (e) => {
            this.debugMode = e.target.checked;
            console.log(`Debug模式: ${this.debugMode ? '开启' : '关闭'}`);
            
            if (this.debugMode) {
                // 保存原始数据
                if (this.bossInfo) {
                    this.originalBossInfo = JSON.parse(JSON.stringify(this.bossInfo));
                    console.log('已保存原始Boss数据');
                    
                    // 修改Boss数据为测试数据
                    this.createDebugBossData();
                    
                    // 清空已发送通知记录，确保测试时能重新发送
                    this.notifiedBosses.clear();
                    console.log('已清空通知记录，准备测试通知功能');
                }
            } else {
                // 恢复原始数据
                if (this.originalBossInfo) {
                    this.bossInfo = JSON.parse(JSON.stringify(this.originalBossInfo));
                    console.log('已恢复原始Boss数据');
                    this.originalBossInfo = null;
                }
            }
            
            // 重新启动或停止文件监控
            this.startFileWatcher();
            
            // 立即更新Boss时间显示
            if (this.bossInfo) {
                this.updateBossRemainingTime();
            }
        });
        
        // 测试通知按钮事件
        this.testNotificationBtn.addEventListener('click', () => {
            this.testNotification();
        });
        
        // 顶部计时器事件
        this.start1minBtn.addEventListener('click', () => this.toggleTopTimer('1min'));
        this.reset1minBtn.addEventListener('click', () => this.resetTopTimer('1min'));
        this.start2minBtn.addEventListener('click', () => this.toggleTopTimer('2min'));
        this.reset2minBtn.addEventListener('click', () => this.resetTopTimer('2min'));
        
        // 文件选择事件
        this.selectFileBtn.addEventListener('click', () => {
            this.bossFileInput.click();
        });
        
        this.bossFileInput.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                if (file.name.endsWith('.json')) {
                    if (isElectron) {
                        // Electron环境：使用文件路径
                        this.selectedFilePath = file.path;
                        localStorage.setItem('selectedBossFile', this.selectedFilePath);
                        
                        this.fileStatus.textContent = `已选择: ${file.name}`;
                        this.fileStatus.className = 'selected';
                        
                        // 显示完整文件路径
                        this.filePathText.textContent = this.selectedFilePath;
                        this.filePathDiv.style.display = 'block';
                        
                        // 立即读取一次文件
                        this.loadBossInfoFromFile();
                        
                        // 启动定时读取
                        this.startFileWatcher();
                        
                        alert('文件选择成功！将每5秒自动更新数据。');
                    } else {
                        // 浏览器环境：读取文件内容
                        this.selectedFileName = file.name;
                        const reader = new FileReader();
                        reader.onload = (event) => {
                            this.selectedFileContent = event.target.result;
                            this.selectedFilePath = null; // 清除Electron路径
                            
                            this.fileStatus.textContent = `已选择: ${file.name}`;
                            this.fileStatus.className = 'selected';
                            
                            // 立即读取一次文件
                            this.loadBossInfoFromFile();
                            
                            alert('文件选择成功！注意：浏览器环境中不会自动更新文件内容。');
                        };
                        reader.onerror = () => {
                            this.fileStatus.textContent = '文件读取失败';
                            this.fileStatus.className = 'error';
                        };
                        reader.readAsText(file);
                    }
                } else {
                    alert('请选择JSON文件！');
                    this.fileStatus.textContent = '文件格式错误';
                    this.fileStatus.className = 'error';
                    
                    // 隐藏文件路径
                    this.filePathDiv.style.display = 'none';
                }
            }
        });
        
        // 弹窗事件
        this.modalMinutes.addEventListener('input', () => {
            this.updateModalDisplay();
        });
        
        this.modalSeconds.addEventListener('input', () => {
            this.updateModalDisplay();
        });
        
        this.modalStartBtn.addEventListener('click', () => {
            this.startNewTimer();
        });
        
        this.modalCancelBtn.addEventListener('click', () => {
            this.closeTimerModal();
        });
        
        this.closeBtn.addEventListener('click', () => {
            this.closeTimerModal();
        });
        
        // 点击弹窗外部关闭
        this.modal.addEventListener('click', (e) => {
            if (e.target === this.modal) {
                this.closeTimerModal();
            }
        });
        
        // 键盘事件
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeTimerModal();
            }
        });
        
        // 筛选按钮事件
        this.filterAllBtn.addEventListener('click', () => {
            this.setFilter('all');
        });
        
        this.filterMvpBtn.addEventListener('click', () => {
            this.setFilter('MVP');
        });
        
        this.filterMiniBtn.addEventListener('click', () => {
            this.setFilter('Mini');
        });
    }
    
    setFilter(filterType) {
        this.currentFilter = filterType;
        
        // 更新按钮样式
        const allFilterButtons = document.querySelectorAll('.filter-btn');
        allFilterButtons.forEach(btn => btn.classList.remove('active'));
        
        if (filterType === 'all') {
            this.filterAllBtn.classList.add('active');
        } else if (filterType === 'MVP') {
            this.filterMvpBtn.classList.add('active');
        } else if (filterType === 'Mini') {
            this.filterMiniBtn.classList.add('active');
        }
        
        // 重新更新boss倒计时列表以应用筛选
        this.updateBossRemainingTime();
    }
    
    startNewTimer() {
        if (!this.currentEditingBoss) return;
        
        const minutes = parseInt(this.modalMinutes.value) || 0;
        const seconds = parseInt(this.modalSeconds.value) || 0;
        const totalSeconds = minutes * 60 + seconds;
        
        if (totalSeconds <= 0) {
            alert('请设置有效的时间！');
            return;
        }
        
        // 检查是否已经有相同boss的计时器
        for (let [id, timer] of this.activeTimers) {
            if (timer.bossId === this.currentEditingBoss.id) {
                if (confirm(`${this.currentEditingBoss.name} 已有活动计时器，是否替换？`)) {
                    this.removeTimer(id);
                    break;
                } else {
                    return;
                }
            }
        }
        
        // 增加1分钟准备时间
        const totalSecondsWithPrep = totalSeconds + 60;
        
        const timerId = ++this.timerIdCounter;
        const timer = {
            id: timerId,
            bossId: this.currentEditingBoss.id,
            bossName: this.currentEditingBoss.name,
            totalSeconds: totalSecondsWithPrep,
            remainingSeconds: totalSecondsWithPrep,
            isPaused: false,
            startTime: Date.now(),
            isInPrepTime: true, // 标记是否在准备时间
            originalSeconds: totalSeconds // 保存原始设定时间
        };
        
        this.activeTimers.set(timerId, timer);
        this.closeTimerModal();
        // 移除这行调用
        // this.renderActiveTimers();
    }
    
    pauseTimer(timerId) {
        const timer = this.activeTimers.get(timerId);
        if (timer) {
            timer.isPaused = !timer.isPaused;
            // 移除这行调用
            // this.renderActiveTimers();
        }
    }
    
    removeTimer(timerId) {
        this.activeTimers.delete(timerId);
        // 移除这行调用
        // this.renderActiveTimers();
    }
    
    startMainLoop() {
        setInterval(() => {
            this.updateTimers();
            // 移除这行调用，避免与updateCountdownList冲突
            // this.renderActiveTimers();
        }, 1000);
    }
    
    updateTimers() {
        const now = Date.now();
        const timersToRemove = [];
        
        for (let [id, timer] of this.activeTimers) {
            if (!timer.isPaused) {
                const elapsed = Math.floor((now - timer.startTime) / 1000);
                timer.remainingSeconds = Math.max(0, timer.totalSeconds - elapsed);
                
                // 检查是否从准备时间转换到实际计时
                if (timer.isInPrepTime && timer.remainingSeconds <= timer.originalSeconds) {
                    timer.isInPrepTime = false;
                }
                
                if (timer.remainingSeconds <= 0) {
                    this.onTimerComplete(timer);
                    timersToRemove.push(id);
                }
            } else {
                // 暂停时更新开始时间
                timer.startTime = now - (timer.totalSeconds - timer.remainingSeconds) * 1000;
            }
        }
        
        // 移除完成的计时器
        timersToRemove.forEach(id => this.removeTimer(id));
    }
    
    renderActiveTimers() {
        if (this.activeTimers.size === 0) {
            this.activeTimersList.innerHTML = '<div class="no-timers">暂无活动的计时器</div>';
            return;
        }
        
        // 按剩余时间排序
        const sortedTimers = Array.from(this.activeTimers.values())
            .sort((a, b) => a.remainingSeconds - b.remainingSeconds);
        
        this.activeTimersList.innerHTML = '';
        
        sortedTimers.forEach(timer => {
            const timerElement = this.createTimerElement(timer);
            this.activeTimersList.appendChild(timerElement);
        });
    }
    
    createTimerElement(timer) {
        const element = document.createElement('div');
        element.className = 'timer-item';
        
        // 根据剩余时间设置样式
        if (timer.remainingSeconds <= 10) {
            element.classList.add('danger');
        } else if (timer.remainingSeconds <= 30) {
            element.classList.add('warning');
        }
        
        // 计算实际剩余时间（不包括准备时间）
        const actualRemainingSeconds = timer.isInPrepTime ? 
            Math.max(0, timer.remainingSeconds - 60) : 
            timer.remainingSeconds;
        
        // 准备时间标记
        const prepTimeText = timer.isInPrepTime ? 
            `<div class="timer-item-prep">准备时间 ${this.formatTime(timer.remainingSeconds - timer.originalSeconds)}</div>` : 
            '';
        
        element.innerHTML = `
            <div class="timer-item-header">
                <div class="timer-item-left">
                    <div class="timer-item-name">${timer.bossName}</div>
                    <div class="timer-item-time">
                        ${this.formatTime(actualRemainingSeconds)}
                        ${prepTimeText}
                    </div>
                </div>
                <div class="timer-item-controls">
                    <button class="btn btn-small btn-secondary" onclick="bossTimer.pauseTimer(${timer.id})">
                        ${timer.isPaused ? '继续' : '暂停'}
                    </button>
                    <button class="btn btn-small btn-danger" onclick="bossTimer.removeTimer(${timer.id})">
                        删除
                    </button>
                </div>
            </div>
        `;
        
        return element;
    }
    
    formatTime(seconds) {
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const remainingSeconds = seconds % 60;
        
        // 计算目标时间（考虑时区）
        const now = new Date();
        const utc = now.getTime() + (now.getTimezoneOffset() * 60000);
        const currentTime = new Date(utc + (3600000 * this.timezone));
        const targetTime = new Date(currentTime.getTime() + (seconds * 1000));
        
        const timeString = hours > 0 
            ? `${String(hours).padStart(2, '0')}:${String(minutes).padStart(2, '0')}:${String(remainingSeconds).padStart(2, '0')}`
            : `${String(minutes).padStart(2, '0')}:${String(remainingSeconds).padStart(2, '0')}`;
            
        // 添加目标时间显示
        const targetTimeString = targetTime.toLocaleTimeString('zh-CN', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: false
        });
        
        return `${timeString} (${targetTimeString})`;
    }
    
    async onTimerComplete(timer) {
        // 播放提示音
        try {
            const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmMgBwWI1PLKdSECKnrG8N0pXRIAAAAA');
            audio.play().catch(() => {});
        } catch (error) {
            // 忽略音频播放错误
        }
        
        // 发送系统通知
        const title = 'Boss计时器';
        const body = `${timer.bossName} 的计时已结束！`;
        
        try {
            if (isElectron && ipcRenderer) {
                await ipcRenderer.invoke('send-notification', title, body);
            } else {
                // 浏览器环境使用Web Notification API
                if ('Notification' in window && Notification.permission === 'granted') {
                    new Notification(title, { body });
                }
            }
        } catch (error) {
            console.error('发送通知失败:', error);
        }
    }
    
    // ===== 顶部计时器方法 =====
    
    // 初始化顶部计时器显示
    initializeTopTimers() {
        this.updateTopTimerDisplay('1min');
        this.updateTopTimerDisplay('2min');
    }
    
    // 切换顶部计时器的开始/暂停状态
    toggleTopTimer(timerId) {
        const timer = this.topTimers[timerId];
        
        if (timer.isRunning) {
            // 暂停计时器
            this.pauseTopTimer(timerId);
        } else {
            // 开始计时器
            this.startTopTimer(timerId);
        }
    }
    
    // 开始顶部计时器
    startTopTimer(timerId) {
        const timer = this.topTimers[timerId];
        const startBtn = timerId === '1min' ? this.start1minBtn : this.start2minBtn;
        const display = timerId === '1min' ? this.timer1minDisplay : this.timer2minDisplay;
        
        if (timer.remaining <= 0) {
            this.resetTopTimer(timerId);
        }
        
        timer.isRunning = true;
        startBtn.textContent = '暂停';
        startBtn.classList.add('pause');
        display.classList.add('running');
        display.classList.remove('finished');
        
        timer.interval = setInterval(() => {
            timer.remaining--;
            this.updateTopTimerDisplay(timerId);
            
            if (timer.remaining <= 0) {
                this.onTopTimerComplete(timerId);
            }
        }, 1000);
    }
    
    // 暂停顶部计时器
    pauseTopTimer(timerId) {
        const timer = this.topTimers[timerId];
        const startBtn = timerId === '1min' ? this.start1minBtn : this.start2minBtn;
        const display = timerId === '1min' ? this.timer1minDisplay : this.timer2minDisplay;
        
        timer.isRunning = false;
        startBtn.textContent = '开始';
        startBtn.classList.remove('pause');
        display.classList.remove('running');
        
        if (timer.interval) {
            clearInterval(timer.interval);
            timer.interval = null;
        }
    }
    
    // 重置顶部计时器
    resetTopTimer(timerId) {
        const timer = this.topTimers[timerId];
        const startBtn = timerId === '1min' ? this.start1minBtn : this.start2minBtn;
        const display = timerId === '1min' ? this.timer1minDisplay : this.timer2minDisplay;
        
        // 停止计时器
        if (timer.interval) {
            clearInterval(timer.interval);
            timer.interval = null;
        }
        
        // 重置状态
        timer.isRunning = false;
        timer.remaining = timer.duration;
        
        // 更新UI
        startBtn.textContent = '开始';
        startBtn.classList.remove('pause');
        display.classList.remove('running', 'finished');
        
        this.updateTopTimerDisplay(timerId);
    }
    
    // 更新顶部计时器显示
    updateTopTimerDisplay(timerId) {
        const timer = this.topTimers[timerId];
        const display = timerId === '1min' ? this.timer1minDisplay : this.timer2minDisplay;
        
        const minutes = Math.floor(timer.remaining / 60);
        const seconds = timer.remaining % 60;
        display.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    }
    
    // 顶部计时器完成时的处理
    onTopTimerComplete(timerId) {
        const timer = this.topTimers[timerId];
        const startBtn = timerId === '1min' ? this.start1minBtn : this.start2minBtn;
        const display = timerId === '1min' ? this.timer1minDisplay : this.timer2minDisplay;
        const timerName = timerId === '1min' ? '1分钟计时器' : '2分钟计时器';
        
        // 停止计时器
        timer.isRunning = false;
        if (timer.interval) {
            clearInterval(timer.interval);
            timer.interval = null;
        }
        
        // 更新UI
        startBtn.textContent = '开始';
        startBtn.classList.remove('pause');
        display.classList.remove('running');
        display.classList.add('finished');
        display.textContent = '00:00';
        
        // 播放提示音（不用alert）
        console.log(`${timerName} 完成！`);
        try {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            oscillator.frequency.value = 800;
            oscillator.type = 'sine';
            
            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 1);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 1);
        } catch (error) {
            console.log('无法播放提示音:', error);
        }
        
        // 3秒后自动重置
        setTimeout(() => {
            this.resetTopTimer(timerId);
        }, 3000);
    }
}

// 全局变量以便HTML中的onclick能访问
let bossTimer;

// 监听Boss信息更新事件（仅在Electron环境中）
if (isElectron && ipcRenderer) {
    ipcRenderer.on('boss-info-updated', async () => {
        if (bossTimer) {
            await bossTimer.loadBossInfo();
        }
    });
}

// 当DOM加载完成后初始化应用
document.addEventListener('DOMContentLoaded', () => {
    bossTimer = new BossTimer();
}); 