const { app, BrowserWindow, ipcMain, Notification, Menu } = require('electron');
const path = require('path');
const fs = require('fs');

// 读取配置文件
let config;
try {
  const configData = fs.readFileSync(path.join(__dirname, 'config.json'), 'utf8');
  config = JSON.parse(configData);
} catch (error) {
  console.error('读取配置文件失败:', error);
  // 使用默认配置
  config = {
    Screen: { Width: 1280, Height: 800 },
    BOSS: []
  };
}

// 读取Boss信息文件
let bossInfo;
try {
  const bossInfoData = fs.readFileSync(path.join(__dirname, 'sBossInfo.json'), 'utf8');
  bossInfo = JSON.parse(bossInfoData);
} catch (error) {
  console.error('读取Boss信息文件失败:', error);
  bossInfo = [];
}

// 监视Boss信息文件变化
fs.watchFile(path.join(__dirname, 'sBossInfo.json'), (curr, prev) => {
  if (curr.mtime !== prev.mtime) {
    try {
      const bossInfoData = fs.readFileSync(path.join(__dirname, 'sBossInfo.json'), 'utf8');
      bossInfo = JSON.parse(bossInfoData);
      if (mainWindow) {
        mainWindow.webContents.send('boss-info-updated');
      }
    } catch (error) {
      console.error('读取更新后的Boss信息文件失败:', error);
    }
  }
});

let mainWindow;
let isAlwaysOnTop = false;

// 创建菜单
function createMenu() {
  const template = [
    {
      label: '文件',
      submenu: [
        {
          label: '退出',
          accelerator: 'CmdOrCtrl+Q',
          click: () => app.quit()
        }
      ]
    },
    {
      label: '窗口',
      submenu: [
        {
          label: '窗口置顶',
          type: 'checkbox',
          checked: isAlwaysOnTop,
          accelerator: 'CmdOrCtrl+T',
          click: () => {
            isAlwaysOnTop = !isAlwaysOnTop;
            if (mainWindow) {
              mainWindow.setAlwaysOnTop(isAlwaysOnTop);
              // 通知渲染进程更新UI
              mainWindow.webContents.send('always-on-top-changed', isAlwaysOnTop);
            }
          }
        },
        { type: 'separator' },
        {
          label: '重新加载',
          accelerator: 'CmdOrCtrl+R',
          click: () => {
            if (mainWindow) {
              mainWindow.reload();
            }
          }
        }
      ]
    },
    {
      label: '帮助',
      submenu: [
        {
          label: '关于',
          click: () => {
            if (mainWindow) {
              mainWindow.webContents.send('show-about-dialog');
            }
          }
        }
      ]
    }
  ];

  const menu = Menu.buildFromTemplate(template);
  Menu.setApplicationMenu(menu);
}

function createWindow() {
  // 创建浏览器窗口
  mainWindow = new BrowserWindow({
    width: config.Screen.Width,
    height: config.Screen.Height,
    webPreferences: {
      nodeIntegration: true,
      contextIsolation: false
    },
    icon: path.join(__dirname, 'assets/icon.png'), // 可选图标
    title: 'Boss时间计时器'
  });

  // 设置窗口置顶状态
  mainWindow.setAlwaysOnTop(isAlwaysOnTop);

  // 加载 index.html
  mainWindow.loadFile('index.html');

  // 打开开发者工具 (可选)
  // mainWindow.webContents.openDevTools();

  // 当窗口被关闭时发出
  mainWindow.on('closed', () => {
    mainWindow = null;
  });
}

// 当 Electron 完成初始化并准备创建浏览器窗口时调用此方法
app.whenReady().then(() => {
  createMenu();
  createWindow();
});

// 当所有窗口都被关闭时退出应用
app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createWindow();
  }
});

// 处理获取配置的IPC请求
ipcMain.handle('get-config', () => {
  return config;
});

// 处理获取Boss信息的IPC请求
ipcMain.handle('get-boss-info', () => {
  return bossInfo;
});

// 处理获取英文名称的IPC请求
ipcMain.handle('get-langs-en', () => {
  try {
    const langsEnPath = path.join(__dirname, 'langs_en.json');
    if (fs.existsSync(langsEnPath)) {
      const langsEnData = fs.readFileSync(langsEnPath, 'utf8');
      return JSON.parse(langsEnData);
    } else {
      console.warn('langs_en.json文件不存在');
      return {};
    }
  } catch (error) {
    console.error('读取langs_en.json失败:', error);
    return {};
  }
});

// 处理读取指定文件内容的IPC请求
ipcMain.handle('read-file-content', async (event, filePath) => {
  try {
    const data = fs.readFileSync(filePath, 'utf8');
    return data;
  } catch (error) {
    throw new Error(`无法读取文件: ${error.message}`);
  }
});

// 处理发送通知的IPC请求
ipcMain.handle('send-notification', (event, title, body) => {
  console.log(`收到通知请求: ${title} - ${body}`);
  
  try {
    if (Notification.isSupported()) {
      const notification = new Notification({
        title: title,
        body: body,
        icon: path.join(__dirname, 'assets/icon.png'), // 可选图标
        silent: false, // 播放系统通知声音
        urgency: 'normal' // 通知优先级
      });
      
      notification.show();
      console.log('Windows系统通知发送成功');
      return { success: true, message: '通知发送成功' };
    } else {
      console.log('系统不支持通知功能');
      return { success: false, message: '系统不支持通知功能' };
    }
  } catch (error) {
    console.error('发送系统通知时出错:', error);
    return { success: false, message: error.message };
  }
});

// 处理窗口置顶的IPC请求
ipcMain.handle('toggle-always-on-top', (event, shouldBeOnTop) => {
  try {
    if (mainWindow) {
      isAlwaysOnTop = shouldBeOnTop;
      mainWindow.setAlwaysOnTop(isAlwaysOnTop);
      
      // 同步更新菜单选项的勾选状态
      const windowMenu = Menu.getApplicationMenu().items.find(item => item.label === '窗口');
      if (windowMenu && windowMenu.submenu) {
        const alwaysOnTopItem = windowMenu.submenu.items.find(item => item.label === '窗口置顶');
        if (alwaysOnTopItem) {
          alwaysOnTopItem.checked = isAlwaysOnTop;
        }
      }
      
      return { success: true, isOnTop: mainWindow.isAlwaysOnTop() };
    }
    return { success: false, message: '主窗口未创建' };
  } catch (error) {
    console.error('设置窗口置顶时出错:', error);
    return { success: false, message: error.message };
  }
}); 