/*
 * Copyright (c) 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */


package java.util.random;

import java.math.BigInteger;
import java.util.stream.Stream;

/**
 * This is a factory class for generating multiple random number generators
 * of a specific <a href="package-summary.html#algorithms">algorithm</a>.
 * {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory} also provides
 * methods for selecting random number generator algorithms.
 *
 * A specific {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory} can be located by using the
 * {@link java.util.random.RandomGeneratorFactory#of(java.lang.String) RandomGeneratorFactory#of(String)} method, where the argument string
 * is the name of the <a href="package-summary.html#algorithms">algorithm</a>
 * required. The method
 * {@link java.util.random.RandomGeneratorFactory#all() RandomGeneratorFactory#all()} produces a non-empty {@link java.util.stream.Stream Stream} of all available
 * {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactorys} that can be searched
 * to locate a {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory} suitable to the task.
 *
 * There are three methods for constructing a RandomGenerator instance,
 * depending on the type of initial seed required.
 * {@link java.util.random.RandomGeneratorFactory#create(long) RandomGeneratorFactory#create(long)} is used for long
 * seed construction,
 * {@link java.util.random.RandomGeneratorFactory#create(byte[]) RandomGeneratorFactory#create(byte[])} is used for byte[]
 * seed construction, and
 * {@link java.util.random.RandomGeneratorFactory#create() RandomGeneratorFactory#create()} is used for random seed
 * construction. Example;
 *
 * <pre>{@code
 *    RandomGeneratorFactory<RandomGenerator> factory = RandomGeneratorFactory.of("Random");
 *
 *     for (int i = 0; i < 10; i++) {
 *         new Thread(() -> {
 *             RandomGenerator random = factory.create(100L);
 *             System.out.println(random.nextDouble());
 *         }).start();
 *     }
 * }</pre>
 *
 * RandomGeneratorFactory also provides methods describing the attributes (or properties)
 * of a generator and can be used to select random number generator
 * <a href="package-summary.html#algorithms">algorithms</a>.
 * These methods are typically used in
 * conjunction with {@link java.util.random.RandomGeneratorFactory#all() RandomGeneratorFactory#all()}. In this example, the code
 * locates the {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory} that produces
 * {@link java.util.random.RandomGenerator RandomGenerators}
 * with the highest number of state bits.
 *
 * <pre>{@code
 *     RandomGeneratorFactory<RandomGenerator> best = RandomGeneratorFactory.all()
 *         .sorted(Comparator.comparingInt(RandomGenerator::stateBits).reversed())
 *         .findFirst()
 *         .orElse(RandomGeneratorFactory.of("Random"));
 *     System.out.println(best.name() + " in " + best.group() + " was selected");
 *
 *     RandomGenerator rng = best.create();
 *     System.out.println(rng.nextLong());
 * }</pre>
 *
 * @since 17
 *
 * @see java.util.random
 *
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class RandomGeneratorFactory<T extends java.util.random.RandomGenerator> {

RandomGeneratorFactory() { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory} that can produce instances of
 * {@link java.util.random.RandomGenerator RandomGenerator} that utilize the {@code name}
 * <a href="package-summary.html#algorithms">algorithm</a>.
 *
 * @implSpec Availability is determined by RandomGeneratorFactory using the
 * service provider API to locate implementations of the RandomGenerator interface.
 *
 * @param name  Name of random number generator
 * <a href="package-summary.html#algorithms">algorithm</a>
 * @param <T> Sub-interface of {@link java.util.random.RandomGenerator RandomGenerator} to produce
 *
 * @return {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory} of {@link java.util.random.RandomGenerator RandomGenerator}
 *
 * @throws java.lang.NullPointerException if name is null
 * @throws java.lang.IllegalArgumentException if the named algorithm is not found
 */

public static <T extends java.util.random.RandomGenerator> java.util.random.RandomGeneratorFactory<T> of(java.lang.String name) { throw new RuntimeException("Stub!"); }

/**
 * Returns a {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory} meeting the minimal requirement
 * of having an algorithm whose state bits are greater than or equal 64.
 *
 * @implSpec  Since algorithms will improve over time, there is no
 * guarantee that this method will return the same algorithm over time.
 *
 * @return a {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory}
 */

public static java.util.random.RandomGeneratorFactory<java.util.random.RandomGenerator> getDefault() { throw new RuntimeException("Stub!"); }

/**
 * Returns a non-empty stream of available {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory(s)}.
 *
 * RandomGenerators that are marked as deprecated are not included in the result.
 *
 * @implSpec Availability is determined by RandomGeneratorFactory using the service provider API
 * to locate implementations of the RandomGenerator interface.
 *
 * @return a non-empty stream of all available {@link java.util.random.RandomGeneratorFactory RandomGeneratorFactory(s)}.
 */

public static java.util.stream.Stream<java.util.random.RandomGeneratorFactory<java.util.random.RandomGenerator>> all() { throw new RuntimeException("Stub!"); }

/**
 * Return the name of the <a href="package-summary.html#algorithms">algorithm</a>
 * used by the random number generator.
 *
 * @return Name of the <a href="package-summary.html#algorithms">algorithm</a>.
 */

public java.lang.String name() { throw new RuntimeException("Stub!"); }

/**
 * Return the group name of the <a href="package-summary.html#algorithms">algorithm</a>
 * used by the random number generator.
 *
 * @return Group name of the <a href="package-summary.html#algorithms">algorithm</a>.
 */

public java.lang.String group() { throw new RuntimeException("Stub!"); }

/**
 * Returns number of bits used by the <a href="package-summary.html#algorithms">algorithm</a>
 * to maintain state of seed.
 *
 * @return number of bits used by the <a href="package-summary.html#algorithms">algorithm</a>
 *         to maintain state of seed.
 */

public int stateBits() { throw new RuntimeException("Stub!"); }

/**
 * Returns the equidistribution of the <a href="package-summary.html#algorithms">algorithm</a>.
 *
 * @return the equidistribution of the <a href="package-summary.html#algorithms">algorithm</a>.
 */

public int equidistribution() { throw new RuntimeException("Stub!"); }

/**
 * Return the period of the <a href="package-summary.html#algorithms">algorithm</a>
 * used by the random number generator.
 * Returns BigInteger.ZERO if period is not determinable.
 *
 * @return BigInteger period.
 */

public java.math.BigInteger period() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator is computed using an arithmetic
 * <a href="package-summary.html#algorithms">algorithm</a>
 * and is statistically deterministic.
 *
 * @return true if random generator is statistical.
 */

public boolean isStatistical() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator is computed using external or entropic
 * sources as inputs.
 *
 * @return true if random generator is stochastic.
 */

public boolean isStochastic() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator uses a hardware device (HRNG) to produce
 * entropic input.
 *
 * @return true if random generator is generated by hardware.
 */

public boolean isHardware() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator can jump an arbitrarily specified distant
 * point in the state cycle.
 *
 * @return true if random generator is arbitrarily jumpable.
 */

public boolean isArbitrarilyJumpable() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator can jump a specified distant point in
 * the state cycle.
 *
 * @return true if random generator is jumpable.
 */

public boolean isJumpable() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator is jumpable and can leap to a very distant
 * point in the state cycle.
 *
 * @return true if random generator is leapable.
 */

public boolean isLeapable() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator can be cloned into a separate object with
 * the same properties but positioned further in the state cycle.
 *
 * @return true if random generator is splittable.
 */

public boolean isSplittable() { throw new RuntimeException("Stub!"); }

/**
 * Return true if random generator can be used to create
 * {@link java.util.stream.Stream Streams} of random numbers.
 *
 * @return true if random generator is streamable.
 */

public boolean isStreamable() { throw new RuntimeException("Stub!"); }

/**
 * Return true if the implementation of RandomGenerator (algorithm) has been
 * marked for deprecation.
 *
 * @implNote Random number generator algorithms evolve over time; new
 *           algorithms will be introduced and old algorithms will
 *           lose standing. If an older algorithm is deemed unsuitable
 *           for continued use, it will be marked as deprecated to indicate
 *           that it may be removed at some point in the future.
 *
 * @return true if the implementation of RandomGenerator (algorithm) has been
 *         marked for deprecation
 */

public boolean isDeprecated() { throw new RuntimeException("Stub!"); }

/**
 * Create an instance of {@link java.util.random.RandomGenerator RandomGenerator} based on
 * <a href="package-summary.html#algorithms">algorithm</a> chosen.
 *
 * @return new in instance of {@link java.util.random.RandomGenerator RandomGenerator}.
 *
 */

public T create() { throw new RuntimeException("Stub!"); }

/**
 * Create an instance of {@link java.util.random.RandomGenerator RandomGenerator} based on
 * <a href="package-summary.html#algorithms">algorithm</a> chosen
 * providing a starting long seed. If long seed is not supported by an
 * algorithm then the no argument form of create is used.
 *
 * @param seed long random seed value.
 *
 * @return new in instance of {@link java.util.random.RandomGenerator RandomGenerator}.
 */

public T create(long seed) { throw new RuntimeException("Stub!"); }

/**
 * Create an instance of {@link java.util.random.RandomGenerator RandomGenerator} based on
 * <a href="package-summary.html#algorithms">algorithm</a> chosen
 * providing a starting byte[] seed. If byte[] seed is not supported by an
 * <a href="package-summary.html#algorithms">algorithm</a> then the no
 * argument form of create is used.
 *
 * @param seed byte array random seed value.
 *
 * @return new in instance of {@link java.util.random.RandomGenerator RandomGenerator}.
 *
 * @throws java.lang.NullPointerException if seed is null.
 */

public T create(byte[] seed) { throw new RuntimeException("Stub!"); }
}

