/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net.wifi.rtt;

import android.net.wifi.ScanResult;
import android.net.wifi.aware.PeerHandle;
import java.util.List;

/**
 * Ranging result for a request started by
 * {@link android.net.wifi.rtt.WifiRttManager#startRanging(android.net.wifi.rtt.RangingRequest,java.util.concurrent.Executor,android.net.wifi.rtt.RangingResultCallback) WifiRttManager#startRanging(RangingRequest, java.util.concurrent.Executor, RangingResultCallback)}.
 * Results are returned in {@link android.net.wifi.rtt.RangingResultCallback#onRangingResults(java.util.List) RangingResultCallback#onRangingResults(List)}.
 * <p>
 * A ranging result is the distance measurement result for a single device specified in the
 * {@link android.net.wifi.rtt.RangingRequest RangingRequest}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class RangingResult implements android.os.Parcelable {

RangingResult() { throw new RuntimeException("Stub!"); }

/**
 * @return The status of ranging measurement: {@link #STATUS_SUCCESS} in case of success, and
 * {@link #STATUS_FAIL} in case of failure.

 * Value is {@link android.net.wifi.rtt.RangingResult#STATUS_SUCCESS}, {@link android.net.wifi.rtt.RangingResult#STATUS_FAIL}, or {@link android.net.wifi.rtt.RangingResult#STATUS_RESPONDER_DOES_NOT_SUPPORT_IEEE80211MC}
 */

public int getStatus() { throw new RuntimeException("Stub!"); }

/**
 * @return The MAC address of the device whose range measurement was requested. Will correspond
 * to the MAC address of the device in the {@link android.net.wifi.rtt.RangingRequest RangingRequest}.
 * <p>
 * Will return a {@code null} for results corresponding to requests issued using a {@code
 * PeerHandle}, i.e. using the {@link android.net.wifi.rtt.RangingRequest.Builder#addWifiAwarePeer(android.net.wifi.aware.PeerHandle) RangingRequest.Builder#addWifiAwarePeer(PeerHandle)} API.
 */

@android.annotation.Nullable
public android.net.MacAddress getMacAddress() { throw new RuntimeException("Stub!"); }

/**
 * @return The PeerHandle of the device whose reange measurement was requested. Will correspond
 * to the PeerHandle of the devices requested using
 * {@link android.net.wifi.rtt.RangingRequest.Builder#addWifiAwarePeer(android.net.wifi.aware.PeerHandle) RangingRequest.Builder#addWifiAwarePeer(PeerHandle)}.
 * <p>
 * Will return a {@code null} for results corresponding to requests issued using a MAC address.
 */

@android.annotation.Nullable
public android.net.wifi.aware.PeerHandle getPeerHandle() { throw new RuntimeException("Stub!"); }

/**
 * @return The distance (in mm) to the device specified by {@link #getMacAddress()} or
 * {@link #getPeerHandle()}.
 * <p>
 * Note: the measured distance may be negative for very close devices.
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

public int getDistanceMm() { throw new RuntimeException("Stub!"); }

/**
 * @return The standard deviation of the measured distance (in mm) to the device specified by
 * {@link #getMacAddress()} or {@link #getPeerHandle()}. The standard deviation is calculated
 * over the measurements executed in a single RTT burst. The number of measurements is returned
 * by {@link #getNumSuccessfulMeasurements()} - 0 successful measurements indicate that the
 * standard deviation is not valid (a valid standard deviation requires at least 2 data points).
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

public int getDistanceStdDevMm() { throw new RuntimeException("Stub!"); }

/**
 * @return The average RSSI, in units of dBm, observed during the RTT measurement.
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

public int getRssi() { throw new RuntimeException("Stub!"); }

/**
 * @return The number of attempted measurements used in the RTT exchange resulting in this set
 * of results. The number of successful measurements is returned by
 * {@link #getNumSuccessfulMeasurements()} which at most, if there are no errors, will be 1
 * less than the number of attempted measurements.
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception. If the value is 0, it should be interpreted as no information available, which may
 * occur for one-sided RTT measurements. Instead {@link android.net.wifi.rtt.RangingRequest#getRttBurstSize() RangingRequest#getRttBurstSize()}
 * should be used instead.
 */

public int getNumAttemptedMeasurements() { throw new RuntimeException("Stub!"); }

/**
 * @return The number of successful measurements used to calculate the distance and standard
 * deviation. If the number of successful measurements if 1 then then standard deviation,
 * returned by {@link #getDistanceStdDevMm()}, is not valid (a 0 is returned for the standard
 * deviation).
 * <p>
 * The total number of measurement attempts is returned by
 * {@link #getNumAttemptedMeasurements()}. The number of successful measurements will be at
 * most 1 less then the number of attempted measurements.
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

public int getNumSuccessfulMeasurements() { throw new RuntimeException("Stub!"); }

/**
 * @return The unverified responder location represented as {@link android.net.wifi.rtt.ResponderLocation ResponderLocation} which
 * captures location information the responder is programmed to broadcast. The responder
 * location is referred to as unverified, because we are relying on the device/site
 * administrator to correctly configure its location data.
 * <p>
 * Will return a {@code null} when the location information cannot be parsed.
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

@android.annotation.Nullable
public android.net.wifi.rtt.ResponderLocation getUnverifiedResponderLocation() { throw new RuntimeException("Stub!"); }

/**
 * @return The Location Configuration Information (LCI) as self-reported by the peer. The format
 * is specified in the IEEE 802.11-2016 specifications, section 9.4.2.22.10.
 * <p>
 * Note: the information is NOT validated - use with caution. Consider validating it with
 * other sources of information before using it.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public byte[] getLci() { throw new RuntimeException("Stub!"); }

/**
 * @return The Location Civic report (LCR) as self-reported by the peer. The format
 * is specified in the IEEE 802.11-2016 specifications, section 9.4.2.22.13.
 * <p>
 * Note: the information is NOT validated - use with caution. Consider validating it with
 * other sources of information before using it.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public byte[] getLcr() { throw new RuntimeException("Stub!"); }

/**
 * @return The timestamp at which the ranging operation was performed. The timestamp is in
 * milliseconds since boot, including time spent in sleep, corresponding to values provided by
 * {@link android.os.SystemClock#elapsedRealtime()}.
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

public long getRangingTimestampMillis() { throw new RuntimeException("Stub!"); }

/**
 * @return The result is true if the IEEE 802.11mc protocol was used. If the result is false,
 * and {@link #is80211azNtbMeasurement()} is also false a one-side RTT result is provided
 * which does not subtract the turnaround time at the responder.
 * <p>
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

public boolean is80211mcMeasurement() { throw new RuntimeException("Stub!"); }

/**
 * @return The result is true if the IEEE 802.11az non-trigger based protocol was used. If the
 * result is false, and {@link #is80211mcMeasurement()} is also false a one-side RTT result
 * is provided which does not subtract the turnaround time at the responder.
 * <p>.
 * Only valid if {@link #getStatus()} returns {@link #STATUS_SUCCESS}, otherwise will throw an
 * exception.
 */

public boolean is80211azNtbMeasurement() { throw new RuntimeException("Stub!"); }

/**
 * Gets minimum time between measurements in microseconds for IEEE 802.11az non-trigger based
 * ranging.
 *
 * The next 11az ranging measurement request must be invoked after the minimum time from the
 * last measurement time {@link #getRangingTimestampMillis()} for the peer. Otherwise, cached
 * ranging result will be returned for the peer.
 */

public long getMinTimeBetweenNtbMeasurementsMicros() { throw new RuntimeException("Stub!"); }

/**
 * Gets maximum time between measurements in microseconds for IEEE 802.11az non-trigger based
 * ranging.
 *
 * The next 11az ranging request needs to be invoked before the maximum time from the last
 * measurement time {@link #getRangingTimestampMillis()}. Otherwise, the non-trigger based
 * ranging session will be terminated and a new ranging negotiation will happen with
 * the responding station.
 */

public long getMaxTimeBetweenNtbMeasurementsMicros() { throw new RuntimeException("Stub!"); }

/**
 * Gets LTF repetitions that the responder station (RSTA) used in the preamble of the
 * responder to initiator (I2R) null data PPDU (NDP) for this result.
 *
 * LTF repetitions is the multiple transmissions of HE-LTF symbols in an HE ranging NDP. An
 * HE-LTF repetition value of 1 indicates no repetitions.
 *
 * @return LTF repetitions count
 */

public int get80211azResponderTxLtfRepetitionsCount() { throw new RuntimeException("Stub!"); }

/**
 * Gets LTF repetitions that the initiator station (ISTA) used in the preamble of the
 * initiator to responder (I2R) null data PPDU (NDP) for this result.
 *
 * LTF repetitions is the multiple transmissions of HE-LTF symbols in an HE ranging NDP. An
 * HE-LTF repetition value of 1 indicates no repetitions.
 *
 * @return LTF repetitions count
 */

public int get80211azInitiatorTxLtfRepetitionsCount() { throw new RuntimeException("Stub!"); }

/**
 * Gets number of transmit spatial streams that the initiator station (ISTA) used for the
 * initiator to responder (I2R) null data PPDU (NDP) for this result.
 *
 * @return Number of spatial streams
 */

public int get80211azNumberOfTxSpatialStreams() { throw new RuntimeException("Stub!"); }

/**
 * Gets number of receive spatial streams that the initiator station (ISTA) used for the
 * initiator to responder (I2R) null data PPDU (NDP) for this result.
 *
 * @return Number of spatial streams
 */

public int get80211azNumberOfRxSpatialStreams() { throw new RuntimeException("Stub!"); }

/**
 * The center frequency of the primary 20 MHz frequency (in MHz) of the channel over
 * which the measurement frames are sent.
 * @return center frequency in Mhz of the channel if available, otherwise {@link #UNSPECIFIED}
 * is returned.
 * <p>
 * @throws java.lang.IllegalStateException if {@link #getStatus()} does not return
 * {@link #STATUS_SUCCESS}.
 */

public int getMeasurementChannelFrequencyMHz() { throw new RuntimeException("Stub!"); }

/**
 * The bandwidth used to transmit the RTT measurement frame.
 * @return one of {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_20MHZ ScanResult#CHANNEL_WIDTH_20MHZ},
 * {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_40MHZ ScanResult#CHANNEL_WIDTH_40MHZ},
 * {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_80MHZ ScanResult#CHANNEL_WIDTH_80MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_160MHZ ScanResult#CHANNEL_WIDTH_160MHZ},
 * {@link android.net.wifi.ScanResult #CHANNEL_WIDTH_80MHZ_PLUS_MHZ} or {@link android.net.wifi.ScanResult #CHANNEL_WIDTH_320MHZ}
 * if available, otherwise {@link #UNSPECIFIED} is returned.
 * <p>
 * Value is {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_20MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_40MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_80MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_160MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_80MHZ_PLUS_MHZ}, or {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_320MHZ}
 * @throws java.lang.IllegalStateException if {@link #getStatus()} does not return
 * {@link #STATUS_SUCCESS}.
 */

public int getMeasurementBandwidth() { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

public void writeToParcel(android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

/** @hide */

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object o) { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.wifi.rtt.RangingResult> CREATOR;
static { CREATOR = null; }

/**
 * Individual range request status, {@link #getStatus()}. Indicates ranging operation failed
 * and the distance value is invalid.
 */

public static final int STATUS_FAIL = 1; // 0x1

/**
 * Individual range request status, {@link #getStatus()}. Indicates that the ranging operation
 * failed because the specified peer does not support IEEE 802.11mc RTT operations. Support by
 * an Access Point can be confirmed using
 * {@link android.net.wifi.ScanResult#is80211mcResponder()}.
 * <p>
 * On such a failure, the individual result fields of {@link android.net.wifi.rtt.RangingResult RangingResult} such as
 * {@link android.net.wifi.rtt.RangingResult#getDistanceMm() RangingResult#getDistanceMm()} are invalid.
 */

public static final int STATUS_RESPONDER_DOES_NOT_SUPPORT_IEEE80211MC = 2; // 0x2

/**
 * Individual range request status, {@link #getStatus()}. Indicates ranging operation was
 * successful and distance value is valid.
 */

public static final int STATUS_SUCCESS = 0; // 0x0

/**
 * The unspecified value.
 */

public static final int UNSPECIFIED = -1; // 0xffffffff
/**
 * Builder class used to construct {@link android.net.wifi.rtt.RangingResult RangingResult} objects.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Sets the Range result status.
 *
 * @param status Ranging result status, if not set defaults to
 *               {@link #STATUS_FAIL}.
 * Value is {@link android.net.wifi.rtt.RangingResult#STATUS_SUCCESS}, {@link android.net.wifi.rtt.RangingResult#STATUS_FAIL}, or {@link android.net.wifi.rtt.RangingResult#STATUS_RESPONDER_DOES_NOT_SUPPORT_IEEE80211MC}
 * @return The builder to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setStatus(int status) { throw new RuntimeException("Stub!"); }

/**
 * Sets the MAC address of the ranging result.
 *
 * @param macAddress Mac address, if not defaults to null.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setMacAddress(@android.annotation.Nullable android.net.MacAddress macAddress) { throw new RuntimeException("Stub!"); }

/**
 * Sets the peer handle. Applicable only for NAN Ranging.
 *
 * @param peerHandle Opaque object used to represent a Wi-Fi Aware peer. If not set,
 *                   defaults to null.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setPeerHandle(@android.annotation.Nullable android.net.wifi.aware.PeerHandle peerHandle) { throw new RuntimeException("Stub!"); }

/**
 * Sets the distance in millimeter.
 *
 * @param distanceMm distance. If not set, defaults to 0.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setDistanceMm(int distanceMm) { throw new RuntimeException("Stub!"); }

/**
 * Sets the standard deviation of the distance in millimeter.
 *
 * @param distanceStdDevMm Standard deviation of the distance measurement. If not set
 *                         defaults to 0.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setDistanceStdDevMm(int distanceStdDevMm) { throw new RuntimeException("Stub!"); }

/**
 * Sets the average RSSI.
 *
 * @param rssi Average RSSI. If not set, defaults to -127.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setRssi(int rssi) { throw new RuntimeException("Stub!"); }

/**
 * Sets the total number of RTT measurements attempted.
 *
 * @param numAttemptedMeasurements Number of attempted measurements. If not set, default
 *                                 to 0.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setNumAttemptedMeasurements(int numAttemptedMeasurements) { throw new RuntimeException("Stub!"); }

/**
 * Sets the total number of successful RTT measurements.
 *
 * @param numSuccessfulMeasurements Number of successful measurements. If not set, default
 *                                 to 0.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setNumSuccessfulMeasurements(int numSuccessfulMeasurements) { throw new RuntimeException("Stub!"); }

/**
 * Sets the Location Configuration Information (LCI).
 *
 * LCI provides data about the access point's (AP) physical location, such as its
 * latitude, longitude, and altitude. The format is specified in the IEEE 802.11-2016
 * specifications, section 9.4.2.22.10.
 *
 * @param lci Location configuration information. If not set, defaults to null.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setLci(@android.annotation.Nullable byte[] lci) { throw new RuntimeException("Stub!"); }

/**
 * Sets the Location Civic Report (LCR).
 *
 * LCR provides additional details about the AP's location in a human-readable format,
 * such as the street address, building name, or floor number. This can be helpful for
 * users to understand the context of their location within a building or complex.
 *
 * The format is
 * specified in the IEEE 802.11-2016 specifications, section 9.4.2.22.13.
 *
 * @param lcr Location civic report. If not set, defaults to null.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setLcr(@android.annotation.Nullable byte[] lcr) { throw new RuntimeException("Stub!"); }

/**
 * Sets Responder Location.
 *
 * ResponderLocation is both a Location Configuration Information (LCI) decoder and a
 * Location Civic Report (LCR) decoder for information received from a Wi-Fi Access Point
 * (AP) during Wi-Fi RTT ranging process.
 *
 * @param responderLocation Responder location. If not set, defaults to null.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setUnverifiedResponderLocation(@android.annotation.Nullable android.net.wifi.rtt.ResponderLocation responderLocation) { throw new RuntimeException("Stub!"); }

/**
 * Sets the time stamp at which the ranging operation was performed.
 *
 * The timestamp is in milliseconds since boot, including time spent in sleep,
 * corresponding to values provided by {@link android.os.SystemClock#elapsedRealtime()}.
 *
 * @param timestamp time stamp in milliseconds. If not set, default to 0.
 * Value is a non-negative timestamp in the
 * {@link android.os.SystemClock#elapsedRealtime()} time base.
 * @return The builder to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setRangingTimestampMillis(long timestamp) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether the ranging measurement was performed using IEEE 802.11mc ranging method.
 * If {@link #set80211mcMeasurement(boolean)} is set as false and
 * {@link #set80211azNtbMeasurement(boolean)} is also set as false, ranging measurement was
 * performed using one-side RTT. If not set, default to false.
 *
 * @param is80211mcMeasurement true for IEEE 802.11mc measure, otherwise false.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder set80211mcMeasurement(boolean is80211mcMeasurement) { throw new RuntimeException("Stub!"); }

/**
 * Sets the center frequency of the primary 20 MHz frequency (in MHz) of the channel over
 * which the measurement frames are sent. If not set, default to
 * {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}
 *
 * @param frequencyMHz Frequency.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setMeasurementChannelFrequencyMHz(int frequencyMHz) { throw new RuntimeException("Stub!"); }

/**
 * Sets the bandwidth used to transmit the RTT measurement frame. If not set, default to
 * {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}.
 *
 * @param measurementBandwidth Measurement bandwidth.
 * Value is {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_20MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_40MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_80MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_160MHZ}, {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_80MHZ_PLUS_MHZ}, or {@link android.net.wifi.ScanResult#CHANNEL_WIDTH_320MHZ}
 * @return The builder to facilitate chaining.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setMeasurementBandwidth(int measurementBandwidth) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether the ranging measurement was performed using IEEE 802.11az non-trigger
 * ranging method. If {@link #set80211azNtbMeasurement(boolean)} is set as false and
 * {@link #set80211mcMeasurement(boolean)} is also set as false, ranging measurement was
 * performed using one-side RTT. If not set defaults to false.
 *
 * @param is80211azNtbMeasurement true for IEEE 802.11az non-trigger based measurement,
 *                                otherwise false.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder set80211azNtbMeasurement(boolean is80211azNtbMeasurement) { throw new RuntimeException("Stub!"); }

/**
 * Sets minimum time between measurements in microseconds for IEEE 802.11az non-trigger
 * based ranging.  If not set, defaults to {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}.
 *
 * @param ntbMinMeasurementTime non-trigger based ranging minimum measurement time.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setMinTimeBetweenNtbMeasurementsMicros(long ntbMinMeasurementTime) { throw new RuntimeException("Stub!"); }

/**
 * Sets maximum time between measurements in microseconds for IEEE 802.11az non-trigger
 * based ranging. If not set, defaults to {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}.
 *
 * @param ntbMaxMeasurementTime non-trigger based ranging maximum measurement time.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder setMaxTimeBetweenNtbMeasurementsMicros(long ntbMaxMeasurementTime) { throw new RuntimeException("Stub!"); }

/**
 * Sets LTF repetitions that the initiator station used in the preamble. If not set,
 * defaults to {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}.
 *
 * @param i2rTxLtfRepetitions LFT repetition count.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder set80211azInitiatorTxLtfRepetitionsCount(int i2rTxLtfRepetitions) { throw new RuntimeException("Stub!"); }

/**
 * Sets LTF repetitions that the responder station used in the preamble. If not set,
 * defaults to {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}.
 *
 * @param r2iTxLtfRepetitions LFT repetition count.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder set80211azResponderTxLtfRepetitionsCount(int r2iTxLtfRepetitions) { throw new RuntimeException("Stub!"); }

/**
 * Sets number of transmit spatial streams that the initiator station used for the
 * ranging result. If not set, defaults to {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}.
 *
 * @param numTxSpatialStreams Number of transmit spatial streams.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder set80211azNumberOfTxSpatialStreams(int numTxSpatialStreams) { throw new RuntimeException("Stub!"); }

/**
 * Sets number of receive spatial streams that the initiator station used for the ranging
 * result. If not set, defaults to {@link android.net.wifi.rtt.RangingResult#UNSPECIFIED RangingResult#UNSPECIFIED}.
 *
 * @param numRxSpatialStreams Number of receive spatial streams.
 * @return The builder to facilitate chaining.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult.Builder set80211azNumberOfRxSpatialStreams(int numRxSpatialStreams) { throw new RuntimeException("Stub!"); }

/**
 * Build {@link android.net.wifi.rtt.RangingResult RangingResult}
 * @return an instance of {@link android.net.wifi.rtt.RangingResult RangingResult}

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.rtt.RangingResult build() { throw new RuntimeException("Stub!"); }
}

}

