/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.location;

import android.Manifest;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.location.provider.ProviderProperties;
import android.os.Bundle;
import android.os.CancellationSignal;
import android.os.Handler;
import android.os.Looper;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.function.Consumer;

/**
 * This class provides access to the system location services. These services allow applications to
 * obtain periodic updates of the device's geographical location, or to be notified when the device
 * enters the proximity of a given geographical location.
 *
 * <p class="note">Unless otherwise noted, all Location API methods require the
 * {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} permissions. If your application only
 * has the coarse permission then providers will still return location results, but the exact
 * location will be obfuscated to a coarse level of accuracy.

 * <br>
 * Requires the {@link android.content.pm.PackageManager#FEATURE_LOCATION PackageManager#FEATURE_LOCATION} feature which can be detected using {@link android.content.pm.PackageManager#hasSystemFeature(String) PackageManager.hasSystemFeature(String)}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class LocationManager {

LocationManager() { throw new RuntimeException("Stub!"); }

/**
 * Returns the current enabled/disabled state of location. To listen for changes, see
 * {@link #MODE_CHANGED_ACTION}.
 *
 * @return true if location is enabled and false if location is disabled.
 */

public boolean isLocationEnabled() { throw new RuntimeException("Stub!"); }

/**
 * Returns the current enabled/disabled status of the given provider. To listen for changes, see
 * {@link #PROVIDERS_CHANGED_ACTION}.
 *
 * Before API version {@link android.os.Build.VERSION_CODES#LOLLIPOP}, this method would throw
 * {@link java.lang.SecurityException SecurityException} if the location permissions were not sufficient to use the
 * specified provider.
 *
 * @param provider a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @return true if the provider exists and is enabled
 *
 * @throws java.lang.IllegalArgumentException if provider is null
 */

public boolean isProviderEnabled(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Gets the last known location from the given provider, or null if there is no last known
 * location. The returned location may be quite old in some circumstances, so the age of the
 * location should always be checked.
 *
 * <p>This will never activate sensors to compute a new location, and will only ever return a
 * cached location.
 *
 * <p>See also {@link #getCurrentLocation(java.lang.String,android.os.CancellationSignal,java.util.concurrent.Executor,java.util.function.Consumer)} which
 * will always attempt to return a current location, but will potentially use additional power
 * in the course of the attempt as compared to this method.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider a provider listed by {@link #getAllProviders()}
 *
 * This value cannot be {@code null}.
 * @return the last known location for the given provider, or null if not available
 *
 * @throws java.lang.SecurityException if no suitable permission is present
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 */

@android.annotation.Nullable
public android.location.Location getLastKnownLocation(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Asynchronously returns a single current location fix from the given provider.
 *
 * <p>See
 * {@link #getCurrentLocation(java.lang.String,android.location.LocationRequest,android.os.CancellationSignal,java.util.concurrent.Executor,java.util.function.Consumer)}
 * for more information.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider           a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param cancellationSignal an optional signal that allows for cancelling this call
 * This value may be {@code null}.
 * @param executor           the callback will take place on this {@link java.util.concurrent.Executor Executor}
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param consumer           the callback invoked with either a {@link android.location.Location Location} or null
 *
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if consumer is null
 * @throws java.lang.SecurityException        if no suitable permission is present
 */

public void getCurrentLocation(@android.annotation.NonNull java.lang.String provider, @android.annotation.Nullable android.os.CancellationSignal cancellationSignal, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.location.Location> consumer) { throw new RuntimeException("Stub!"); }

/**
 * Asynchronously returns a single current location fix from the given provider based on the
 * given {@link android.location.LocationRequest LocationRequest}. This may activate sensors in order to compute a new location,
 * unlike {@link #getLastKnownLocation(java.lang.String)}, which will only return a cached fix if
 * available. The given callback will be invoked once and only once, either with a valid
 * location or with a null location if the provider was unable to generate a valid location.
 *
 * <p>A client may supply an optional {@link android.os.CancellationSignal CancellationSignal}. If this is used to cancel the
 * operation, no callback should be expected after the cancellation.
 *
 * <p>This method may return locations from the very recent past (on the order of several
 * seconds), but will never return older locations (for example, several minutes old or older).
 * Clients may rely upon the guarantee that if this method returns a location, it will represent
 * the best estimation of the location of the device in the present moment.
 *
 * <p>Clients calling this method from the background may notice that the method fails to
 * determine a valid location fix more often than while in the foreground. Background
 * applications may be throttled in their location accesses to some degree.
 *
 * The given location request may be used to provide hints on how a fresh location is computed
 * if necessary. In particular {@link android.location.LocationRequest#getDurationMillis() LocationRequest#getDurationMillis()} can be used to
 * provide maximum duration allowed before failing. The system will always cap the maximum
 * amount of time a request for current location may run to some reasonable value (less than a
 * minute for example) before the request is failed.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider           a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param locationRequest    the location request containing location parameters
 * This value cannot be {@code null}.
 * @param cancellationSignal an optional signal that allows for cancelling this call
 * This value may be {@code null}.
 * @param executor           the callback will take place on this {@link java.util.concurrent.Executor Executor}
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param consumer           the callback invoked with either a {@link android.location.Location Location} or null
 *
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if consumer is null
 * @throws java.lang.SecurityException        if no suitable permission is present
 */

public void getCurrentLocation(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.LocationRequest locationRequest, @android.annotation.Nullable android.os.CancellationSignal cancellationSignal, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.location.Location> consumer) { throw new RuntimeException("Stub!"); }

/**
 * Register for a single location update using the named provider and a callback.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,long,float,android.location.LocationListener,android.os.Looper)} for
 * more detail on how to use this method.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param listener the listener to receive location updates
 * This value cannot be {@code null}.
 * @param looper   the looper handling listener callbacks, or null to use the looper of the
 *                 calling thread
 *
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException        if no suitable permission is present
 * @deprecated Use {@link #getCurrentLocation(java.lang.String,android.os.CancellationSignal,java.util.concurrent.Executor,java.util.function.Consumer)}
 * instead as it does not carry a risk of extreme battery drain.
 */

@Deprecated
public void requestSingleUpdate(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.LocationListener listener, @android.annotation.Nullable android.os.Looper looper) { throw new RuntimeException("Stub!"); }

/**
 * Register for a single location update using a Criteria and a callback.
 *
 * <p>Note: Since Android KitKat, Criteria requests will always result in using the
 * {@link #FUSED_PROVIDER}.
 *
 * <p>See {@link #requestLocationUpdates(long,float,android.location.Criteria,android.app.PendingIntent)} for more detail
 * on how to use this method.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param criteria contains parameters to choose the appropriate provider for location updates
 * This value cannot be {@code null}.
 * @param listener the listener to receive location updates
 * This value cannot be {@code null}.
 * @param looper   the looper handling listener callbacks, or null to use the looper of the
 *                 calling thread
 *
 * @throws java.lang.IllegalArgumentException if criteria is null
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException        if no suitable permission is present
 * @deprecated Use {@link #getCurrentLocation(java.lang.String,android.os.CancellationSignal,java.util.concurrent.Executor,java.util.function.Consumer)}
 * instead as it does not carry a risk of extreme battery drain.
 */

@Deprecated
public void requestSingleUpdate(@android.annotation.NonNull android.location.Criteria criteria, @android.annotation.NonNull android.location.LocationListener listener, @android.annotation.Nullable android.os.Looper looper) { throw new RuntimeException("Stub!"); }

/**
 * Register for a single location update using a named provider and pending intent.
 *
 * <p>See {@link #requestLocationUpdates(long,float,android.location.Criteria,android.app.PendingIntent)} for more detail
 * on how to use this method.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider      a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param pendingIntent the pending intent to send location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if intent is null
 * @throws java.lang.SecurityException        if no suitable permission is present
 * @deprecated Use {@link #getCurrentLocation(java.lang.String,android.os.CancellationSignal,java.util.concurrent.Executor,java.util.function.Consumer)}
 * instead as it does not carry a risk of extreme battery drain.
 */

@Deprecated
public void requestSingleUpdate(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.app.PendingIntent pendingIntent) { throw new RuntimeException("Stub!"); }

/**
 * Register for a single location update using a Criteria and pending intent.
 *
 * <p>Note: Since Android KitKat, Criteria requests will always result in using the
 * {@link #FUSED_PROVIDER}.
 *
 * <p>See {@link #requestLocationUpdates(long,float,android.location.Criteria,android.app.PendingIntent)} for more detail
 * on how to use this method.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param criteria      contains parameters to choose the appropriate provider for location
 *                      updates
 * This value cannot be {@code null}.
 * @param pendingIntent the pending intent to send location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if intent is null
 * @throws java.lang.SecurityException        if no suitable permission is present
 * @deprecated Use {@link #getCurrentLocation(java.lang.String,android.os.CancellationSignal,java.util.concurrent.Executor,java.util.function.Consumer)}
 * instead as it does not carry a risk of extreme battery drain.
 */

@Deprecated
public void requestSingleUpdate(@android.annotation.NonNull android.location.Criteria criteria, @android.annotation.NonNull android.app.PendingIntent pendingIntent) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates from the given provider with the given arguments, and a
 * callback on the {@link android.os.Looper Looper} of the calling thread.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,android.location.LocationRequest,java.util.concurrent.Executor,android.location.LocationListener)}
 * for more detail on how this method works.
 *
 * <p class="note"> Prior to Jellybean, the minTime parameter was only a hint, and some location
 * provider implementations ignored it. For Jellybean and onwards however, it is mandatory for
 * Android compatible devices to observe both the minTime and minDistance parameters.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider     a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param minTimeMs    minimum time interval between location updates in milliseconds
 * @param minDistanceM minimum distance between location updates in meters
 * @param listener     the listener to receive location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.RuntimeException if the calling thread has no Looper
 * @throws java.lang.SecurityException if no suitable permission is present
 */

public void requestLocationUpdates(@android.annotation.NonNull java.lang.String provider, long minTimeMs, float minDistanceM, @android.annotation.NonNull android.location.LocationListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates from the given provider with the given arguments, and a
 * callback on the specified {@link android.os.Looper Looper}.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,android.location.LocationRequest,java.util.concurrent.Executor,android.location.LocationListener)}
 * for more detail on how this method works.
 *
 * <p class="note">Prior to Jellybean, the minTime parameter was only a hint, and some location
 * provider implementations ignored it. For Jellybean and onwards however, it is mandatory for
 * Android compatible devices to observe both the minTime and minDistance parameters.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider     a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param minTimeMs    minimum time interval between location updates in milliseconds
 * @param minDistanceM minimum distance between location updates in meters
 * @param listener     the listener to receive location updates
 * This value cannot be {@code null}.
 * @param looper       the looper handling listener callbacks, or null to use the looper of the
 *                     calling thread
 *
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException if no suitable permission is present
 */

public void requestLocationUpdates(@android.annotation.NonNull java.lang.String provider, long minTimeMs, float minDistanceM, @android.annotation.NonNull android.location.LocationListener listener, @android.annotation.Nullable android.os.Looper looper) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates using the named provider, and a callback on
 * the specified {@link java.util.concurrent.Executor Executor}.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,android.location.LocationRequest,java.util.concurrent.Executor,android.location.LocationListener)}
 * for more detail on how this method works.
 *
 * <p class="note">Prior to Jellybean, the minTime parameter was only a hint, and some location
 * provider implementations ignored it. For Jellybean and onwards however, it is mandatory for
 * Android compatible devices to observe both the minTime and minDistance parameters.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider     a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param minTimeMs    minimum time interval between location updates in milliseconds
 * @param minDistanceM minimum distance between location updates in meters
 * @param executor     the executor handling listener callbacks
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener     the listener to receive location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException if no suitable permission is present
 */

public void requestLocationUpdates(@android.annotation.NonNull java.lang.String provider, long minTimeMs, float minDistanceM, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.LocationListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates using a provider selected through the given Criteria, and a
 * callback on the specified {@link android.os.Looper Looper}.
 *
 * <p>Note: Since Android KitKat, Criteria requests will always result in using the
 * {@link #FUSED_PROVIDER}.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,android.location.LocationRequest,java.util.concurrent.Executor,android.location.LocationListener)}
 * for more detail on how this method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param minTimeMs minimum time interval between location updates in milliseconds
 * @param minDistanceM minimum distance between location updates in meters
 * @param criteria contains parameters to choose the appropriate provider for location updates
 * This value cannot be {@code null}.
 * @param listener the listener to receive location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if criteria is null
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException if no suitable permission is present
 *
 * @deprecated Use
 * {@link #requestLocationUpdates(java.lang.String,long,float,android.location.LocationListener,android.os.Looper)} instead to
 * explicitly select a provider.

 * @param looper This value may be {@code null}.
 */

@Deprecated
public void requestLocationUpdates(long minTimeMs, float minDistanceM, @android.annotation.NonNull android.location.Criteria criteria, @android.annotation.NonNull android.location.LocationListener listener, @android.annotation.Nullable android.os.Looper looper) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates using a provider selected through the given Criteria, and a
 * callback on the specified {@link java.util.concurrent.Executor Executor}.
 *
 * <p>Note: Since Android KitKat, Criteria requests will always result in using the
 * {@link #FUSED_PROVIDER}.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,android.location.LocationRequest,java.util.concurrent.Executor,android.location.LocationListener)}
 * for more detail on how this method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param minTimeMs minimum time interval between location updates in milliseconds
 * @param minDistanceM minimum distance between location updates in meters
 * @param criteria contains parameters to choose the appropriate provider for location updates
 * This value cannot be {@code null}.
 * @param executor the executor handling listener callbacks
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener the listener to receive location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if criteria is null
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException        if no suitable permission is present
 *
 * @deprecated Use
 * {@link #requestLocationUpdates(java.lang.String,long,float,java.util.concurrent.Executor,android.location.LocationListener)} instead to
 * explicitly select a provider.
 */

@Deprecated
public void requestLocationUpdates(long minTimeMs, float minDistanceM, @android.annotation.NonNull android.location.Criteria criteria, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.LocationListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates using the named provider, and callbacks delivered via the
 * provided {@link android.app.PendingIntent PendingIntent}.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,android.location.LocationRequest,android.app.PendingIntent)} for more
 * detail on how this method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider      a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param minTimeMs     minimum time interval between location updates in milliseconds
 * @param minDistanceM  minimum distance between location updates in meters
 * @param pendingIntent the pending intent to send location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if pendingIntent is null
 * @throws java.lang.SecurityException if no suitable permission is present
 */

public void requestLocationUpdates(@android.annotation.NonNull java.lang.String provider, long minTimeMs, float minDistanceM, @android.annotation.NonNull android.app.PendingIntent pendingIntent) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates using a provider selected through the given Criteria, and
 * callbacks delivered via the provided {@link android.app.PendingIntent PendingIntent}.
 *
 * <p>Note: Since Android KitKat, Criteria requests will always result in using the
 * {@link #FUSED_PROVIDER}.
 *
 * <p>See {@link #requestLocationUpdates(java.lang.String,long,float,android.app.PendingIntent)} for more detail on
 * how this method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param minTimeMs minimum time interval between location updates in milliseconds
 * @param minDistanceM minimum distance between location updates in meters
 * @param criteria contains parameters to choose the appropriate provider for location updates
 * This value cannot be {@code null}.
 * @param pendingIntent the pending intent to send location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if pendingIntent is null
 * @throws java.lang.SecurityException if no suitable permission is present
 *
 * @deprecated Use {@link #requestLocationUpdates(java.lang.String,long,float,android.app.PendingIntent)} instead
 * to explicitly select a provider.
 */

@Deprecated
public void requestLocationUpdates(long minTimeMs, float minDistanceM, @android.annotation.NonNull android.location.Criteria criteria, @android.annotation.NonNull android.app.PendingIntent pendingIntent) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates from the specified provider, using a {@link android.location.LocationRequest LocationRequest},
 * and a callback on the specified {@link java.util.concurrent.Executor Executor}.
 *
 * <p>Only one request can be registered for each unique listener/provider pair, so any
 * subsequent requests with the same provider and listener will overwrite all associated
 * arguments. The same listener may be used across multiple providers with different requests
 * for each provider.
 *
 * <p>It may take some time to receive the first location update depending on the conditions the
 * device finds itself in. In order to take advantage of cached locations, application may
 * consider using {@link #getLastKnownLocation(java.lang.String)} or {@link #getCurrentLocation(java.lang.String,android.location.LocationRequest,android.os.CancellationSignal,java.util.concurrent.Executor,java.util.function.Consumer)} instead.
 *
 * <p>See {@link android.location.LocationRequest LocationRequest} documentation for an explanation of various request parameters
 * and how they can affect the received locations.
 *
 * <p>If your application wants to passively observe location updates from all providers, then
 * use the {@link #PASSIVE_PROVIDER}. This provider does not turn on or modify active location
 * providers, so you do not need to be as careful about minimum time and minimum distance
 * parameters. However, if your application performs heavy work on a location update (such as
 * network activity) then you should set an explicit fastest interval on your location request
 * in case another application enables a location provider with extremely fast updates.
 *
 * <p>In case the provider you have selected is disabled, location updates will cease, and a
 * provider availability update will be sent. As soon as the provider is enabled again, another
 * provider availability update will be sent and location updates will resume.
 *
 * <p>Locations returned from {@link #GPS_PROVIDER} are with respect to the primary GNSS antenna
 * position within the device. {@link #getGnssAntennaInfos()} may be used to determine the GNSS
 * antenna position with respect to the Android Coordinate System, and convert between them if
 * necessary. This is generally only necessary for high accuracy applications.
 *
 * <p>When location callbacks are invoked, the system will hold a wakelock on your
 * application's behalf for some period of time, but not indefinitely. If your application
 * requires a long running wakelock within the location callback, you should acquire it
 * yourself.
 *
 * <p>Spamming location requests is a drain on system resources, and the system has preventative
 * measures in place to ensure that this behavior will never result in more locations than could
 * be achieved with a single location request with an equivalent interval that is left in place
 * the whole time. As part of this amelioration, applications that target Android S and above
 * may receive cached or historical locations through their listener. These locations will never
 * be older than the interval of the location request.
 *
 * <p>To unregister for location updates, use {@link #removeUpdates(android.location.LocationListener)}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param locationRequest the location request containing location parameters
 * This value cannot be {@code null}.
 * @param executor the executor handling listener callbacks
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener the listener to receive location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if locationRequest is null
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException if no suitable permission is present
 */

public void requestLocationUpdates(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.LocationRequest locationRequest, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.LocationListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Register for location updates from the specified provider, using a {@link android.location.LocationRequest LocationRequest},
 * and callbacks delivered via the provided {@link android.app.PendingIntent PendingIntent}.
 *
 * <p>The delivered pending intents will contain extras with the callback information. The keys
 * used for the extras are {@link #KEY_LOCATION_CHANGED} and {@link #KEY_PROVIDER_ENABLED}. See
 * the documentation for each respective extra key for information on the values.
 *
 * <p>To unregister for location updates, use {@link #removeUpdates(android.app.PendingIntent)}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param provider a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param locationRequest the location request containing location parameters
 * This value cannot be {@code null}.
 * @param pendingIntent the pending intent to send location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if locationRequest is null
 * @throws java.lang.IllegalArgumentException if pendingIntent is null
 * @throws java.lang.SecurityException if no suitable permission is present
 */

public void requestLocationUpdates(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.LocationRequest locationRequest, @android.annotation.NonNull android.app.PendingIntent pendingIntent) { throw new RuntimeException("Stub!"); }

/**
 * Requests that the given provider flush any batched locations to listeners. The given listener
 * (registered with the provider) will have {@link android.location.LocationListener#onFlushComplete(int) LocationListener#onFlushComplete(int)}
 * invoked with the given result code after any locations that were flushed have been delivered.
 * If {@link #removeUpdates(android.location.LocationListener)} is invoked before the flush callback is executed,
 * then the flush callback will never be executed.
 *
 * @param provider    a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param listener    a listener registered under the provider
 * This value cannot be {@code null}.
 * @param requestCode an arbitrary integer passed through to
 *                    {@link android.location.LocationListener#onFlushComplete(int) LocationListener#onFlushComplete(int)}
 *
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if listener is null or is not registered under the provider
 */

public void requestFlush(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.LocationListener listener, int requestCode) { throw new RuntimeException("Stub!"); }

/**
 * Requests that the given provider flush any batched locations to listeners. The given
 * PendingIntent (registered with the provider) will be sent with {@link #KEY_FLUSH_COMPLETE}
 * present in the extra keys, and {@code requestCode} as the corresponding value.
 *
 * @param provider      a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param pendingIntent a pendingIntent registered under the provider
 * This value cannot be {@code null}.
 * @param requestCode   an arbitrary integer that will be passed back as the extra value for
 *                      {@link #KEY_FLUSH_COMPLETE}
 *
 * @throws java.lang.IllegalArgumentException if provider is null or doesn't exist
 * @throws java.lang.IllegalArgumentException if pending intent is null or is not registered under the
 *                                  provider
 */

public void requestFlush(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.app.PendingIntent pendingIntent, int requestCode) { throw new RuntimeException("Stub!"); }

/**
 * Removes all location updates for the specified {@link android.location.LocationListener LocationListener}. The given listener
 * is guaranteed not to receive any invocations that <b>happens-after</b> this method is
 * invoked.
 *
 * <p>If the given listener has any batched requests, this method will not flush any incomplete
 * location batches before stopping location updates. If you wish to flush any pending locations
 * before stopping, you must first call {@link #requestFlush(java.lang.String,android.location.LocationListener,int)} and
 * then call this method once the flush is complete. If this method is invoked before the flush
 * is complete, you may not receive the flushed locations.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param listener listener that no longer needs location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if listener is null
 */

public void removeUpdates(@android.annotation.NonNull android.location.LocationListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Removes location updates for the specified {@link android.app.PendingIntent PendingIntent}. Following this call, the
 * PendingIntent will no longer receive location updates.
 *
 * <p>See {@link #removeUpdates(android.location.LocationListener)} for more detail on how this method works.
 *
 * @param pendingIntent pending intent that no longer needs location updates
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if pendingIntent is null
 */

public void removeUpdates(@android.annotation.NonNull android.app.PendingIntent pendingIntent) { throw new RuntimeException("Stub!"); }

/**
 * Returns true if the given location provider exists on this device, irrespective of whether
 * it is currently enabled or not.
 *
 * @param provider a potential location provider
 * This value cannot be {@code null}.
 * @return true if the location provider exists, false otherwise
 *
 * @throws java.lang.IllegalArgumentException if provider is null
 */

public boolean hasProvider(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Returns a list of the names of all available location providers. All providers are returned,
 * including those that are currently disabled.
 *
 * @return list of provider names

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getAllProviders() { throw new RuntimeException("Stub!"); }

/**
 * Returns a list of the names of available location providers. If {@code enabledOnly} is false,
 * this is functionally the same as {@link #getAllProviders()}.
 *
 * @param enabledOnly if true then only enabled providers are included
 * @return list of provider names

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<java.lang.String> getProviders(boolean enabledOnly) { throw new RuntimeException("Stub!"); }

/**
 * Returns a list of the names of available location providers that satisfy the given criteria.
 *
 * @param criteria the criteria that providers must match
 * This value cannot be {@code null}.
 * @param enabledOnly if true then only enabled providers are included
 * @return list of provider names
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if criteria is null
 *
 * @deprecated Criteria based APIs are deprecated, prefer to select a provider explicitly.
 */

@Deprecated
@android.annotation.NonNull
public java.util.List<java.lang.String> getProviders(@android.annotation.NonNull android.location.Criteria criteria, boolean enabledOnly) { throw new RuntimeException("Stub!"); }

/**
 * Returns the name of the provider that best meets the given criteria. Only providers that are
 * permitted to be accessed by the caller will be returned. If several providers meet the
 * criteria, the one with the best accuracy is returned. If no provider meets the criteria, the
 * criteria are loosened in the following order:
 *
 * <ul>
 * <li> power requirement
 * <li> accuracy
 * <li> bearing
 * <li> speed
 * <li> altitude
 * </ul>
 *
 * <p> Note that the requirement on monetary cost is not removed in this process.
 *
 * @param criteria the criteria that need to be matched
 * This value cannot be {@code null}.
 * @param enabledOnly if true then only enabled providers are included
 * @return name of the provider that best matches the criteria, or null if none match
 *
 * @throws java.lang.IllegalArgumentException if criteria is null
 *
 * @deprecated Criteria based APIs are deprecated, prefer to select a provider explicitly.
 */

@Deprecated
@android.annotation.Nullable
public java.lang.String getBestProvider(@android.annotation.NonNull android.location.Criteria criteria, boolean enabledOnly) { throw new RuntimeException("Stub!"); }

/**
 * Returns the information about the location provider with the given name, or null if no
 * provider exists by that name.
 *
 * @param provider a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @return location provider information, or null if provider does not exist
 *
 * @throws java.lang.IllegalArgumentException if provider is null
 *
 * @deprecated This method has no way to indicate that a provider's properties are unknown, and
 * so may return incorrect results on rare occasions. Use {@link #getProviderProperties(java.lang.String)}
 * instead.
 */

@Deprecated
@android.annotation.Nullable
public android.location.LocationProvider getProvider(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Returns the properties of the given provider, or null if the properties are currently
 * unknown. Provider properties may change over time, although this is discouraged, and should
 * be rare. The most common transition is when provider properties go from being unknown to
 * known, which is most common near boot time.
 *
 * @param provider a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @return location provider properties, or null if properties are currently unknown
 *
 * @throws java.lang.IllegalArgumentException if provider is null or does not exist
 */

@android.annotation.Nullable
public android.location.provider.ProviderProperties getProviderProperties(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Sends additional commands to a location provider. Can be used to support provider specific
 * extensions to the Location Manager API.
 *
 * @param provider a provider listed by {@link #getAllProviders()}
 * This value cannot be {@code null}.
 * @param command  name of the command to send to the provider
 * This value cannot be {@code null}.
 * @param extras   optional arguments for the command, or null
 * @return true always, the return value may be ignored
 */

public boolean sendExtraCommand(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull java.lang.String command, @android.annotation.Nullable android.os.Bundle extras) { throw new RuntimeException("Stub!"); }

/**
 * Creates a test location provider and adds it to the set of active providers. This provider
 * will replace any provider with the same name that exists prior to this call.
 *
 * @param provider the provider name
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null
 * @throws java.lang.SecurityException if {@link android.app.AppOpsManager#OPSTR_MOCK_LOCATION
 * mock location app op} is not set to {@link android.app.AppOpsManager#MODE_ALLOWED
 * allowed} for your app.
 
 * @param powerUsage Value is {@link android.location.provider.ProviderProperties#POWER_USAGE_LOW}, {@link android.location.provider.ProviderProperties#POWER_USAGE_MEDIUM}, or {@link android.location.provider.ProviderProperties#POWER_USAGE_HIGH}

 * @param accuracy Value is {@link android.location.provider.ProviderProperties#ACCURACY_FINE}, or {@link android.location.provider.ProviderProperties#ACCURACY_COARSE}
 */

public void addTestProvider(@android.annotation.NonNull java.lang.String provider, boolean requiresNetwork, boolean requiresSatellite, boolean requiresCell, boolean hasMonetaryCost, boolean supportsAltitude, boolean supportsSpeed, boolean supportsBearing, int powerUsage, int accuracy) { throw new RuntimeException("Stub!"); }

/**
 * Creates a test location provider and adds it to the set of active providers. This provider
 * will replace any provider with the same name that exists prior to this call.
 *
 * @param provider the provider name
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null
 * @throws java.lang.IllegalArgumentException if properties is null
 * @throws java.lang.SecurityException if {@link android.app.AppOpsManager#OPSTR_MOCK_LOCATION
 * mock location app op} is not set to {@link android.app.AppOpsManager#MODE_ALLOWED
 * allowed} for your app.

 * @param properties This value cannot be {@code null}.
 */

public void addTestProvider(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.provider.ProviderProperties properties) { throw new RuntimeException("Stub!"); }

/**
 * Creates a test location provider and adds it to the set of active providers. This provider
 * will replace any provider with the same name that exists prior to this call.
 *
 * @param provider the provider name
 * This value cannot be {@code null}.
 * @param properties the provider properties
 * This value cannot be {@code null}.
 * @param extraAttributionTags additional attribution tags associated with this provider
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null
 * @throws java.lang.IllegalArgumentException if properties is null
 * @throws java.lang.SecurityException if {@link android.app.AppOpsManager#OPSTR_MOCK_LOCATION
 * mock location app op} is not set to {@link android.app.AppOpsManager#MODE_ALLOWED
 * allowed} for your app.
 */

public void addTestProvider(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.provider.ProviderProperties properties, @android.annotation.NonNull java.util.Set<java.lang.String> extraAttributionTags) { throw new RuntimeException("Stub!"); }

/**
 * Removes the test location provider with the given name or does nothing if no such test
 * location provider exists.
 *
 * @param provider the provider name
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if provider is null
 * @throws java.lang.SecurityException if {@link android.app.AppOpsManager#OPSTR_MOCK_LOCATION
 * mock location app op} is not set to {@link android.app.AppOpsManager#MODE_ALLOWED
 * allowed} for your app.
 */

public void removeTestProvider(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Sets a new location for the given test provider. This location will be identiable as a mock
 * location to all clients via {@link android.location.Location#isMock() Location#isMock()}.
 *
 * <p>The location object must have a minimum number of fields set to be considered valid, as
 * per documentation on {@link android.location.Location Location} class.
 *
 * @param provider the provider name
 * This value cannot be {@code null}.
 * @param location the mock location
 *
 * This value cannot be {@code null}.
 * @throws java.lang.SecurityException if {@link android.app.AppOpsManager#OPSTR_MOCK_LOCATION
 * mock location app op} is not set to {@link android.app.AppOpsManager#MODE_ALLOWED
 * allowed} for your app.
 * @throws java.lang.IllegalArgumentException if the provider is null or not a test provider
 * @throws java.lang.IllegalArgumentException if the location is null or incomplete
 */

public void setTestProviderLocation(@android.annotation.NonNull java.lang.String provider, @android.annotation.NonNull android.location.Location location) { throw new RuntimeException("Stub!"); }

/**
 * Does nothing.
 *
 * @deprecated This method has always been a no-op, and may be removed in the future.

 * @param provider This value cannot be {@code null}.
 */

@Deprecated
public void clearTestProviderLocation(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Sets the given test provider to be enabled or disabled.
 *
 * @param provider the provider name
 * This value cannot be {@code null}.
 * @param enabled the mock enabled value
 *
 * @throws java.lang.SecurityException if {@link android.app.AppOpsManager#OPSTR_MOCK_LOCATION
 * mock location app op} is not set to {@link android.app.AppOpsManager#MODE_ALLOWED
 * allowed} for your app.
 * @throws java.lang.IllegalArgumentException if provider is null or not a test provider
 */

public void setTestProviderEnabled(@android.annotation.NonNull java.lang.String provider, boolean enabled) { throw new RuntimeException("Stub!"); }

/**
 * Equivalent to calling {@link #setTestProviderEnabled(java.lang.String,boolean)} to disable a test
 * provider.
 *
 * @deprecated Use {@link #setTestProviderEnabled(java.lang.String,boolean)} instead.

 * @param provider This value cannot be {@code null}.
 */

@Deprecated
public void clearTestProviderEnabled(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * This method has no effect as provider status has been deprecated and is no longer supported.
 *
 * @deprecated This method has no effect.
 
 * @param provider This value cannot be {@code null}.

 * @param extras This value may be {@code null}.
 */

@Deprecated
public void setTestProviderStatus(@android.annotation.NonNull java.lang.String provider, int status, @android.annotation.Nullable android.os.Bundle extras, long updateTime) { throw new RuntimeException("Stub!"); }

/**
 * This method has no effect as provider status has been deprecated and is no longer supported.
 *
 * @deprecated This method has no effect.

 * @param provider This value cannot be {@code null}.
 */

@Deprecated
public void clearTestProviderStatus(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Sets a proximity alert for the location given by the position (latitude, longitude) and the
 * given radius.
 *
 * <p>When the device detects that it has entered or exited the area surrounding the location,
 * the given PendingIntent will be fired.
 *
 * <p>The fired intent will have a boolean extra added with key {@link #KEY_PROXIMITY_ENTERING}.
 * If the value is true, the device is entering the proximity region; if false, it is exiting.
 *
 * <p>Due to the approximate nature of position estimation, if the device passes through the
 * given area briefly, it is possible that no Intent will be fired. Similarly, an intent could
 * be fired if the device passes very close to the given area but does not actually enter it.
 *
 * <p>Before API version 17, this method could be used with
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} or
 * {@link android.Manifest.permission#ACCESS_COARSE_LOCATION}. From API version 17 and onwards,
 * this method requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION} permission.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param latitude      the latitude of the central point of the alert region
 * @param longitude     the longitude of the central point of the alert region
 * @param radius        the radius of the central point of the alert region in meters
 * @param expiration    expiration realtime for this proximity alert in milliseconds, or -1 to
 *                      indicate no expiration
 * @param pendingIntent a {@link android.app.PendingIntent PendingIntent} that will sent when entry to or exit from the
 *                      alert region is detected
 * This value cannot be {@code null}.
 * @throws java.lang.SecurityException if {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 *                           permission is not present
 */

public void addProximityAlert(double latitude, double longitude, float radius, long expiration, @android.annotation.NonNull android.app.PendingIntent pendingIntent) { throw new RuntimeException("Stub!"); }

/**
 * Removes the proximity alert with the given PendingIntent.
 *
 * <p>Before API version 17, this method could be used with
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} or
 * {@link android.Manifest.permission#ACCESS_COARSE_LOCATION}.
 * From API version 17 and onwards, this method requires
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} permission.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_COARSE_LOCATION} or {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param intent the PendingIntent that no longer needs to be notified of
 * proximity alerts
 *
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if intent is null
 * @throws java.lang.SecurityException if {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * permission is not present
 */

public void removeProximityAlert(@android.annotation.NonNull android.app.PendingIntent intent) { throw new RuntimeException("Stub!"); }

/**
 * Returns the supported capabilities of the GNSS chipset.

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.location.GnssCapabilities getGnssCapabilities() { throw new RuntimeException("Stub!"); }

/**
 * Returns the model year of the GNSS hardware and software build, or 0 if the model year
 * is before 2016.
 */

public int getGnssYearOfHardware() { throw new RuntimeException("Stub!"); }

/**
 * Returns the model name (including vendor and hardware/software version) of the GNSS hardware
 * driver, or null if this information is not available.
 *
 * <p>No device-specific serial number or ID is returned from this API.
 */

@android.annotation.Nullable
public java.lang.String getGnssHardwareModelName() { throw new RuntimeException("Stub!"); }

/**
 * Returns the current list of GNSS antenna infos, or null if unknown or unsupported.
 *
 * @see #getGnssCapabilities()
 */

@android.annotation.Nullable
public java.util.List<android.location.GnssAntennaInfo> getGnssAntennaInfos() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves information about the current status of the GPS engine. This should only be called
 * from within the {@link android.location.GpsStatus.Listener#onGpsStatusChanged GpsStatus.Listener#onGpsStatusChanged} callback to ensure that the
 * data is copied atomically.
 *
 * The caller may either pass in an existing {@link android.location.GpsStatus GpsStatus} object to be overwritten, or pass
 * null to create a new {@link android.location.GpsStatus GpsStatus} object.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param status object containing GPS status details, or null.
 * @return status object containing updated GPS status.
 *
 * @deprecated GpsStatus APIs are deprecated, use {@link android.location.GnssStatus GnssStatus} APIs instead. No longer
 * supported in apps targeting S and above.
 */

@Deprecated
@android.annotation.Nullable
public android.location.GpsStatus getGpsStatus(@android.annotation.Nullable android.location.GpsStatus status) { throw new RuntimeException("Stub!"); }

/**
 * Adds a GPS status listener.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param listener GPS status listener object to register
 * @return true if the listener was successfully added
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 *
 * @deprecated Use {@link #registerGnssStatusCallback(android.location.GnssStatus.Callback,android.os.Handler)} or {@link
 * #registerGnssStatusCallback(java.util.concurrent.Executor,android.location.GnssStatus.Callback)} instead.
 */

@Deprecated
public boolean addGpsStatusListener(android.location.GpsStatus.Listener listener) { throw new RuntimeException("Stub!"); }

/**
 * Removes a GPS status listener.
 *
 * @param listener GPS status listener object to remove
 *
 * @deprecated use {@link #unregisterGnssStatusCallback(android.location.GnssStatus.Callback)} instead. No longer
 * supported in apps targeting S and above.
 */

@Deprecated
public void removeGpsStatusListener(android.location.GpsStatus.Listener listener) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS status callback. This method must be called from a {@link android.os.Looper Looper} thread,
 * and callbacks will occur on that looper.
 *
 * <p>See {@link #registerGnssStatusCallback(java.util.concurrent.Executor,android.location.GnssStatus.Callback)} for more detail on
 * how this method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param callback the callback to register
 * This value cannot be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 *
 * @deprecated Use {@link #registerGnssStatusCallback(android.location.GnssStatus.Callback,android.os.Handler)} or {@link
 * #registerGnssStatusCallback(java.util.concurrent.Executor,android.location.GnssStatus.Callback)} instead.
 */

@Deprecated
public boolean registerGnssStatusCallback(@android.annotation.NonNull android.location.GnssStatus.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS status callback.
 *
 * <p>See {@link #registerGnssStatusCallback(java.util.concurrent.Executor,android.location.GnssStatus.Callback)} for more detail on
 * how this method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param callback the callback to register
 * This value cannot be {@code null}.
 * @param handler  the handler the callback runs on
 * This value may be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if callback is null
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean registerGnssStatusCallback(@android.annotation.NonNull android.location.GnssStatus.Callback callback, @android.annotation.Nullable android.os.Handler handler) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS status callback. GNSS status information will only be received while the
 * {@link #GPS_PROVIDER} is enabled, and while the client app is in the foreground.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param executor the executor that the callback runs on
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback the callback to register
 * This value cannot be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if callback is null
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean registerGnssStatusCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.GnssStatus.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Removes a GNSS status callback.
 *
 * @param callback GNSS status callback object to remove

 * This value cannot be {@code null}.
 */

public void unregisterGnssStatusCallback(@android.annotation.NonNull android.location.GnssStatus.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * No-op method to keep backward-compatibility.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @deprecated Use {@link #addNmeaListener(android.location.OnNmeaMessageListener,android.os.Handler)} or {@link
 * #addNmeaListener(java.util.concurrent.Executor,android.location.OnNmeaMessageListener)} instead.

 * @param listener This value cannot be {@code null}.
 */

@Deprecated
public boolean addNmeaListener(@android.annotation.NonNull android.location.GpsStatus.NmeaListener listener) { throw new RuntimeException("Stub!"); }

/**
 * No-op method to keep backward-compatibility.
 *
 * @deprecated Use {@link #removeNmeaListener(android.location.OnNmeaMessageListener)} instead.

 * @param listener This value cannot be {@code null}.
 */

@Deprecated
public void removeNmeaListener(@android.annotation.NonNull android.location.GpsStatus.NmeaListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Adds an NMEA listener.
 *
 * <p>See {@link #addNmeaListener(java.util.concurrent.Executor,android.location.OnNmeaMessageListener)} for more detail on how this
 * method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param listener the listener to register
 * This value cannot be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 * @deprecated Use {@link #addNmeaListener(android.location.OnNmeaMessageListener,android.os.Handler)} or {@link
 * #addNmeaListener(java.util.concurrent.Executor,android.location.OnNmeaMessageListener)} instead.
 */

@Deprecated
public boolean addNmeaListener(@android.annotation.NonNull android.location.OnNmeaMessageListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Adds an NMEA listener.
 *
 * <p>See {@link #addNmeaListener(java.util.concurrent.Executor,android.location.OnNmeaMessageListener)} for more detail on how this
 * method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param listener the listener to register
 * This value cannot be {@code null}.
 * @param handler  the handler that the listener runs on
 * This value may be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean addNmeaListener(@android.annotation.NonNull android.location.OnNmeaMessageListener listener, @android.annotation.Nullable android.os.Handler handler) { throw new RuntimeException("Stub!"); }

/**
 * Adds an NMEA listener. GNSS NMEA information will only be received while the
 * {@link #GPS_PROVIDER} is enabled, and while the client app is in the foreground.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param listener the listener to register
 * This value cannot be {@code null}.
 * @param executor the executor that the listener runs on
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if listener is null
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean addNmeaListener(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.OnNmeaMessageListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Removes an NMEA listener.
 *
 * @param listener a {@link android.location.OnNmeaMessageListener OnNmeaMessageListener} object to remove

 * This value cannot be {@code null}.
 */

public void removeNmeaListener(@android.annotation.NonNull android.location.OnNmeaMessageListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS measurements callback which will run on a binder thread.
 *
 * <p>See {@link #registerGnssMeasurementsCallback(java.util.concurrent.Executor,android.location.GnssMeasurementsEvent.Callback)
 * for more detail on how this method works.
 *
 * @param callback a {@link GnssMeasurementsEvent.Callback} object to register
 * @return {@code true} always
 *
 * @deprecated Use {@link
 * #registerGnssMeasurementsCallback(GnssMeasurementsEvent.Callback, Handler)} or {@link
 * #registerGnssMeasurementsCallback(Executor, GnssMeasurementsEvent.Callback)} instead.
 
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}

 * @param callback This value cannot be {@code null}.
 */

@Deprecated
public boolean registerGnssMeasurementsCallback(@android.annotation.NonNull android.location.GnssMeasurementsEvent.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS measurements callback.
 *
 * <p>See {@link #registerGnssMeasurementsCallback(java.util.concurrent.Executor,android.location.GnssMeasurementsEvent.Callback)
 * for more detail on how this method works.
 *
 * @param callback a {@link GnssMeasurementsEvent.Callback} object to register
 * @param handler  the handler that the callback runs on
 * @return {@code true} always
 *
 * @throws IllegalArgumentException if callback is null
 * @throws SecurityException if the ACCESS_FINE_LOCATION permission is not present

 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 
 * @param callback This value cannot be {@code null}.

 * @param handler This value may be {@code null}.
 */

public boolean registerGnssMeasurementsCallback(@android.annotation.NonNull android.location.GnssMeasurementsEvent.Callback callback, @android.annotation.Nullable android.os.Handler handler) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS measurements callback. GNSS measurements information will only be received
 * while the {@link #GPS_PROVIDER} is enabled, and while the client app is in the foreground.
 *
 * <p>Not all GNSS chipsets support measurements updates, see {@link #getGnssCapabilities()}.
 *
 * <p class="caution">On Android R devices that have not yet upgraded to Android R QPR1, using
 * this API will cause unavoidable crashes in the client application when GNSS measurements
 * are received. If a client needs to receive GNSS measurements on Android R devices that have
 * not been upgraded to QPR1, clients are instead encouraged to use
 * <a href="https://developer.android.com/reference/androidx/core/location/LocationManagerCompat#registerGnssMeasurementsCallback(android.location.LocationManager,java.util.concurrent.Executor,android.location.GnssMeasurementsEvent.Callback)">LocationManagerCompat.registerGnssMeasurementsCallback()</a>
 * from the compat libraries instead to avoid this crash.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param executor the executor that the callback runs on
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback the callback to register
 * This value cannot be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if callback is null
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean registerGnssMeasurementsCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.GnssMeasurementsEvent.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS measurement callback.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param request  extra parameters to pass to GNSS measurement provider. For example, if {@link android.location.GnssMeasurementRequest#isFullTracking()                  } is true, GNSS chipset switches off
 *                 duty cycling.
 * This value cannot be {@code null}.
 * @param executor the executor that the callback runs on
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback a {@link android.location.GnssMeasurementsEvent.Callback GnssMeasurementsEvent.Callback} object to register.
 * This value cannot be {@code null}.
 * @return {@code true} always if the callback was added successfully, {@code false} otherwise.
 * @throws java.lang.IllegalArgumentException if request is null
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if callback is null
 * @throws java.lang.SecurityException        if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean registerGnssMeasurementsCallback(@android.annotation.NonNull android.location.GnssMeasurementRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.GnssMeasurementsEvent.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Unregisters a GPS Measurement callback.
 *
 * @param callback a {@link android.location.GnssMeasurementsEvent.Callback GnssMeasurementsEvent.Callback} object to remove.

 * This value cannot be {@code null}.
 */

public void unregisterGnssMeasurementsCallback(@android.annotation.NonNull android.location.GnssMeasurementsEvent.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS antenna info listener that will receive all changes to antenna info. Use
 * {@link #getGnssAntennaInfos()} to get current antenna info.
 *
 * <p>Not all GNSS chipsets support antenna info updates, see {@link #getGnssCapabilities()}. If
 * unsupported, the listener will never be invoked.
 *
 * <p>Prior to Android S, this requires the {@link android.Manifest.permission#ACCESS_FINE_LOCATION Manifest.permission#ACCESS_FINE_LOCATION}
 * permission.
 *
 * @param executor the executor that the listener runs on
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener the listener to register
 * This value cannot be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if listener is null
 */

public boolean registerAntennaInfoListener(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.GnssAntennaInfo.Listener listener) { throw new RuntimeException("Stub!"); }

/**
 * Unregisters a GNSS antenna info listener.
 *
 * @param listener a {@link android.location.GnssAntennaInfo.Listener GnssAntennaInfo.Listener} object to remove

 * This value cannot be {@code null}.
 */

public void unregisterAntennaInfoListener(@android.annotation.NonNull android.location.GnssAntennaInfo.Listener listener) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS navigation message callback which will run on a binder thread.
 *
 * <p>See
 * {@link #registerGnssNavigationMessageCallback(java.util.concurrent.Executor,android.location.GnssNavigationMessage.Callback)} for
 * more detail on how this method works.
 *
 * @param callback the callback to register
 * This value cannot be {@code null}.
 * @return {@code true} always
 *
 * @deprecated Use {@link
 * #registerGnssNavigationMessageCallback(android.location.GnssNavigationMessage.Callback,android.os.Handler)} or {@link
 * #registerGnssNavigationMessageCallback(java.util.concurrent.Executor,android.location.GnssNavigationMessage.Callback)} instead.
 */

@Deprecated
public boolean registerGnssNavigationMessageCallback(@android.annotation.NonNull android.location.GnssNavigationMessage.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS navigation message callback.
 *
 * <p>See
 * {@link #registerGnssNavigationMessageCallback(java.util.concurrent.Executor,android.location.GnssNavigationMessage.Callback)} for
 * more detail on how this method works.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param callback the callback to register
 * This value cannot be {@code null}.
 * @param handler  the handler that the callback runs on
 * This value may be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if callback is null
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean registerGnssNavigationMessageCallback(@android.annotation.NonNull android.location.GnssNavigationMessage.Callback callback, @android.annotation.Nullable android.os.Handler handler) { throw new RuntimeException("Stub!"); }

/**
 * Registers a GNSS navigation message callback. GNSS navigation messages will only be received
 * while the {@link #GPS_PROVIDER} is enabled, and while the client app is in the foreground.
 *
 * <p>Not all GNSS chipsets support navigation message updates, see
 * {@link #getGnssCapabilities()}.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @param executor the executor that the callback runs on
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback the callback to register
 * This value cannot be {@code null}.
 * @return {@code true} always
 *
 * @throws java.lang.IllegalArgumentException if executor is null
 * @throws java.lang.IllegalArgumentException if callback is null
 * @throws java.lang.SecurityException if the ACCESS_FINE_LOCATION permission is not present
 */

public boolean registerGnssNavigationMessageCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.location.GnssNavigationMessage.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Unregisters a GNSS Navigation Message callback.
 *
 * @param callback a {@link android.location.GnssNavigationMessage.Callback GnssNavigationMessage.Callback} object to remove.

 * This value cannot be {@code null}.
 */

public void unregisterGnssNavigationMessageCallback(@android.annotation.NonNull android.location.GnssNavigationMessage.Callback callback) { throw new RuntimeException("Stub!"); }

/**
 * Broadcast intent action when GNSS capabilities change. This is most common at boot time as
 * GNSS capabilities are queried from the chipset. Includes an intent extra,
 * {@link #EXTRA_GNSS_CAPABILITIES}, with the new {@link android.location.GnssCapabilities GnssCapabilities}.
 *
 * @see #EXTRA_GNSS_CAPABILITIES
 * @see #getGnssCapabilities()
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String ACTION_GNSS_CAPABILITIES_CHANGED = "android.location.action.GNSS_CAPABILITIES_CHANGED";

/**
 * Intent extra included with {@link #ACTION_GNSS_CAPABILITIES_CHANGED} broadcasts, containing
 * the new {@link android.location.GnssCapabilities GnssCapabilities}.
 *
 * @see #ACTION_GNSS_CAPABILITIES_CHANGED
 */

public static final java.lang.String EXTRA_GNSS_CAPABILITIES = "android.location.extra.GNSS_CAPABILITIES";

/**
 * Intent extra included with {@link #MODE_CHANGED_ACTION} broadcasts, containing the boolean
 * enabled state of location.
 *
 * @see #MODE_CHANGED_ACTION
 */

public static final java.lang.String EXTRA_LOCATION_ENABLED = "android.location.extra.LOCATION_ENABLED";

/**
 * Intent extra included with {@link #PROVIDERS_CHANGED_ACTION} broadcasts, containing the
 * boolean enabled state of the location provider that has changed.
 *
 * @see #PROVIDERS_CHANGED_ACTION
 * @see #EXTRA_PROVIDER_NAME
 */

public static final java.lang.String EXTRA_PROVIDER_ENABLED = "android.location.extra.PROVIDER_ENABLED";

/**
 * Intent extra included with {@link #PROVIDERS_CHANGED_ACTION} broadcasts, containing the name
 * of the location provider that has changed.
 *
 * @see #PROVIDERS_CHANGED_ACTION
 * @see #EXTRA_PROVIDER_ENABLED
 */

public static final java.lang.String EXTRA_PROVIDER_NAME = "android.location.extra.PROVIDER_NAME";

/**
 * Standard name of the fused location provider.
 *
 * <p>If present, this provider may combine inputs from several other location providers to
 * provide the best possible location fix. It is implicitly used for all requestLocationUpdates
 * APIs that involve a {@link android.location.Criteria Criteria}.
 */

public static final java.lang.String FUSED_PROVIDER = "fused";

/**
 * Standard name of the GNSS location provider.
 *
 * <p>If present, this provider determines location using GNSS satellites. The responsiveness
 * and accuracy of location fixes may depend on GNSS signal conditions.
 *
 * <p>Locations returned from this provider are with respect to the primary GNSS antenna
 * position within the device. {@link #getGnssAntennaInfos()} may be used to determine the GNSS
 * antenna position with respect to the Android Coordinate System, and convert between them if
 * necessary. This is generally only necessary for high accuracy applications.
 *
 * <p>The extras Bundle for locations derived by this location provider may contain the
 * following key/value pairs:
 * <ul>
 * <li> satellites - the number of satellites used to derive the fix
 * </ul>
 */

public static final java.lang.String GPS_PROVIDER = "gps";

/**
 * Key used for an extra holding an integer request code when location flush completion is sent
 * using a PendingIntent.
 *
 * @see #requestFlush(String, PendingIntent, int)
 */

public static final java.lang.String KEY_FLUSH_COMPLETE = "flushComplete";

/**
 * Key used for an extra holding a array of {@link android.location.Location Location}s when a location change is sent
 * using a PendingIntent. This key will only be present if the location change includes
 * multiple (ie, batched) locations, otherwise only {@link #KEY_LOCATION_CHANGED} will be
 * present. Use {@link android.content.Intent#getParcelableArrayExtra(java.lang.String) Intent#getParcelableArrayExtra(String)} to retrieve the locations.
 *
 * <p>The array of locations will never be empty, and will ordered from earliest location to
 * latest location, the same as with {@link android.location.LocationListener#onLocationChanged(java.util.List) LocationListener#onLocationChanged(List)}.
 *
 * @see #requestLocationUpdates(String, LocationRequest, PendingIntent)
 */

public static final java.lang.String KEY_LOCATIONS = "locations";

/**
 * Key used for an extra holding a {@link android.location.Location Location} value when a location change is sent using
 * a PendingIntent. If the location change includes a list of batched locations via
 * {@link #KEY_LOCATIONS} then this key will still be present, and will hold the last location
 * in the batch. Use {@link android.content.Intent#getParcelableExtra(java.lang.String) Intent#getParcelableExtra(String)} to retrieve the location.
 *
 * @see #requestLocationUpdates(String, LocationRequest, PendingIntent)
 */

public static final java.lang.String KEY_LOCATION_CHANGED = "location";

/**
 * Key used for an extra holding a boolean enabled/disabled status value when a provider
 * enabled/disabled event is broadcast using a PendingIntent.
 *
 * @see #requestLocationUpdates(String, LocationRequest, PendingIntent)
 */

public static final java.lang.String KEY_PROVIDER_ENABLED = "providerEnabled";

/**
 * Key used for the Bundle extra holding a boolean indicating whether
 * a proximity alert is entering (true) or exiting (false)..
 */

public static final java.lang.String KEY_PROXIMITY_ENTERING = "entering";

/**
 * This key is no longer in use.
 *
 * <p>Key used for a Bundle extra holding an Integer status value when a status change is
 * broadcast using a PendingIntent.
 *
 * @deprecated Status changes are deprecated and no longer broadcast from Android Q onwards.
 */

@Deprecated public static final java.lang.String KEY_STATUS_CHANGED = "status";

/**
 * Broadcast intent action when the device location enabled state changes. From Android R and
 * above, will include a boolean intent extra, {@link #EXTRA_LOCATION_ENABLED}, with the enabled
 * state of location.
 *
 * @see #EXTRA_LOCATION_ENABLED
 * @see #isLocationEnabled()
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String MODE_CHANGED_ACTION = "android.location.MODE_CHANGED";

/**
 * Standard name of the network location provider.
 *
 * <p>If present, this provider determines location based on nearby of cell tower and WiFi
 * access points. Operation of this provider may require a data connection.
 */

public static final java.lang.String NETWORK_PROVIDER = "network";

/**
 * A special location provider for receiving locations without actively initiating a location
 * fix. This location provider is always present.
 *
 * <p>This provider can be used to passively receive location updates when other applications or
 * services request them without actually requesting the locations yourself. This provider will
 * only return locations generated by other providers.
 */

public static final java.lang.String PASSIVE_PROVIDER = "passive";

/**
 * Broadcast intent action when the set of enabled location providers changes. To check the
 * status of a provider, use {@link #isProviderEnabled(java.lang.String)}. From Android Q and above, will
 * include a string intent extra, {@link #EXTRA_PROVIDER_NAME}, with the name of the provider
 * whose state has changed. From Android R and above, will include a boolean intent extra,
 * {@link #EXTRA_PROVIDER_ENABLED}, with the enabled state of the provider.
 *
 * @see #EXTRA_PROVIDER_NAME
 * @see #EXTRA_PROVIDER_ENABLED
 * @see #isProviderEnabled(String)
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String PROVIDERS_CHANGED_ACTION = "android.location.PROVIDERS_CHANGED";
}

