/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.location;

import java.util.concurrent.Executor;

/**
 * Used for receiving notifications when the device location has changed. These methods are called
 * when the listener has been registered with the LocationManager.
 *
 * <div class="special reference">
 * <h3>Developer Guides</h3>
 * <p>For more information about identifying user location, read the
 * <a href="{@docRoot}guide/topics/location/obtaining-user-location.html">Obtaining User
 * Location</a> developer guide.</p>
 * </div>
 *
 * @see android.location.LocationManager#requestLocationUpdates(String, LocationRequest, Executor, LocationListener)
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface LocationListener {

/**
 * Called when the location has changed. A wakelock may be held on behalf on the listener for
 * some brief amount of time as this callback executes. If this callback performs long running
 * operations, it is the client's responsibility to obtain their own wakelock if necessary.
 *
 * @param location the updated location

 * This value cannot be {@code null}.
 */

public void onLocationChanged(@android.annotation.NonNull android.location.Location location);

/**
 * Called when the location has changed and locations are being delivered in batches. The
 * default implementation calls through to {@link #onLocationChanged(android.location.Location)} with all
 * locations in the batch. The list of locations is always guaranteed to be non-empty, and is
 * always guaranteed to be ordered from earliest location to latest location (so that the
 * earliest location in the batch is at index 0 in the list, and the latest location in the
 * batch is at index size-1 in the list).
 *
 * @see android.location.LocationRequest#getMaxUpdateDelayMillis()
 * @param locations the location list

 * This value cannot be {@code null}.
 */

public default void onLocationChanged(@android.annotation.NonNull java.util.List<android.location.Location> locations) { throw new RuntimeException("Stub!"); }

/**
 * Invoked when a flush operation is complete and after flushed locations have been delivered.
 *
 * @param requestCode the request code passed into
 *                    {@link android.location.LocationManager#requestFlush(java.lang.String,android.location.LocationListener,int) LocationManager#requestFlush(String, LocationListener, int)}
 */

public default void onFlushComplete(int requestCode) { throw new RuntimeException("Stub!"); }

/**
 * This callback will never be invoked on Android Q and above, and providers can be considered
 * as always in the {@link android.location.LocationProvider#AVAILABLE LocationProvider#AVAILABLE} state.
 *
 * <p class="note">Note that this method only has a default implementation on Android R and
 * above, and this method must still be overridden in order to run successfully on Android
 * versions below R. LocationListenerCompat from the compat libraries may be used to avoid the
 * need to override for older platforms.
 *
 * @deprecated This callback will never be invoked on Android Q and above.
 */

@Deprecated
public default void onStatusChanged(java.lang.String provider, int status, android.os.Bundle extras) { throw new RuntimeException("Stub!"); }

/**
 * Called when a provider this listener is registered with becomes enabled.
 *
 * <p class="note">Note that this method only has a default implementation on Android R and
 * above, and this method must still be overridden in order to run successfully on Android
 * versions below R. LocationListenerCompat from the compat libraries may be used to avoid the
 * need to override for older platforms.
 *
 * @param provider the name of the location provider

 * This value cannot be {@code null}.
 */

public default void onProviderEnabled(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }

/**
 * Called when the provider this listener is registered with becomes disabled. If a provider is
 * disabled when this listener is registered, this callback will be invoked immediately.
 *
 * <p class="note">Note that this method only has a default implementation on Android R and
 * above, and this method must still be overridden in order to run successfully on Android
 * versions below R. LocationListenerCompat from the compat libraries may be used to avoid the
 * need to override for older platforms.
 *
 * @param provider the name of the location provider

 * This value cannot be {@code null}.
 */

public default void onProviderDisabled(@android.annotation.NonNull java.lang.String provider) { throw new RuntimeException("Stub!"); }
}

