/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.app.appsearch;

import android.app.appsearch.exceptions.AppSearchException;
import android.app.appsearch.observer.ObserverCallback;
import android.app.appsearch.observer.ObserverSpec;
import java.util.concurrent.Executor;

/**
 * Provides a connection to all AppSearch databases the querying application has been granted access
 * to.
 *
 * <p>This class is thread safe.
 *
 * @see android.app.appsearch.AppSearchSession
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class GlobalSearchSession implements java.io.Closeable {

GlobalSearchSession() { throw new RuntimeException("Stub!"); }

/**
 * Retrieves {@link android.app.appsearch.GenericDocument GenericDocument} documents, belonging to the specified package name and
 * database name and identified by the namespace and ids in the request, from the {@link android.app.appsearch.GlobalSearchSession GlobalSearchSession} database.
 *
 * <p>If the package or database doesn't exist or if the calling package doesn't have access,
 * the gets will be handled as failures in an {@link android.app.appsearch.AppSearchBatchResult AppSearchBatchResult} object in the
 * callback.
 *
 * @param packageName the name of the package to get from
 * This value cannot be {@code null}.
 * @param databaseName the name of the database to get from
 * This value cannot be {@code null}.
 * @param request a request containing a namespace and IDs to get documents for.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive the pending result of performing this operation. The keys
 *     of the returned {@link android.app.appsearch.AppSearchBatchResult AppSearchBatchResult} are the input IDs. The values are the
 *     returned {@link android.app.appsearch.GenericDocument GenericDocument}s on success, or a failed {@link android.app.appsearch.AppSearchResult AppSearchResult}
 *     otherwise. IDs that are not found will return a failed {@link android.app.appsearch.AppSearchResult AppSearchResult} with a
 *     result code of {@link android.app.appsearch.AppSearchResult#RESULT_NOT_FOUND AppSearchResult#RESULT_NOT_FOUND}. If an unexpected internal error
 *     occurs in the AppSearch service, {@link android.app.appsearch.BatchResultCallback#onSystemError BatchResultCallback#onSystemError} will be
 *     invoked with a {@link java.lang.Throwable Throwable}.

 * This value cannot be {@code null}.
 */

public void getByDocumentId(@android.annotation.NonNull java.lang.String packageName, @android.annotation.NonNull java.lang.String databaseName, @android.annotation.NonNull android.app.appsearch.GetByDocumentIdRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.app.appsearch.BatchResultCallback<java.lang.String,android.app.appsearch.GenericDocument> callback) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves documents from all AppSearch databases that the querying application has access to.
 *
 * <p>Applications can be granted access to documents by specifying {@link android.app.appsearch.SetSchemaRequest.Builder#setSchemaTypeVisibilityForPackage  } when building a schema.
 *
 * <p>Document access can also be granted to system UIs by specifying {@link android.app.appsearch.SetSchemaRequest.Builder#setSchemaTypeDisplayedBySystem  } when building a schema.
 *
 * <p>See {@link android.app.appsearch.AppSearchSession#search AppSearchSession#search} for a detailed explanation on forming a query string.
 *
 * <p>This method is lightweight. The heavy work will be done in {@link android.app.appsearch.SearchResults#getNextPage  }.
 *
 * @param queryExpression query string to search.
 * This value cannot be {@code null}.
 * @param searchSpec spec for setting document filters, adding projection, setting term match
 *     type, etc.
 * This value cannot be {@code null}.
 * @return a {@link android.app.appsearch.SearchResults SearchResults} object for retrieved matched documents.
 
 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.app.appsearch.SearchResults search(@android.annotation.NonNull java.lang.String queryExpression, @android.annotation.NonNull android.app.appsearch.SearchSpec searchSpec) { throw new RuntimeException("Stub!"); }

/**
 * Retrieves the collection of schemas most recently successfully provided to {@link android.app.appsearch.AppSearchSession#setSchema  } for any types belonging to the requested package and database
 * that the caller has been granted access to.
 *
 * <p>If the requested package/database combination does not exist or the caller has not been
 * granted access to it, then an empty GetSchemaResponse will be returned.
 *
 * @param packageName the package that owns the requested {@link android.app.appsearch.AppSearchSchema AppSearchSchema} instances.
 * This value cannot be {@code null}.
 * @param databaseName the database that owns the requested {@link android.app.appsearch.AppSearchSchema AppSearchSchema} instances.
 * This value cannot be {@code null}.
 * @param executor This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback This value cannot be {@code null}.
 * @return The pending {@link android.app.appsearch.GetSchemaResponse GetSchemaResponse} containing the schemas that the caller has
 *     access to or an empty GetSchemaResponse if the request package and database does not
 *     exist, has not set a schema or contains no schemas that are accessible to the caller.
 */

public void getSchema(@android.annotation.NonNull java.lang.String packageName, @android.annotation.NonNull java.lang.String databaseName, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<android.app.appsearch.GetSchemaResponse>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Reports that a particular document has been used from a system surface.
 *
 * <p>See {@link android.app.appsearch.AppSearchSession#reportUsage AppSearchSession#reportUsage} for a general description of document usage, as
 * well as an API that can be used by the app itself.
 *
 * <p>Usage reported via this method is accounted separately from usage reported via {@link android.app.appsearch.AppSearchSession#reportUsage  } and may be accessed using the constants {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_SYSTEM_USAGE_COUNT  } and {@link android.app.appsearch.SearchSpec#RANKING_STRATEGY_SYSTEM_USAGE_LAST_USED_TIMESTAMP  }.
 *
 * @param request The usage reporting request.
 * This value cannot be {@code null}.
 * @param executor Executor on which to invoke the callback.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback Callback to receive errors. If the operation succeeds, the callback will be
 *     invoked with an {@link android.app.appsearch.AppSearchResult AppSearchResult} whose value is {@code null}. The callback will be
 *     invoked with an {@link android.app.appsearch.AppSearchResult AppSearchResult} of {@link android.app.appsearch.AppSearchResult#RESULT_SECURITY_ERROR AppSearchResult#RESULT_SECURITY_ERROR}
 *     if this API is invoked by an app which is not part of the system.
 */

public void reportSystemUsage(@android.annotation.NonNull android.app.appsearch.ReportSystemUsageRequest request, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<android.app.appsearch.AppSearchResult<java.lang.Void>> callback) { throw new RuntimeException("Stub!"); }

/**
 * Adds an {@link android.app.appsearch.observer.ObserverCallback ObserverCallback} to monitor changes within the databases owned by {@code
 * targetPackageName} if they match the given {@link
 * android.app.appsearch.observer.ObserverSpec}.
 *
 * <p>The observer callback is only triggered for data that changes after it is registered. No
 * notification about existing data is sent as a result of registering an observer. To find out
 * about existing data, you must use the {@link android.app.appsearch.GlobalSearchSession#search GlobalSearchSession#search} API.
 *
 * <p>If the data owned by {@code targetPackageName} is not visible to you, the registration
 * call will succeed but no notifications will be dispatched. Notifications could start flowing
 * later if {@code targetPackageName} changes its schema visibility settings.
 *
 * <p>If no package matching {@code targetPackageName} exists on the system, the registration
 * call will succeed but no notifications will be dispatched. Notifications could start flowing
 * later if {@code targetPackageName} is installed and starts indexing data.
 *
 * @param targetPackageName Package whose changes to monitor
 * This value cannot be {@code null}.
 * @param spec Specification of what types of changes to listen for
 * This value cannot be {@code null}.
 * @param executor Executor on which to call the {@code observer} callback methods.
 * This value cannot be {@code null}.
 * @param observer Callback to trigger when a schema or document changes
 * This value cannot be {@code null}.
 * @throws android.app.appsearch.exceptions.AppSearchException If an unexpected error occurs when trying to register an observer.
 */

public void registerObserverCallback(@android.annotation.NonNull java.lang.String targetPackageName, @android.annotation.NonNull android.app.appsearch.observer.ObserverSpec spec, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.app.appsearch.observer.ObserverCallback observer) throws android.app.appsearch.exceptions.AppSearchException { throw new RuntimeException("Stub!"); }

/**
 * Removes previously registered {@link android.app.appsearch.observer.ObserverCallback ObserverCallback} instances from the system.
 *
 * <p>All instances of {@link android.app.appsearch.observer.ObserverCallback ObserverCallback} which are registered to observe {@code
 * targetPackageName} and compare equal to the provided callback using the provided argument's
 * {@code ObserverCallback#equals} will be removed.
 *
 * <p>If no matching observers have been registered, this method has no effect. If multiple
 * matching observers have been registered, all will be removed.
 *
 * @param targetPackageName Package which the observers to be removed are listening to.
 * This value cannot be {@code null}.
 * @param observer Callback to unregister.
 * This value cannot be {@code null}.
 * @throws android.app.appsearch.exceptions.AppSearchException if an error occurs trying to remove the observer, such as a
 *     failure to communicate with the system service. Note that no error will be thrown if the
 *     provided observer doesn't match any registered observer.
 */

public void unregisterObserverCallback(@android.annotation.NonNull java.lang.String targetPackageName, @android.annotation.NonNull android.app.appsearch.observer.ObserverCallback observer) throws android.app.appsearch.exceptions.AppSearchException { throw new RuntimeException("Stub!"); }

/**
 * Closes the {@link android.app.appsearch.GlobalSearchSession GlobalSearchSession}. Persists all mutations, including usage reports, to
 * disk.
 */

public void close() { throw new RuntimeException("Stub!"); }
}

