/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.net.wifi;

import android.content.Context;
import java.util.List;
import android.net.wifi.hotspot2.PasspointConfiguration;
import java.util.concurrent.Executor;
import android.os.Handler;
import java.net.InetAddress;
import java.util.Set;
import java.util.function.BiConsumer;
import android.os.Parcelable;

/**
 * This class provides the primary API for managing all aspects of Wi-Fi
 * connectivity.
 * <p>
 * On releases before {@link android.os.Build.VERSION_CODES#N}, this object
 * should only be obtained from an {@linkplain android.content.Context#getApplicationContext()  application context}, and not from any other derived context to avoid memory
 * leaks within the calling process.
 * <p>
 * It deals with several categories of items:
 * </p>
 * <ul>
 * <li>The list of configured networks. The list can be viewed and updated, and
 * attributes of individual entries can be modified.</li>
 * <li>The currently active Wi-Fi network, if any. Connectivity can be
 * established or torn down, and dynamic information about the state of the
 * network can be queried.</li>
 * <li>Results of access point scans, containing enough information to make
 * decisions about what access point to connect to.</li>
 * <li>It defines the names of various Intent actions that are broadcast upon
 * any sort of change in Wi-Fi state.
 * </ul>
 * <p>
 * This is the API to use when performing Wi-Fi specific operations. To perform
 * operations that pertain to network connectivity at an abstract level, use
 * {@link android.net.ConnectivityManager}.
 * </p>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class WifiManager {

WifiManager() { throw new RuntimeException("Stub!"); }

/**
 * Return a list of all the networks configured for the current foreground
 * user.
 *
 * Not all fields of WifiConfiguration are returned. Only the following
 * fields are filled in:
 * <ul>
 * <li>networkId</li>
 * <li>SSID</li>
 * <li>BSSID</li>
 * <li>priority</li>
 * <li>allowedProtocols</li>
 * <li>allowedKeyManagement</li>
 * <li>allowedAuthAlgorithms</li>
 * <li>allowedPairwiseCiphers</li>
 * <li>allowedGroupCiphers</li>
 * <li>status</li>
 * </ul>
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION} and {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @return a list of network configurations in the form of a list
 * of {@link android.net.wifi.WifiConfiguration WifiConfiguration} objects.
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return an
 * empty list.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps will have access to the full list.
 * <li>Callers with Carrier privilege will receive a restricted list only containing
 * configurations which they created.
 * </ul>
 */

@Deprecated
public java.util.List<android.net.wifi.WifiConfiguration> getConfiguredNetworks() { throw new RuntimeException("Stub!"); }

/**
 * Return a list of all the networks previously configured by the calling app. Can
 * be called by Device Owner (DO), Profile Owner (PO), Callers with Carrier privilege and
 * system apps.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @return a list of network configurations in the form of a list
 * of {@link android.net.wifi.WifiConfiguration WifiConfiguration} objects.
 * This value cannot be {@code null}.
 * @throws {@link java.lang.SecurityException} if the caller is not allowed to call this API
 */

@android.annotation.NonNull
public java.util.List<android.net.wifi.WifiConfiguration> getCallerConfiguredNetworks() { throw new RuntimeException("Stub!"); }

/**
 * Add a new network description to the set of configured networks.
 * The {@code networkId} field of the supplied configuration object
 * is ignored.
 * <p/>
 * The new network will be marked DISABLED by default. To enable it,
 * called {@link #enableNetwork}.
 *
 * @param config the set of variables that describe the configuration,
 *            contained in a {@link android.net.wifi.WifiConfiguration WifiConfiguration} object.
 *            If the {@link android.net.wifi.WifiConfiguration WifiConfiguration} has an Http Proxy set
 *            the calling app must be System, or be provisioned as the Profile or Device Owner.
 * @return the ID of the newly created network description. This is used in
 *         other operations to specified the network to be acted upon.
 *         Returns {@code -1} on failure.
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code -1}.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 */

@Deprecated
public int addNetwork(android.net.wifi.WifiConfiguration config) { throw new RuntimeException("Stub!"); }

/**
 * This is a new version of {@link #addNetwork(android.net.wifi.WifiConfiguration)} which returns more detailed
 * failure codes. The usage of this API is limited to Device Owner (DO), Profile Owner (PO),
 * system app, and privileged apps.
 * <p>
 * Add a new network description to the set of configured networks. The {@code networkId}
 * field of the supplied configuration object is ignored. The new network will be marked
 * DISABLED by default. To enable it, call {@link #enableNetwork}.
 * <p>
 * @param config the set of variables that describe the configuration,
 *            contained in a {@link android.net.wifi.WifiConfiguration WifiConfiguration} object.
 *            If the {@link android.net.wifi.WifiConfiguration WifiConfiguration} has an Http Proxy set
 *            the calling app must be System, or be provisioned as the Profile or Device Owner.
 * This value cannot be {@code null}.
 * @return A {@link android.net.wifi.WifiManager.AddNetworkResult AddNetworkResult} Object.
 * This value cannot be {@code null}.
 * @throws {@link java.lang.SecurityException SecurityException} if the calling app is not a Device Owner (DO),
 *                           Profile Owner (PO), system app, or a privileged app that has one of
 *                           the permissions required by this API.
 * @throws {@link java.lang.IllegalArgumentException IllegalArgumentException} if the input configuration is null or if the
 *            security type in input configuration is not supported.
 */

@android.annotation.NonNull
public android.net.wifi.WifiManager.AddNetworkResult addNetworkPrivileged(@android.annotation.NonNull android.net.wifi.WifiConfiguration config) { throw new RuntimeException("Stub!"); }

/**
 * Update the network description of an existing configured network.
 *
 * @param config the set of variables that describe the configuration,
 *            contained in a {@link android.net.wifi.WifiConfiguration WifiConfiguration} object. It may
 *            be sparse, so that only the items that are being changed
 *            are non-<code>null</code>. The {@code networkId} field
 *            must be set to the ID of the existing network being updated.
 *            If the {@link android.net.wifi.WifiConfiguration WifiConfiguration} has an Http Proxy set
 *            the calling app must be System, or be provisioned as the Profile or Device Owner.
 * @return Returns the {@code networkId} of the supplied
 *         {@code WifiConfiguration} on success.
 *         <br/>
 *         Returns {@code -1} on failure, including when the {@code networkId}
 *         field of the {@code WifiConfiguration} does not refer to an
 *         existing network.
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code -1}.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 */

@Deprecated
public int updateNetwork(android.net.wifi.WifiConfiguration config) { throw new RuntimeException("Stub!"); }

/**
 * Provide a list of network suggestions to the device. See {@link android.net.wifi.WifiNetworkSuggestion WifiNetworkSuggestion}
 * for a detailed explanation of the parameters.
 * When the device decides to connect to one of the provided network suggestions, platform sends
 * a directed broadcast {@link #ACTION_WIFI_NETWORK_SUGGESTION_POST_CONNECTION} to the app if
 * the network was created with
 * {@link android.net.wifi.WifiNetworkSuggestion.Builder#setIsAppInteractionRequired(boolean) WifiNetworkSuggestion.Builder#setIsAppInteractionRequired(boolean)} flag set and the
 * app holds {@link android.Manifest.permission#ACCESS_FINE_LOCATION ACCESS_FINE_LOCATION}
 * permission.
 *<p>
 * NOTE:
 * <ul>
 * <li> These networks are just a suggestion to the platform. The platform will ultimately
 * decide on which network the device connects to. </li>
 * <li> When an app is uninstalled or disabled, all its suggested networks are discarded.
 * If the device is currently connected to a suggested network which is being removed then the
 * device will disconnect from that network.</li>
 * <li> If user reset network settings, all added suggestions will be discarded. Apps can use
 * {@link #getNetworkSuggestions()} to check if their suggestions are in the device.</li>
 * <li> In-place modification of existing suggestions are allowed.</li>
 * <ul>
 * <li> If the provided suggestions include any previously provided suggestions by the app,
 * previous suggestions will be updated.</li>
 * <li>If one of the provided suggestions marks a previously unmetered suggestion as metered and
 * the device is currently connected to that suggested network, then the device will disconnect
 * from that network. The system will immediately re-evaluate all the network candidates
 * and possibly reconnect back to the same suggestion. This disconnect is to make sure that any
 * traffic flowing over unmetered networks isn't accidentally continued over a metered network.
 * </li>
 * <li>
 * On {@link android.os.Build.VERSION_CODES#TIRAMISU} or above If one of the provided
 * suggestions marks a previously trusted suggestion as untrusted and the device is currently
 * connected to that suggested network, then the device will disconnect from that network. The
 * system will immediately re-evaluate all the network candidates. This disconnect is to make
 * sure device will not remain connected to an untrusted network without a related
 * {@link android.net.NetworkRequest}.
 * </li>
 * </ul>
 * </ul>
 *
 * <br>
 * Requires {@link android.Manifest.permission#CHANGE_WIFI_STATE}
 * @param networkSuggestions List of network suggestions provided by the app.
 * This value cannot be {@code null}.
 * @return Status code for the operation. One of the STATUS_NETWORK_SUGGESTIONS_ values.
 * Value is {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_SUCCESS}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_INTERNAL}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_APP_DISALLOWED}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_DUPLICATE}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_EXCEEDS_MAX_PER_APP}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_REMOVE_INVALID}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_NOT_ALLOWED}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_INVALID}, or {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_RESTRICTED_BY_ADMIN}
 * @throws {@link java.lang.SecurityException SecurityException} if the caller is missing required permissions.
 * @see android.net.wifi.WifiNetworkSuggestion#equals(Object)
 */

public int addNetworkSuggestions(@android.annotation.NonNull java.util.List<android.net.wifi.WifiNetworkSuggestion> networkSuggestions) { throw new RuntimeException("Stub!"); }

/**
 * Remove some or all of the network suggestions that were previously provided by the app.
 * If one of the suggestions being removed was used to establish connection to the current
 * network, then the device will immediately disconnect from that network. This method is same
 * as {@link #removeNetworkSuggestions(java.util.List,int)} with
 * {@link #ACTION_REMOVE_SUGGESTION_DISCONNECT}
 *
 * See {@link android.net.wifi.WifiNetworkSuggestion WifiNetworkSuggestion} for a detailed explanation of the parameters.
 * See {@link android.net.wifi.WifiNetworkSuggestion#equals(java.lang.Object) WifiNetworkSuggestion#equals(Object)} for the equivalence evaluation used.
 * <p></
 * Note: Use {@link #removeNetworkSuggestions(java.util.List,int)}. An {@code action} of
 * {@link #ACTION_REMOVE_SUGGESTION_DISCONNECT} is equivalent to the current behavior.
 *
 * <br>
 * Requires {@link android.Manifest.permission#CHANGE_WIFI_STATE}
 * @param networkSuggestions List of network suggestions to be removed. Pass an empty list
 *                           to remove all the previous suggestions provided by the app.
 * This value cannot be {@code null}.
 * @return Status code for the operation. One of the {@code STATUS_NETWORK_SUGGESTIONS_*}
 * values. Any matching suggestions are removed from the device and will not be considered for
 * any further connection attempts.
 
 * Value is {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_SUCCESS}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_INTERNAL}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_APP_DISALLOWED}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_DUPLICATE}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_EXCEEDS_MAX_PER_APP}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_REMOVE_INVALID}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_NOT_ALLOWED}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_INVALID}, or {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_RESTRICTED_BY_ADMIN}
 */

public int removeNetworkSuggestions(@android.annotation.NonNull java.util.List<android.net.wifi.WifiNetworkSuggestion> networkSuggestions) { throw new RuntimeException("Stub!"); }

/**
 * Remove some or all of the network suggestions that were previously provided by the app.
 * If one of the suggestions being removed was used to establish connection to the current
 * network, then the specified action will be executed.
 *
 * See {@link android.net.wifi.WifiNetworkSuggestion WifiNetworkSuggestion} for a detailed explanation of the parameters.
 * See {@link android.net.wifi.WifiNetworkSuggestion#equals(java.lang.Object) WifiNetworkSuggestion#equals(Object)} for the equivalence evaluation used.
 *
 * <br>
 * Requires {@link android.Manifest.permission#CHANGE_WIFI_STATE}
 * @param networkSuggestions List of network suggestions to be removed. Pass an empty list
 *                           to remove all the previous suggestions provided by the app.
 * This value cannot be {@code null}.
 * @param action Desired action to execute after removing the suggestion. One of
 *               {@code ACTION_REMOVE_SUGGESTION_*}
 * Value is {@link android.net.wifi.WifiManager#ACTION_REMOVE_SUGGESTION_LINGER}, or {@link android.net.wifi.WifiManager#ACTION_REMOVE_SUGGESTION_DISCONNECT}
 * @return Status code for the operation. One of the {@code STATUS_NETWORK_SUGGESTIONS_*}
 * values. Any matching suggestions are removed from the device and will not be considered for
 * further connection attempts.
 
 * Value is {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_SUCCESS}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_INTERNAL}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_APP_DISALLOWED}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_DUPLICATE}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_EXCEEDS_MAX_PER_APP}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_REMOVE_INVALID}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_NOT_ALLOWED}, {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_INVALID}, or {@link android.net.wifi.WifiManager#STATUS_NETWORK_SUGGESTIONS_ERROR_RESTRICTED_BY_ADMIN}
 */

public int removeNetworkSuggestions(@android.annotation.NonNull java.util.List<android.net.wifi.WifiNetworkSuggestion> networkSuggestions, int action) { throw new RuntimeException("Stub!"); }

/**
 * Get all network suggestions provided by the calling app.
 * See {@link #addNetworkSuggestions(java.util.List)}
 * See {@link #removeNetworkSuggestions(java.util.List)}
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @return a list of {@link android.net.wifi.WifiNetworkSuggestion WifiNetworkSuggestion}

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.List<android.net.wifi.WifiNetworkSuggestion> getNetworkSuggestions() { throw new RuntimeException("Stub!"); }

/**
 * Returns the max number of network suggestions that are allowed per app on the device.
 * @see #addNetworkSuggestions(List)
 * @see #removeNetworkSuggestions(List)
 */

public int getMaxNumberOfNetworkSuggestionsPerApp() { throw new RuntimeException("Stub!"); }

/**
 * Add or update a Passpoint configuration.  The configuration provides a credential
 * for connecting to Passpoint networks that are operated by the Passpoint
 * service provider specified in the configuration.
 *
 * Each configuration is uniquely identified by a unique key which depends on the contents of
 * the configuration. This allows the caller to install multiple profiles with the same FQDN
 * (Fully qualified domain name). Therefore, in order to update an existing profile, it is
 * first required to remove it using {@link android.net.wifi.WifiManager#removePasspointConfiguration(java.lang.String) WifiManager#removePasspointConfiguration(String)}.
 * Otherwise, a new profile will be added with both configuration.
 *
 * Deprecated for general app usage - except DO/PO apps.
 * See {@link android.net.wifi.WifiNetworkSuggestion.Builder#setPasspointConfig(android.net.wifi.hotspot2.PasspointConfiguration) WifiNetworkSuggestion.Builder#setPasspointConfig(PasspointConfiguration)} to
 * create a passpoint suggestion.
 * See {@link #addNetworkSuggestions(java.util.List)}, {@link #removeNetworkSuggestions(java.util.List)} for new
 * API to add Wi-Fi networks for consideration when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#R} or above, this API will always fail and throw
 * {@link java.lang.IllegalArgumentException IllegalArgumentException}.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 *
 * @param config The Passpoint configuration to be added
 * @throws java.lang.IllegalArgumentException if configuration is invalid or Passpoint is not enabled on
 *                                  the device.
 */

public void addOrUpdatePasspointConfiguration(android.net.wifi.hotspot2.PasspointConfiguration config) { throw new RuntimeException("Stub!"); }

/**
 * Remove the Passpoint configuration identified by its FQDN (Fully Qualified Domain Name) added
 * by the caller.
 *
 * @param fqdn The FQDN of the Passpoint configuration added by the caller to be removed
 * @throws java.lang.IllegalArgumentException if no configuration is associated with the given FQDN or
 *                                  Passpoint is not enabled on the device.
 * @deprecated This will be non-functional in a future release.
 * <br>
 * Requires {@code android.Manifest.permission.NETWORK_SETTINGS} or
 * {@code android.Manifest.permission.NETWORK_CARRIER_PROVISIONING}.
 */

@Deprecated
public void removePasspointConfiguration(java.lang.String fqdn) { throw new RuntimeException("Stub!"); }

/**
 * Return the list of installed Passpoint configurations added by the caller.
 *
 * An empty list will be returned when no configurations are installed.
 *
 * @return A list of {@link android.net.wifi.hotspot2.PasspointConfiguration PasspointConfiguration} added by the caller
 * @deprecated This will be non-functional in a future release.
 * <br>
 * Requires {@code android.Manifest.permission.NETWORK_SETTINGS} or
 * {@code android.Manifest.permission.NETWORK_SETUP_WIZARD}.
 */

@Deprecated
public java.util.List<android.net.wifi.hotspot2.PasspointConfiguration> getPasspointConfigurations() { throw new RuntimeException("Stub!"); }

/**
 * Remove the specified network from the list of configured networks.
 * This may result in the asynchronous delivery of state change
 * events.
 *
 * Applications are not allowed to remove networks created by other
 * applications.
 *
 * @param netId the ID of the network as returned by {@link #addNetwork} or {@link
 *        #getConfiguredNetworks}.
 * @return {@code true} if the operation succeeded
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code false}.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 */

@Deprecated
public boolean removeNetwork(int netId) { throw new RuntimeException("Stub!"); }

/**
 * Remove all configured networks that were not created by the calling app. Can only
 * be called by a Device Owner (DO) app.
 *
 * <br>
 * Requires {@link android.Manifest.permission#CHANGE_WIFI_STATE}
 * @return {@code true} if at least one network is removed, {@code false} otherwise
 * @throws {@link java.lang.SecurityException} if the caller is not a Device Owner app
 */

public boolean removeNonCallerConfiguredNetworks() { throw new RuntimeException("Stub!"); }

/**
 * Allow a previously configured network to be associated with. If
 * <code>attemptConnect</code> is true, an attempt to connect to the selected
 * network is initiated. This may result in the asynchronous delivery
 * of state change events.
 * <p>
 * <b>Note:</b> Network communication may not use Wi-Fi even if Wi-Fi is connected;
 * traffic may instead be sent through another network, such as cellular data,
 * Bluetooth tethering, or Ethernet. For example, traffic will never use a
 * Wi-Fi network that does not provide Internet access (e.g. a wireless
 * printer), if another network that does offer Internet access (e.g.
 * cellular data) is available. Applications that need to ensure that their
 * network traffic uses Wi-Fi should use APIs such as
 * {@link android.net.Network#bindSocket(java.net.Socket) Network#bindSocket(java.net.Socket)},
 * {@link android.net.Network#openConnection(java.net.URL) Network#openConnection(java.net.URL)}, or
 * {@link android.net.ConnectivityManager#bindProcessToNetwork ConnectivityManager#bindProcessToNetwork} to do so.
 *
 * Applications are not allowed to enable networks created by other
 * applications.
 *
 * @param netId the ID of the network as returned by {@link #addNetwork} or {@link
 *        #getConfiguredNetworks}.
 * @param attemptConnect The way to select a particular network to connect to is specify
 *        {@code true} for this parameter.
 * @return {@code true} if the operation succeeded
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code false}.
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 */

@Deprecated
public boolean enableNetwork(int netId, boolean attemptConnect) { throw new RuntimeException("Stub!"); }

/**
 * Disable a configured network. The specified network will not be
 * a candidate for associating. This may result in the asynchronous
 * delivery of state change events.
 *
 * Applications are not allowed to disable networks created by other
 * applications.
 *
 * @param netId the ID of the network as returned by {@link #addNetwork} or {@link
 *        #getConfiguredNetworks}.
 * @return {@code true} if the operation succeeded
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code false}.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 */

@Deprecated
public boolean disableNetwork(int netId) { throw new RuntimeException("Stub!"); }

/**
 * Disassociate from the currently active access point. This may result
 * in the asynchronous delivery of state change events.
 * @return {@code true} if the operation succeeded
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code false}.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 */

@Deprecated
public boolean disconnect() { throw new RuntimeException("Stub!"); }

/**
 * Reconnect to the currently active access point, if we are currently
 * disconnected. This may result in the asynchronous delivery of state
 * change events.
 * @return {@code true} if the operation succeeded
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code false}.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 */

@Deprecated
public boolean reconnect() { throw new RuntimeException("Stub!"); }

/**
 * Reconnect to the currently active access point, even if we are already
 * connected. This may result in the asynchronous delivery of state
 * change events.
 * @return {@code true} if the operation succeeded
 *
 * @deprecated
 * a) See {@link android.net.wifi.WifiNetworkSpecifier.Builder#build() WifiNetworkSpecifier.Builder#build()} for new
 * mechanism to trigger connection to a Wi-Fi network.
 * b) See {@link #addNetworkSuggestions(java.util.List)},
 * {@link #removeNetworkSuggestions(java.util.List)} for new API to add Wi-Fi networks for consideration
 * when auto-connecting to wifi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always return false.
 */

@Deprecated
public boolean reassociate() { throw new RuntimeException("Stub!"); }

/**
 * Check that the supplicant daemon is responding to requests.
 * @return {@code true} if we were able to communicate with the supplicant and
 * it returned the expected response to the PING message.
 * @deprecated Will return the output of {@link #isWifiEnabled()} instead.
 */

@Deprecated
public boolean pingSupplicant() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this adapter supports WifiP2pManager (Wi-Fi Direct)
 */

public boolean isP2pSupported() { throw new RuntimeException("Stub!"); }

/**
 * Query whether or not the device supports Station (STA) + Access point (AP) concurrency.
 *
 * @return true if this device supports STA + AP concurrency, false otherwise.
 */

public boolean isStaApConcurrencySupported() { throw new RuntimeException("Stub!"); }

/**
 * Query whether or not the device supports concurrent station (STA) connections for local-only
 * connections using {@link android.net.wifi.WifiNetworkSpecifier WifiNetworkSpecifier}.
 *
 * @return true if this device supports multiple STA concurrency for this use-case, false
 * otherwise.
 */

public boolean isStaConcurrencyForLocalOnlyConnectionsSupported() { throw new RuntimeException("Stub!"); }

/**
 * Query whether or not the device supports concurrent station (STA) connections for
 * make-before-break wifi to wifi switching.
 *
 * Note: This is an internal feature which is not available to apps.
 *
 * @return true if this device supports multiple STA concurrency for this use-case, false
 * otherwise.
 */

public boolean isMakeBeforeBreakWifiSwitchingSupported() { throw new RuntimeException("Stub!"); }

/**
 * Query whether or not the device supports concurrent station (STA) connections for multi
 * internet connections.
 *
 * @return true if this device supports multiple STA concurrency for this use-case, false
 * otherwise.
 */

public boolean isStaConcurrencyForMultiInternetSupported() { throw new RuntimeException("Stub!"); }

/**
 * @deprecated Please use {@link android.content.pm.PackageManager#hasSystemFeature(String)}
 * with {@link android.content.pm.PackageManager#FEATURE_WIFI_RTT}.
 *
 * @return true if this adapter supports Device-to-AP RTT
 */

@Deprecated
public boolean isDeviceToApRttSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this adapter supports offloaded connectivity scan
 */

public boolean isPreferredNetworkOffloadSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this adapter supports Tunnel Directed Link Setup
 */

public boolean isTdlsSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this adapter supports advanced power/performance counters
 */

public boolean isEnhancedPowerReportingSupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if the chipset supports 2.4GHz band.
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean is24GHzBandSupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if the chipset supports 5GHz band.
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean is5GHzBandSupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if the chipset supports the 60GHz frequency band.
 *
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean is60GHzBandSupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if the chipset supports 6GHz band.
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean is6GHzBandSupported() { throw new RuntimeException("Stub!"); }

/**
 * Check if the chipset supports a certain Wi-Fi standard.
 * @param standard the IEEE 802.11 standard to check on.
 *        valid values from {@link android.net.wifi.ScanResult ScanResult}'s {@code WIFI_STANDARD_}
 * Value is {@link android.net.wifi.ScanResult#WIFI_STANDARD_UNKNOWN}, {@link android.net.wifi.ScanResult#WIFI_STANDARD_LEGACY}, {@link android.net.wifi.ScanResult#WIFI_STANDARD_11N}, {@link android.net.wifi.ScanResult#WIFI_STANDARD_11AC}, {@link android.net.wifi.ScanResult#WIFI_STANDARD_11AX}, {@link android.net.wifi.ScanResult#WIFI_STANDARD_11AD}, or {@link android.net.wifi.ScanResult#WIFI_STANDARD_11BE}
 * @return {@code true} if supported, {@code false} otherwise.
 */

public boolean isWifiStandardSupported(int standard) { throw new RuntimeException("Stub!"); }

/**
 * Query whether or not the device supports concurrency of Station (STA) + multiple access
 * points (AP) (where the APs bridged together).
 *
 * See {@link android.net.wifi.SoftApConfiguration.Builder#setBands(int[]) SoftApConfiguration.Builder#setBands(int[])}
 * or {@link android.net.wifi.SoftApConfiguration.Builder#setChannels(android.util.SparseIntArray) SoftApConfiguration.Builder#setChannels(android.util.SparseIntArray)} to configure
 * bridged AP when the bridged AP supported.
 *
 * @return true if this device supports concurrency of STA + multiple APs which are bridged
 *         together, false otherwise.
 */

public boolean isStaBridgedApConcurrencySupported() { throw new RuntimeException("Stub!"); }

/**
 * Query whether or not the device supports multiple Access point (AP) which are bridged
 * together.
 *
 * See {@link android.net.wifi.SoftApConfiguration.Builder#setBands(int[]) SoftApConfiguration.Builder#setBands(int[])}
 * or {@link android.net.wifi.SoftApConfiguration.Builder#setChannels(android.util.SparseIntArray) SoftApConfiguration.Builder#setChannels(android.util.SparseIntArray)} to configure
 * bridged AP when the bridged AP supported.
 *
 * @return true if this device supports concurrency of multiple AP which bridged together,
 *         false otherwise.
 */

public boolean isBridgedApConcurrencySupported() { throw new RuntimeException("Stub!"); }

/**
 * Request a scan for access points. Returns immediately. The availability
 * of the results is made known later by means of an asynchronous event sent
 * on completion of the scan.
 * <p>
 * To initiate a Wi-Fi scan, declare the
 * {@link android.Manifest.permission#CHANGE_WIFI_STATE}
 * permission in the manifest, and perform these steps:
 * </p>
 * <ol style="1">
 * <li>Invoke the following method:
 * {@code ((WifiManager) getSystemService(WIFI_SERVICE)).startScan()}</li>
 * <li>
 * Register a BroadcastReceiver to listen to
 * {@code SCAN_RESULTS_AVAILABLE_ACTION}.</li>
 * <li>When a broadcast is received, call:
 * {@code ((WifiManager) getSystemService(WIFI_SERVICE)).getScanResults()}</li>
 * </ol>
 * @return {@code true} if the operation succeeded, i.e., the scan was initiated.
 * @deprecated The ability for apps to trigger scan requests will be removed in a future
 * release.
 */

@Deprecated
public boolean startScan() { throw new RuntimeException("Stub!"); }

/**
 * Return dynamic information about the current Wi-Fi connection, if any is active.
 * <p>
 *
 * @return the Wi-Fi information, contained in {@link android.net.wifi.WifiInfo WifiInfo}.
 *
 * @deprecated Starting with {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES#S}, WifiInfo retrieval is moved to
 * {@link android.net.ConnectivityManager ConnectivityManager} API surface. WifiInfo is attached in
 * {@link android.net.NetworkCapabilities#getTransportInfo() NetworkCapabilities#getTransportInfo()} which is available via callback in
 * {@link android.net.ConnectivityManager.NetworkCallback#onCapabilitiesChanged(android.net.Network,android.net.NetworkCapabilities) NetworkCallback#onCapabilitiesChanged(Network, NetworkCapabilities)} or on-demand from
 * {@link android.net.ConnectivityManager#getNetworkCapabilities(android.net.Network) ConnectivityManager#getNetworkCapabilities(Network)}.
 *
 *</p>
 * Usage example:
 * <pre>
 * final NetworkRequest request =
 *      new NetworkRequest.Builder()
 *      .addTransportType(NetworkCapabilities.TRANSPORT_WIFI)
 *      .build();
 * final ConnectivityManager connectivityManager =
 *      context.getSystemService(ConnectivityManager.class);
 * final NetworkCallback networkCallback = new NetworkCallback() {
 *      ...
 *      &#64;Override
 *      void onAvailable(Network network) {}
 *
 *      &#64;Override
 *      void onCapabilitiesChanged(Network network, NetworkCapabilities networkCapabilities) {
 *          WifiInfo wifiInfo = (WifiInfo) networkCapabilities.getTransportInfo();
 *      }
 *      // etc.
 * };
 * connectivityManager.requestNetwork(request, networkCallback); // For request
 * connectivityManager.registerNetworkCallback(request, networkCallback); // For listen
 * </pre>
 * <p>
 * <b>Compatibility Notes:</b>
 * <li>Apps can continue using this API, however newer features
 * such as ability to mask out location sensitive data in WifiInfo will not be supported
 * via this API. </li>
 * <li>On devices supporting concurrent connections (indicated via
 * {@link #isStaConcurrencyForLocalOnlyConnectionsSupported()}, etc) this API will return
 * the details of the internet providing connection (if any) to all apps, except for the apps
 * that triggered the creation of the concurrent connection. For such apps, this API will return
 * the details of the connection they created. e.g. apps using {@link android.net.wifi.WifiNetworkSpecifier WifiNetworkSpecifier} will
 * trigger a concurrent connection on supported devices and hence this API will provide
 * details of their peer to peer connection (not the internet providing connection). This
 * is to maintain backwards compatibility with behavior on single STA devices.</li>
 * </p>
 */

@Deprecated
public android.net.wifi.WifiInfo getConnectionInfo() { throw new RuntimeException("Stub!"); }

/**
 * Return the results of the latest access point scan.
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE} and {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * @return the list of access points found in the most recent scan. An app must hold
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION ACCESS_FINE_LOCATION} permission
 * and {@link android.Manifest.permission#ACCESS_WIFI_STATE} permission
 * in order to get valid results.
 */

public java.util.List<android.net.wifi.ScanResult> getScanResults() { throw new RuntimeException("Stub!"); }

/**
 * Check if scanning is always available.
 *
 * If this return {@code true}, apps can issue {@link #startScan} and fetch scan results
 * even when Wi-Fi is turned off.
 *
 * To change this setting, see {@link #ACTION_REQUEST_SCAN_ALWAYS_AVAILABLE}.
 * @deprecated The ability for apps to trigger scan requests will be removed in a future
 * release.
 */

@Deprecated
public boolean isScanAlwaysAvailable() { throw new RuntimeException("Stub!"); }

/**
 * Tell the device to persist the current list of configured networks.
 * <p>
 * Note: It is possible for this method to change the network IDs of
 * existing networks. You should assume the network IDs can be different
 * after calling this method.
 *
 * @return {@code false}.
 * @deprecated There is no need to call this method -
 * {@link #addNetwork(android.net.wifi.WifiConfiguration)}, {@link #updateNetwork(android.net.wifi.WifiConfiguration)}
 * and {@link #removeNetwork(int)} already persist the configurations automatically.
 */

@Deprecated
public boolean saveConfiguration() { throw new RuntimeException("Stub!"); }

/**
 * Return the DHCP-assigned addresses from the last successful DHCP request,
 * if any.
 *
 * @return the DHCP information
 *
 * @deprecated Use the methods on {@link android.net.LinkProperties} which can be obtained
 * either via {@link android.net.ConnectivityManager.NetworkCallback#onLinkPropertiesChanged(android.net.Network,android.net.LinkProperties) NetworkCallback#onLinkPropertiesChanged(Network, LinkProperties)} or
 * {@link android.net.ConnectivityManager#getLinkProperties(android.net.Network) ConnectivityManager#getLinkProperties(Network)}.
 *
 * <p>
 * <b>Compatibility Notes:</b>
 * <li>On devices supporting concurrent connections (indicated via
 * {@link #isStaConcurrencyForLocalOnlyConnectionsSupported()}, etc), this API will return
 * the details of the internet providing connection (if any) to all apps, except for the apps
 * that triggered the creation of the concurrent connection. For such apps, this API will return
 * the details of the connection they created. e.g. apps using {@link android.net.wifi.WifiNetworkSpecifier WifiNetworkSpecifier} will
 * trigger a concurrent connection on supported devices and hence this API will provide
 * details of their peer to peer connection (not the internet providing connection). This
 * is to maintain backwards compatibility with behavior on single STA devices.</li>
 * </p>
 */

@Deprecated
public android.net.DhcpInfo getDhcpInfo() { throw new RuntimeException("Stub!"); }

/**
 * Enable or disable Wi-Fi.
 * <p>
 * Applications must have the {@link android.Manifest.permission#CHANGE_WIFI_STATE}
 * permission to toggle wifi.
 *
 * @param enabled {@code true} to enable, {@code false} to disable.
 * @return {@code false} if the request cannot be satisfied; {@code true} indicates that wifi is
 *         either already in the requested state, or in progress toward the requested state.
 * @throws  {@link java.lang.SecurityException} if the caller is missing required permissions.
 *
 * @deprecated Starting with Build.VERSION_CODES#Q, applications are not allowed to
 * enable/disable Wi-Fi.
 * <b>Compatibility Note:</b> For applications targeting
 * {@link android.os.Build.VERSION_CODES#Q} or above, this API will always fail and return
 * {@code false}. If apps are targeting an older SDK ({@link android.os.Build.VERSION_CODES#P}
 * or below), they can continue to use this API.
 * <p>
 * Deprecation Exemptions:
 * <ul>
 * <li>Device Owner (DO), Profile Owner (PO) and system apps.
 * </ul>
 *
 * Starting with {@link android.os.Build.VERSION_CODES#TIRAMISU}, DO/COPE may set
 * a user restriction (DISALLOW_CHANGE_WIFI_STATE) to only allow DO/PO to use this API.
 */

@Deprecated
public boolean setWifiEnabled(boolean enabled) { throw new RuntimeException("Stub!"); }

/**
 * Registers a {@link android.net.wifi.WifiManager.SubsystemRestartTrackingCallback SubsystemRestartTrackingCallback} to listen to Wi-Fi subsystem restarts.
 * The subsystem may restart due to internal recovery mechanisms or via user action.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @see #unregisterSubsystemRestartTrackingCallback(SubsystemRestartTrackingCallback)
 *
 * @param executor Executor to execute callback on
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback {@link android.net.wifi.WifiManager.SubsystemRestartTrackingCallback SubsystemRestartTrackingCallback} to register

 * This value cannot be {@code null}.
 */

public void registerSubsystemRestartTrackingCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.wifi.WifiManager.SubsystemRestartTrackingCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Unregisters a {@link android.net.wifi.WifiManager.SubsystemRestartTrackingCallback SubsystemRestartTrackingCallback} registered with
 * {@link #registerSubsystemRestartTrackingCallback(java.util.concurrent.Executor,android.net.wifi.WifiManager.SubsystemRestartTrackingCallback)}
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param callback {@link android.net.wifi.WifiManager.SubsystemRestartTrackingCallback SubsystemRestartTrackingCallback} to unregister

 * This value cannot be {@code null}.
 */

public void unregisterSubsystemRestartTrackingCallback(@android.annotation.NonNull android.net.wifi.WifiManager.SubsystemRestartTrackingCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Gets the Wi-Fi enabled state.
 * @return One of {@link #WIFI_STATE_DISABLED},
 *         {@link #WIFI_STATE_DISABLING}, {@link #WIFI_STATE_ENABLED},
 *         {@link #WIFI_STATE_ENABLING}, {@link #WIFI_STATE_UNKNOWN}
 * @see #isWifiEnabled()
 */

public int getWifiState() { throw new RuntimeException("Stub!"); }

/**
 * Return whether Wi-Fi is enabled or disabled.
 * @return {@code true} if Wi-Fi is enabled
 * @see #getWifiState()
 */

public boolean isWifiEnabled() { throw new RuntimeException("Stub!"); }

/**
 * Calculates the level of the signal. This should be used any time a signal
 * is being shown.
 *
 * @param rssi The power of the signal measured in RSSI.
 * @param numLevels The number of levels to consider in the calculated level.
 * @return A level of the signal, given in the range of 0 to numLevels-1 (both inclusive).
 * @deprecated Callers should use {@link #calculateSignalLevel(int)} instead to get the
 * signal level using the system default RSSI thresholds, or otherwise compute the RSSI level
 * themselves using their own formula.
 */

@Deprecated
public static int calculateSignalLevel(int rssi, int numLevels) { throw new RuntimeException("Stub!"); }

/**
 * Given a raw RSSI, return the RSSI signal quality rating using the system default RSSI
 * quality rating thresholds.
 * @param rssi a raw RSSI value, in dBm, usually between -55 and -90
 * @return the RSSI signal quality rating, in the range
 * [0, {@link #getMaxSignalLevel()}], where 0 is the lowest (worst signal) RSSI
 * rating and {@link #getMaxSignalLevel()} is the highest (best signal) RSSI rating.

 * Value is 0 or greater
 */

public int calculateSignalLevel(int rssi) { throw new RuntimeException("Stub!"); }

/**
 * Get the system default maximum signal level.
 * This is the maximum RSSI level returned by {@link #calculateSignalLevel(int)}.

 * @return Value is 0 or greater
 */

public int getMaxSignalLevel() { throw new RuntimeException("Stub!"); }

/**
 * Compares two signal strengths.
 *
 * @param rssiA The power of the first signal measured in RSSI.
 * @param rssiB The power of the second signal measured in RSSI.
 * @return Returns <0 if the first signal is weaker than the second signal,
 *         0 if the two signals have the same strength, and >0 if the first
 *         signal is stronger than the second signal.
 */

public static int compareSignalLevel(int rssiA, int rssiB) { throw new RuntimeException("Stub!"); }

/**
 * Request a local only hotspot that an application can use to communicate between co-located
 * devices connected to the created WiFi hotspot.  The network created by this method will not
 * have Internet access.  Each application can make a single request for the hotspot, but
 * multiple applications could be requesting the hotspot at the same time.  When multiple
 * applications have successfully registered concurrently, they will be sharing the underlying
 * hotspot. {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#onStarted(android.net.wifi.WifiManager.LocalOnlyHotspotReservation) LocalOnlyHotspotCallback#onStarted(LocalOnlyHotspotReservation)} is called
 * when the hotspot is ready for use by the application.
 * <p>
 * Each application can make a single active call to this method. The {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#onStarted(android.net.wifi.WifiManager.LocalOnlyHotspotReservation)  } callback supplies the
 * requestor with a {@link android.net.wifi.WifiManager.LocalOnlyHotspotReservation LocalOnlyHotspotReservation} that contains a
 * {@link android.net.wifi.SoftApConfiguration SoftApConfiguration} with the SSID, security type and credentials needed to connect
 * to the hotspot.  Communicating this information is up to the application.
 * <p>
 * If the LocalOnlyHotspot cannot be created, the {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#onFailed(int) LocalOnlyHotspotCallback#onFailed(int)}
 * method will be called. Example failures include errors bringing up the network or if
 * there is an incompatible operating mode.  For example, if the user is currently using Wifi
 * Tethering to provide an upstream to another device, LocalOnlyHotspot may not start due to
 * an incompatible mode. The possible error codes include:
 * {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#ERROR_NO_CHANNEL LocalOnlyHotspotCallback#ERROR_NO_CHANNEL},
 * {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#ERROR_GENERIC LocalOnlyHotspotCallback#ERROR_GENERIC},
 * {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#ERROR_INCOMPATIBLE_MODE LocalOnlyHotspotCallback#ERROR_INCOMPATIBLE_MODE} and
 * {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#ERROR_TETHERING_DISALLOWED LocalOnlyHotspotCallback#ERROR_TETHERING_DISALLOWED}.
 * <p>
 * Internally, requests will be tracked to prevent the hotspot from being torn down while apps
 * are still using it.  The {@link android.net.wifi.WifiManager.LocalOnlyHotspotReservation LocalOnlyHotspotReservation} object passed in the  {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#onStarted(android.net.wifi.WifiManager.LocalOnlyHotspotReservation)  } call should be closed when
 * the LocalOnlyHotspot is no longer needed using {@link android.net.wifi.WifiManager.LocalOnlyHotspotReservation#close() LocalOnlyHotspotReservation#close()}.
 * Since the hotspot may be shared among multiple applications, removing the final registered
 * application request will trigger the hotspot teardown.  This means that applications should
 * not listen to broadcasts containing wifi state to determine if the hotspot was stopped after
 * they are done using it. Additionally, once {@link android.net.wifi.WifiManager.LocalOnlyHotspotReservation#close() LocalOnlyHotspotReservation#close()} is
 * called, applications will not receive callbacks of any kind.
 * <p>
 * Applications should be aware that the user may also stop the LocalOnlyHotspot through the
 * Settings UI; it is not guaranteed to stay up as long as there is a requesting application.
 * The requestors will be notified of this case via
 * {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#onStopped() LocalOnlyHotspotCallback#onStopped()}.  Other cases may arise where the hotspot is
 * torn down (Emergency mode, etc).  Application developers should be aware that it can stop
 * unexpectedly, but they will receive a notification if they have properly registered.
 * <p>
 * Applications should also be aware that this network will be shared with other applications.
 * Applications are responsible for protecting their data on this network (e.g. TLS).
 * <p>
 * Applications targeting {@link android.os.Build.VERSION_CODES#TIRAMISU} or later need to have
 * the following permissions: {@link android.Manifest.permission#CHANGE_WIFI_STATE} and
 * {@link android.Manifest.permission#NEARBY_WIFI_DEVICES}.
 * Applications targeting {@link android.os.Build.VERSION_CODES#S Build.VERSION_CODES#S} or prior SDK levels need to have the
 * following permissions: {@link android.Manifest.permission#CHANGE_WIFI_STATE} and
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION}
 * Callers without
 * the permissions will trigger a {@link java.lang.SecurityException}.
 * <p>
 * @param callback LocalOnlyHotspotCallback for the application to receive updates about
 * operating status.
 * @param handler Handler to be used for callbacks.  If the caller passes a null Handler, the
 * main thread will be used.
 */

public void startLocalOnlyHotspot(android.net.wifi.WifiManager.LocalOnlyHotspotCallback callback, @android.annotation.Nullable android.os.Handler handler) { throw new RuntimeException("Stub!"); }

/**
 * Enable/Disable TDLS on a specific local route.
 *
 * <p>
 * TDLS enables two wireless endpoints to talk to each other directly
 * without going through the access point that is managing the local
 * network. It saves bandwidth and improves quality of the link.
 * </p>
 * <p>
 * This API enables/disables the option of using TDLS. If enabled, the
 * underlying hardware is free to use TDLS or a hop through the access
 * point. If disabled, existing TDLS session is torn down and
 * hardware is restricted to use access point for transferring wireless
 * packets. Default value for all routes is 'disabled', meaning restricted
 * to use access point for transferring packets.
 * </p>
 *
 * @param remoteIPAddress IP address of the endpoint to setup TDLS with
 * @param enable true = setup and false = tear down TDLS
 */

public void setTdlsEnabled(java.net.InetAddress remoteIPAddress, boolean enable) { throw new RuntimeException("Stub!"); }

/**
 * Similar to {@link #setTdlsEnabled(java.net.InetAddress,boolean) }, except
 * this version allows you to specify remote endpoint with a MAC address.
 * @param remoteMacAddress MAC address of the remote endpoint such as 00:00:0c:9f:f2:ab
 * @param enable true = setup and false = tear down TDLS
 */

public void setTdlsEnabledWithMacAddress(java.lang.String remoteMacAddress, boolean enable) { throw new RuntimeException("Stub!"); }

/**
 * Control whether the device will automatically search for and connect to Wi-Fi networks -
 * auto-join Wi-Fi networks. Disabling this option will not impact manual connections - i.e.
 * the user will still be able to manually select and connect to a Wi-Fi network. Disabling
 * this option significantly impacts the device connectivity and is a restricted operation
 * (see below for permissions). Note that disabling this operation will also disable
 * connectivity initiated scanning operations.
 * <p>
 * Disabling the auto-join configuration is a temporary operation (with the exception of a
 * DO/PO caller): it will be reset (to enabled) when the device reboots or the user toggles
 * Wi-Fi off/on. When the caller is a DO/PO then toggling Wi-Fi will not reset the
 * configuration. Additionally, if a DO/PO disables auto-join then it cannot be (re)enabled by
 * a non-DO/PO caller.
 *
 * @param allowAutojoin true to allow auto-join, false to disallow auto-join
 *
 * Available for DO/PO apps.
 * Other apps require {@code android.Manifest.permission#NETWORK_SETTINGS} or
 * {@code android.Manifest.permission#MANAGE_WIFI_NETWORK_SELECTION} permission.
 */

public void allowAutojoinGlobal(boolean allowAutojoin) { throw new RuntimeException("Stub!"); }

/**
 * Query whether or not auto-join global is enabled/disabled
 * @see #allowAutojoinGlobal(boolean)
 *
 * Available for DO/PO apps.
 * Other apps require {@code android.Manifest.permission#NETWORK_SETTINGS} or
 * {@code android.Manifest.permission#MANAGE_WIFI_NETWORK_SELECTION} permission.
 *
 * @param executor The executor on which callback will be invoked.
 * This value cannot be {@code null}.
 * @param resultsCallback An asynchronous callback that will return {@code Boolean} indicating
 *                        whether auto-join global is enabled/disabled.
 *
 * This value cannot be {@code null}.
 * @throws java.lang.SecurityException if the caller does not have permission.
 * @throws java.lang.NullPointerException if the caller provided invalid inputs.
 */

public void queryAutojoinGlobal(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.Consumer<java.lang.Boolean> resultsCallback) { throw new RuntimeException("Stub!"); }

/**
 * WPS suport has been deprecated from Client mode and this method will immediately trigger
 * {@link android.net.wifi.WifiManager.WpsCallback#onFailed(int) WpsCallback#onFailed(int)} with a generic error.
 *
 * @param config WPS configuration (does not support {@link android.net.wifi.WpsInfo#LABEL WpsInfo#LABEL})
 * @param listener for callbacks on success or failure. Can be null.
 * @throws java.lang.IllegalStateException if the WifiManager instance needs to be initialized again
 * @deprecated This API is deprecated
 */

@Deprecated
public void startWps(android.net.wifi.WpsInfo config, android.net.wifi.WifiManager.WpsCallback listener) { throw new RuntimeException("Stub!"); }

/**
 * WPS support has been deprecated from Client mode and this method will immediately trigger
 * {@link android.net.wifi.WifiManager.WpsCallback#onFailed(int) WpsCallback#onFailed(int)} with a generic error.
 *
 * @param listener for callbacks on success or failure. Can be null.
 * @throws java.lang.IllegalStateException if the WifiManager instance needs to be initialized again
 * @deprecated This API is deprecated
 */

@Deprecated
public void cancelWps(android.net.wifi.WifiManager.WpsCallback listener) { throw new RuntimeException("Stub!"); }

/**
 * Creates a new WifiLock.
 *
 * @param lockType the type of lock to create. See {@link #WIFI_MODE_FULL_HIGH_PERF}
 * and {@link #WIFI_MODE_FULL_LOW_LATENCY} for descriptions of the types of Wi-Fi locks.
 * @param tag a tag for the WifiLock to identify it in debugging messages.  This string is
 *            never shown to the user under normal conditions, but should be descriptive
 *            enough to identify your application and the specific WifiLock within it, if it
 *            holds multiple WifiLocks.
 *
 * @return a new, unacquired WifiLock with the given tag.
 *
 * @see android.net.wifi.WifiManager.WifiLock
 */

public android.net.wifi.WifiManager.WifiLock createWifiLock(int lockType, java.lang.String tag) { throw new RuntimeException("Stub!"); }

/**
 * Creates a new WifiLock.
 *
 * @param tag a tag for the WifiLock to identify it in debugging messages.  This string is
 *            never shown to the user under normal conditions, but should be descriptive
 *            enough to identify your application and the specific WifiLock within it, if it
 *            holds multiple WifiLocks.
 *
 * @return a new, unacquired WifiLock with the given tag.
 *
 * @see android.net.wifi.WifiManager.WifiLock
 *
 * @deprecated This API is non-functional.
 */

@Deprecated
public android.net.wifi.WifiManager.WifiLock createWifiLock(java.lang.String tag) { throw new RuntimeException("Stub!"); }

/**
 * Create a new MulticastLock
 *
 * @param tag a tag for the MulticastLock to identify it in debugging
 *            messages.  This string is never shown to the user under
 *            normal conditions, but should be descriptive enough to
 *            identify your application and the specific MulticastLock
 *            within it, if it holds multiple MulticastLocks.
 *
 * @return a new, unacquired MulticastLock with the given tag.
 *
 * @see android.net.wifi.WifiManager.MulticastLock
 */

public android.net.wifi.WifiManager.MulticastLock createMulticastLock(java.lang.String tag) { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports WPA3-Personal SAE
 */

public boolean isWpa3SaeSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports WPA3-Enterprise Suite-B-192
 */

public boolean isWpa3SuiteBSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports Wi-Fi Enhanced Open (OWE)
 */

public boolean isEnhancedOpenSupported() { throw new RuntimeException("Stub!"); }

/**
 * Wi-Fi Easy Connect (DPP) introduces standardized mechanisms to simplify the provisioning and
 * configuration of Wi-Fi devices.
 * For more details, visit <a href="https://www.wi-fi.org/">https://www.wi-fi.org/</a> and
 * search for "Easy Connect" or "Device Provisioning Protocol specification".
 *
 * @return true if this device supports Wi-Fi Easy-connect (Device Provisioning Protocol)
 */

public boolean isEasyConnectSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports Wi-Fi Easy Connect (DPP) Enrollee Responder mode.
 */

public boolean isEasyConnectEnrolleeResponderModeSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports WAPI.
 */

public boolean isWapiSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports WPA3 SAE Public Key.
 */

public boolean isWpa3SaePublicKeySupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports Wi-Fi Passpoint Terms and Conditions feature.
 */

public boolean isPasspointTermsAndConditionsSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports WPA3 SAE Hash-to-Element.
 */

public boolean isWpa3SaeH2eSupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports Wi-Fi Display R2.
 */

public boolean isWifiDisplayR2Supported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports RFC 7542 decorated identity.
 */

public boolean isDecoratedIdentitySupported() { throw new RuntimeException("Stub!"); }

/**
 * @return true if this device supports Trust On First Use (TOFU).
 */

public boolean isTrustOnFirstUseSupported() { throw new RuntimeException("Stub!"); }

/**
 * Wi-Fi Easy Connect DPP AKM enables provisioning and configuration of Wi-Fi devices without
 * the need of using the device PSK passphrase.
 * For more details, visit <a href="https://www.wi-fi.org/">https://www.wi-fi.org/</a> and
 * search for "Easy Connect" or "Device Provisioning Protocol specification".
 *
 * @return true if this device supports Wi-Fi Easy-connect DPP (Device Provisioning Protocol)
 * AKM, false otherwise.
 */

public boolean isEasyConnectDppAkmSupported() { throw new RuntimeException("Stub!"); }

/**
 * Register a callback for Scan Results. See {@link android.net.wifi.WifiManager.ScanResultsCallback ScanResultsCallback}.
 * Caller will receive the event when scan results are available.
 * Caller should use {@link android.net.wifi.WifiManager#getScanResults() WifiManager#getScanResults()} requires
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} to get the scan results.
 * Caller can remove a previously registered callback using
 * {@link android.net.wifi.WifiManager#unregisterScanResultsCallback(android.net.wifi.WifiManager.ScanResultsCallback) WifiManager#unregisterScanResultsCallback(ScanResultsCallback)}
 * Same caller can add multiple listeners.
 * <p>
 * Applications should have the
 * {@link android.Manifest.permission#ACCESS_WIFI_STATE} permission. Callers
 * without the permission will trigger a {@link java.lang.SecurityException}.
 * <p>
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param executor The executor to execute the callback of the {@code callback} object.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback callback for Scan Results events

 * This value cannot be {@code null}.
 */

public void registerScanResultsCallback(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.wifi.WifiManager.ScanResultsCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Allow callers to unregister a previously registered callback. After calling this method,
 * applications will no longer receive Scan Results events.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param callback callback to unregister for Scan Results events

 * This value cannot be {@code null}.
 */

public void unregisterScanResultsCallback(@android.annotation.NonNull android.net.wifi.WifiManager.ScanResultsCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Add a listener for suggestion networks. See {@link android.net.wifi.WifiManager.SuggestionConnectionStatusListener SuggestionConnectionStatusListener}.
 * Caller will receive the event when suggested network have connection failure.
 * Caller can remove a previously registered listener using
 * {@link android.net.wifi.WifiManager#removeSuggestionConnectionStatusListener(android.net.wifi.WifiManager.SuggestionConnectionStatusListener) WifiManager#removeSuggestionConnectionStatusListener(
 * SuggestionConnectionStatusListener)}
 * Same caller can add multiple listeners to monitor the event.
 * <p>
 * Applications should have the
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION} and
 * {@link android.Manifest.permission#ACCESS_WIFI_STATE} permissions.
 * Callers without the permission will trigger a {@link java.lang.SecurityException}.
 * <p>
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_FINE_LOCATION} and {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param executor The executor to execute the listener of the {@code listener} object.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener listener for suggestion network connection failure.

 * This value cannot be {@code null}.
 */

public void addSuggestionConnectionStatusListener(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.wifi.WifiManager.SuggestionConnectionStatusListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Allow callers to remove a previously registered listener. After calling this method,
 * applications will no longer receive suggestion connection events through that listener.
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param listener listener to remove.

 * This value cannot be {@code null}.
 */

public void removeSuggestionConnectionStatusListener(@android.annotation.NonNull android.net.wifi.WifiManager.SuggestionConnectionStatusListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Get the persisted Wi-Fi scan throttle state. Defaults to true, unless changed by the user via
 * Developer options.
 *
 * <p>
 * The throttling limits for apps are described in
 * <a href="Wi-Fi Scan Throttling">
 * https://developer.android.com/guide/topics/connectivity/wifi-scan#wifi-scan-throttling</a>
 * </p>
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @return true to indicate that scan throttling is enabled, false to indicate that scan
 * throttling is disabled.
 */

public boolean isScanThrottleEnabled() { throw new RuntimeException("Stub!"); }

/**
 * Get the persisted Wi-Fi auto wakeup feature state. Defaults to false, unless changed by the
 * user via Settings.
 *
 * <p>
 * The feature is described in
 * <a href="Wi-Fi Turn on automatically">
 * https://source.android.com/devices/tech/connect/wifi-infrastructure
 * #turn_on_wi-fi_automatically
 * </a>
 *
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @return true to indicate that wakeup feature is enabled, false to indicate that wakeup
 * feature is disabled.
 */

public boolean isAutoWakeupEnabled() { throw new RuntimeException("Stub!"); }

/**
 * Get the carrier network offload state for merged or unmerged networks for specified
 * subscription.
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param subscriptionId subscription ID see {@link android.telephony.SubscriptionInfo#getSubscriptionId() SubscriptionInfo#getSubscriptionId()}
 * @param merged True for carrier merged network, false otherwise.
 *               See {@link android.net.wifi.WifiNetworkSuggestion.Builder#setCarrierMerged(boolean) WifiNetworkSuggestion.Builder#setCarrierMerged(boolean)}
 * @return True to indicate that carrier network offload is enabled, false otherwise.
 */

public boolean isCarrierNetworkOffloadEnabled(int subscriptionId, boolean merged) { throw new RuntimeException("Stub!"); }

/**
 * Add a listener for Wi-Fi network suggestion user approval status.
 * See {@link android.net.wifi.WifiManager.SuggestionUserApprovalStatusListener SuggestionUserApprovalStatusListener}.
 * Caller will receive a callback immediately after adding a listener and when the user approval
 * status of the caller has changed.
 * Caller can remove a previously registered listener using
 * {@link android.net.wifi.WifiManager#removeSuggestionUserApprovalStatusListener(android.net.wifi.WifiManager.SuggestionUserApprovalStatusListener) WifiManager#removeSuggestionUserApprovalStatusListener(
 * SuggestionUserApprovalStatusListener)}
 * A caller can add multiple listeners to monitor the event.
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param executor The executor to execute the listener of the {@code listener} object.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param listener listener for suggestion user approval status changes.

 * This value cannot be {@code null}.
 */

public void addSuggestionUserApprovalStatusListener(@android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.net.wifi.WifiManager.SuggestionUserApprovalStatusListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Allow callers to remove a previously registered listener using
 * {@link #addSuggestionUserApprovalStatusListener(java.util.concurrent.Executor,android.net.wifi.WifiManager.SuggestionUserApprovalStatusListener)}. After calling this method,
 * applications will no longer receive network suggestion user approval status change through
 * that listener.
 
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}

 * @param listener This value cannot be {@code null}.
 */

public void removeSuggestionUserApprovalStatusListener(@android.annotation.NonNull android.net.wifi.WifiManager.SuggestionUserApprovalStatusListener listener) { throw new RuntimeException("Stub!"); }

/**
 * Flush Passpoint ANQP cache, and clear pending ANQP requests. Allows the caller to reset the
 * Passpoint ANQP state, if required.
 *
 * Notes:
 * 1. Flushing the ANQP cache may cause delays in discovering and connecting to Passpoint
 * networks.
 * 2. Intended to be used by Device Owner (DO), Profile Owner (PO), Settings and provisioning
 * apps.
 */

public void flushPasspointAnqpCache() { throw new RuntimeException("Stub!"); }

/**
 * If the device supports Wi-Fi Passpoint, the user can explicitly enable or disable it.
 * That status can be queried using this method.
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @return {@code true} if Wi-Fi Passpoint is enabled
 *
 */

public boolean isWifiPasspointEnabled() { throw new RuntimeException("Stub!"); }

/**
 * The device may support concurrent connections to multiple internet-providing Wi-Fi
 * networks (APs) - that is indicated by
 * {@link android.net.wifi.WifiManager#isStaConcurrencyForMultiInternetSupported() WifiManager#isStaConcurrencyForMultiInternetSupported()}.
 * This method indicates whether or not the feature is currently enabled.
 * A value of {@link android.net.wifi.WifiManager#WIFI_MULTI_INTERNET_MODE_DISABLED WifiManager#WIFI_MULTI_INTERNET_MODE_DISABLED} indicates that the feature
 * is disabled, a value of {@link android.net.wifi.WifiManager#WIFI_MULTI_INTERNET_MODE_DBS_AP WifiManager#WIFI_MULTI_INTERNET_MODE_DBS_AP} or
 * {@link android.net.wifi.WifiManager#WIFI_MULTI_INTERNET_MODE_MULTI_AP WifiManager#WIFI_MULTI_INTERNET_MODE_MULTI_AP} indicates that the feature is enabled.
 *
 * The app can register to receive the corresponding Wi-Fi networks using the
 * {@link android.net.ConnectivityManager#registerNetworkCallback(android.net.NetworkRequest,android.net.ConnectivityManager.NetworkCallback) ConnectivityManager#registerNetworkCallback(NetworkRequest, NetworkCallback)} API with
 * a {@link android.net.wifi.WifiNetworkSpecifier WifiNetworkSpecifier} configured using the
 * {@link android.net.wifi.WifiNetworkSpecifier.Builder#setBand(int) WifiNetworkSpecifier.Builder#setBand(int)} method.
 
 * <br>
 * Requires {@link android.Manifest.permission#ACCESS_WIFI_STATE}

 * @return Value is {@link android.net.wifi.WifiManager#WIFI_MULTI_INTERNET_MODE_DISABLED}, {@link android.net.wifi.WifiManager#WIFI_MULTI_INTERNET_MODE_DBS_AP}, or {@link android.net.wifi.WifiManager#WIFI_MULTI_INTERNET_MODE_MULTI_AP}
 */

public int getStaConcurrencyForMultiInternetMode() { throw new RuntimeException("Stub!"); }

/**
 * Queries the framework to determine whether the specified interface can be created, and if
 * so - what other interfaces would be torn down by the framework to allow this creation if
 * it were requested. The result is returned via the specified {@link java.util.function.BiConsumer BiConsumer} callback
 * which returns two arguments:
 * <li>First argument: a {@code boolean} - indicating whether or not the interface can be
 * created.</li>
 * <li>Second argument: a {@code Set<InterfaceCreationImpact>} - if the interface can be
 * created (first argument is {@code true} then this is the set of interface types which
 * will be removed and the packages which requested them. Possibly an empty set. If the
 * first argument is {@code false}, then an empty set will be returned here.</li>
 * <p>
 * Interfaces, input and output, are specified using the {@code WIFI_INTERFACE_*} constants:
 * {@link #WIFI_INTERFACE_TYPE_STA}, {@link #WIFI_INTERFACE_TYPE_AP},
 * {@link #WIFI_INTERFACE_TYPE_AWARE}, or {@link #WIFI_INTERFACE_TYPE_DIRECT}.
 * <p>
 * This method does not actually create the interface. That operation is handled by the
 * framework when a particular service method is called. E.g. a Wi-Fi Direct interface may be
 * created when various methods of {@link android.net.wifi.p2p.WifiP2pManager} are called,
 * similarly for Wi-Fi Aware and {@link android.net.wifi.aware.WifiAwareManager}.
 * <p>
 * Note: the information returned via this method is the current snapshot of the system. It may
 * change due to actions of the framework or other apps.
 *
 * <br>
 * Requires {@link android.Manifest.permission#MANAGE_WIFI_INTERFACES} and {@link android.Manifest.permission#ACCESS_WIFI_STATE}
 * @param interfaceType The interface type whose possible creation is being queried.
 * Value is {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_STA}, {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_AP}, {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_AWARE}, or {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_DIRECT}
 * @param requireNewInterface Indicates that the query is for a new interface of the specified
 *                             type - an existing interface won't meet the query. Some
 *                             operations (such as Wi-Fi Direct and Wi-Fi Aware are a shared
 *                             resource and so may not need a new interface).
 * @param executor An {@link java.util.concurrent.Executor Executor} on which to return the result.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param resultCallback The asynchronous callback which will return two argument: a
 * {@code boolean} (whether the interface can be created), and a
 * {@code Set<InterfaceCreationImpact>} (a set of {@link android.net.wifi.WifiManager.InterfaceCreationImpact InterfaceCreationImpact}:
 *                       interfaces which will be destroyed when the interface is created
 *                       and the packages which requested them and thus may be impacted).

 * This value cannot be {@code null}.
 */

public void reportCreateInterfaceImpact(int interfaceType, boolean requireNewInterface, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull java.util.function.BiConsumer<java.lang.Boolean,java.util.Set<android.net.wifi.WifiManager.InterfaceCreationImpact>> resultCallback) { throw new RuntimeException("Stub!"); }

/**
 * Activity Action: Pick a Wi-Fi network to connect to.
 * <p>Input: Nothing.
 * <p>Output: Nothing.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.ACTIVITY_INTENT_ACTION) public static final java.lang.String ACTION_PICK_WIFI_NETWORK = "android.net.wifi.PICK_WIFI_NETWORK";

/**
 * If one of the removed suggestions is currently connected, trigger an immediate disconnect
 * after suggestions removal
 */

public static final int ACTION_REMOVE_SUGGESTION_DISCONNECT = 2; // 0x2

/**
 * If one of the removed suggestions is currently connected, that network will be disconnected
 * after a short delay as opposed to immediately (which will be done by
 * {@link #ACTION_REMOVE_SUGGESTION_DISCONNECT}). The {@link android.net.ConnectivityManager ConnectivityManager} may call the
 * {@link android.net.ConnectivityManager.NetworkCallback#onLosing(android.net.Network,int) NetworkCallback#onLosing(Network, int)} on such networks.
 */

public static final int ACTION_REMOVE_SUGGESTION_LINGER = 1; // 0x1

/**
 * Activity Action: Show a system activity that allows the user to enable
 * scans to be available even with Wi-Fi turned off.
 *
 * <p>Notification of the result of this activity is posted using the
 * {@link android.app.Activity#onActivityResult} callback. The
 * <code>resultCode</code>
 * will be {@link android.app.Activity#RESULT_OK} if scan always mode has
 * been turned on or {@link android.app.Activity#RESULT_CANCELED} if the user
 * has rejected the request or an error has occurred.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.ACTIVITY_INTENT_ACTION) public static final java.lang.String ACTION_REQUEST_SCAN_ALWAYS_AVAILABLE = "android.net.wifi.action.REQUEST_SCAN_ALWAYS_AVAILABLE";

/**
 * Directed broadcast intent action indicating that the device has connected to one of the
 * network suggestions provided by the app. This will be sent post connection to a network
 * which was created with {@link android.net.wifi.WifiNetworkSuggestion.Builder#setIsAppInteractionRequired(boolean) WifiNetworkSuggestion.Builder#setIsAppInteractionRequired(
 * boolean)}
 * flag set.
 * <p>
 * Note: The broadcast is sent to the app only if it holds
 * {@link android.Manifest.permission#ACCESS_FINE_LOCATION ACCESS_FINE_LOCATION} permission.
 *
 * @see #EXTRA_NETWORK_SUGGESTION
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String ACTION_WIFI_NETWORK_SUGGESTION_POST_CONNECTION = "android.net.wifi.action.WIFI_NETWORK_SUGGESTION_POST_CONNECTION";

/**
 * Broadcast intent action indicating whether Wi-Fi scanning is currently available.
 * Available extras:
 * - {@link #EXTRA_SCAN_AVAILABLE}
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String ACTION_WIFI_SCAN_AVAILABILITY_CHANGED = "android.net.wifi.action.WIFI_SCAN_AVAILABILITY_CHANGED";

/**
 * The error code if there was a problem authenticating.
 * @deprecated This is no longer supported.
 */

@Deprecated public static final int ERROR_AUTHENTICATING = 1; // 0x1

/**
 * The lookup key for a String giving the BSSID of the access point to which
 * we are connected. No longer used.
 */

@Deprecated public static final java.lang.String EXTRA_BSSID = "bssid";

/**
 * The lookup key for a {@link android.net.NetworkInfo} object associated with the
 * Wi-Fi network. Retrieve with
 * {@link android.content.Intent#getParcelableExtra(String)}.
 */

public static final java.lang.String EXTRA_NETWORK_INFO = "networkInfo";

/**
 * Sent as as a part of {@link #ACTION_WIFI_NETWORK_SUGGESTION_POST_CONNECTION} that holds
 * an instance of {@link android.net.wifi.WifiNetworkSuggestion WifiNetworkSuggestion} corresponding to the connected network.
 */

public static final java.lang.String EXTRA_NETWORK_SUGGESTION = "android.net.wifi.extra.NETWORK_SUGGESTION";

/**
 * The lookup key for an {@code int} giving the new RSSI in dBm.
 */

public static final java.lang.String EXTRA_NEW_RSSI = "newRssi";

/**
 * The lookup key for a {@link android.net.wifi.SupplicantState SupplicantState} describing the new state
 * Retrieve with
 * {@link android.content.Intent#getParcelableExtra(String)}.
 * @deprecated This is no longer supported.
 */

@Deprecated public static final java.lang.String EXTRA_NEW_STATE = "newState";

/**
 * The previous Wi-Fi state.
 *
 * @see #EXTRA_WIFI_STATE
 */

public static final java.lang.String EXTRA_PREVIOUS_WIFI_STATE = "previous_wifi_state";

/**
 * Lookup key for a {@code boolean} extra in intent {@link #SCAN_RESULTS_AVAILABLE_ACTION}
 * representing if the scan was successful or not.
 * Scans may fail for multiple reasons, these may include:
 * <ol>
 * <li>An app requested too many scans in a certain period of time.
 * This may lead to additional scan request rejections via "scan throttling" for both
 * foreground and background apps.
 * Note: Apps holding android.Manifest.permission.NETWORK_SETTINGS permission are
 * exempted from scan throttling.
 * </li>
 * <li>The device is idle and scanning is disabled.</li>
 * <li>Wifi hardware reported a scan failure.</li>
 * </ol>
 * @return true scan was successful, results are updated
 * @return false scan was not successful, results haven't been updated since previous scan
 */

public static final java.lang.String EXTRA_RESULTS_UPDATED = "resultsUpdated";

/**
 * A boolean extra indicating whether scanning is currently available.
 * Sent in the broadcast {@link #ACTION_WIFI_SCAN_AVAILABILITY_CHANGED}.
 * Its value is true if scanning is currently available, false otherwise.
 */

public static final java.lang.String EXTRA_SCAN_AVAILABLE = "android.net.wifi.extra.SCAN_AVAILABLE";

/**
 * The lookup key for a boolean that indicates whether a connection to
 * the supplicant daemon has been gained or lost. {@code true} means
 * a connection now exists.
 * Retrieve it with {@link android.content.Intent#getBooleanExtra(String,boolean)}.
 * @deprecated This is no longer supported.
 */

@Deprecated public static final java.lang.String EXTRA_SUPPLICANT_CONNECTED = "connected";

/**
 * The lookup key for a {@link android.net.wifi.SupplicantState SupplicantState} describing the supplicant
 * error code if any
 * Retrieve with
 * {@link android.content.Intent#getIntExtra(String, int)}.
 * @see #ERROR_AUTHENTICATING
 * @deprecated This is no longer supported.
 */

@Deprecated public static final java.lang.String EXTRA_SUPPLICANT_ERROR = "supplicantError";

/**
 * The lookup key for a {@link android.net.wifi.WifiInfo} object giving the
 * information about the access point to which we are connected.
 * No longer used.
 */

@Deprecated public static final java.lang.String EXTRA_WIFI_INFO = "wifiInfo";

/**
 * The lookup key for an int that indicates whether Wi-Fi is enabled,
 * disabled, enabling, disabling, or unknown.  Retrieve it with
 * {@link android.content.Intent#getIntExtra(String,int)}.
 *
 * @see #WIFI_STATE_DISABLED
 * @see #WIFI_STATE_DISABLING
 * @see #WIFI_STATE_ENABLED
 * @see #WIFI_STATE_ENABLING
 * @see #WIFI_STATE_UNKNOWN
 */

public static final java.lang.String EXTRA_WIFI_STATE = "wifi_state";

/**
 * The network IDs of the configured networks could have changed.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String NETWORK_IDS_CHANGED_ACTION = "android.net.wifi.NETWORK_IDS_CHANGED";

/**
 * Broadcast intent action indicating that the state of Wi-Fi connectivity
 * has changed. An extra provides the new state
 * in the form of a {@link android.net.NetworkInfo} object.  No network-related
 * permissions are required to subscribe to this broadcast.
 *
 * <p class="note">This broadcast is not delivered to manifest receivers in
 * applications that target API version 26 or later.
 * @see #EXTRA_NETWORK_INFO
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String NETWORK_STATE_CHANGED_ACTION = "android.net.wifi.STATE_CHANGE";

/**
 * The RSSI (signal strength) has changed.
 *
 * Receiver Required Permission: android.Manifest.permission.ACCESS_WIFI_STATE
 * @see #EXTRA_NEW_RSSI
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String RSSI_CHANGED_ACTION = "android.net.wifi.RSSI_CHANGED";

/**
 * An access point scan has completed, and results are available.
 * Call {@link #getScanResults()} to obtain the results.
 * The broadcast intent may contain an extra field with the key {@link #EXTRA_RESULTS_UPDATED}
 * and a {@code boolean} value indicating if the scan was successful.
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String SCAN_RESULTS_AVAILABLE_ACTION = "android.net.wifi.SCAN_RESULTS";

/**
 * Reason code if one or more of the network suggestions added already exists in platform's
 * database.
 * Note: this code will not be returned with Android 11 as in-place modification is allowed,
 * please check {@link #addNetworkSuggestions(java.util.List)}.
 * @see android.net.wifi.WifiNetworkSuggestion#equals(Object)
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_DUPLICATE = 3; // 0x3

/**
 * Reason code if the number of network suggestions provided by the app crosses the max
 * threshold set per app.
 * The framework will reject all suggestions provided by {@link #addNetworkSuggestions(java.util.List)} if
 * the total size exceeds the limit.
 * @see #getMaxNumberOfNetworkSuggestionsPerApp()
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_EXCEEDS_MAX_PER_APP = 4; // 0x4

/**
 * Reason code if one or more of the network suggestions added is invalid. Framework will reject
 * all the suggestions in the list.
 * The framework will reject all suggestions provided by {@link #addNetworkSuggestions(java.util.List)}
 * if one or more of them is invalid.
 * Please use {@link android.net.wifi.WifiNetworkSuggestion.Builder WifiNetworkSuggestion.Builder} to create network suggestions.
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_INVALID = 7; // 0x7

/**
 * Reason code if one or more of the network suggestions added is not allowed.
 * The framework will reject all suggestions provided by {@link #addNetworkSuggestions(java.util.List)}
 * if one or more of them is not allowed.
 * This error may be caused by suggestion is using SIM-based encryption method, but calling app
 * is not carrier privileged.
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_ADD_NOT_ALLOWED = 6; // 0x6

/**
 * Reason code if the user has disallowed "android:change_wifi_state" app-ops from the app.
 * @see android.app.AppOpsManager#unsafeCheckOp(String, int, String).
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_APP_DISALLOWED = 2; // 0x2

/**
 * Reason code if there was an internal error in the platform while processing the addition or
 * removal of suggestions.
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_INTERNAL = 1; // 0x1

/**
 * Reason code if one or more of the network suggestions removed does not exist in platform's
 * database.
 * The framework won't remove any suggestions if one or more of suggestions provided
 * by {@link #removeNetworkSuggestions(java.util.List)} does not exist in database.
 * @see android.net.wifi.WifiNetworkSuggestion#equals(Object)
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_REMOVE_INVALID = 5; // 0x5

/**
 * Reason code if {@link android.os.UserManager#DISALLOW_ADD_WIFI_CONFIG} user restriction
 * is set and calling app is restricted by device admin.
 */

public static final int STATUS_NETWORK_SUGGESTIONS_ERROR_RESTRICTED_BY_ADMIN = 8; // 0x8

/**
 * Reason code if all of the network suggestions were successfully added or removed.
 */

public static final int STATUS_NETWORK_SUGGESTIONS_SUCCESS = 0; // 0x0

/**
 * Status code if the calling app was approved by virtue of being a carrier privileged app.
 * @see android.telephony.TelephonyManager#hasCarrierPrivileges().
 */

public static final int STATUS_SUGGESTION_APPROVAL_APPROVED_BY_CARRIER_PRIVILEGE = 4; // 0x4

/**
 * Status code if the calling app got the user approval for suggestions.
 */

public static final int STATUS_SUGGESTION_APPROVAL_APPROVED_BY_USER = 2; // 0x2

/**
 * Status code if the calling app is still pending user approval for suggestions.
 */

public static final int STATUS_SUGGESTION_APPROVAL_PENDING = 1; // 0x1

/**
 * Status code if the calling app suggestions were rejected by the user.
 */

public static final int STATUS_SUGGESTION_APPROVAL_REJECTED_BY_USER = 3; // 0x3

/**
 * Status code if suggestion approval status is unknown, an App which hasn't made any
 * suggestions will get this code.
 */

public static final int STATUS_SUGGESTION_APPROVAL_UNKNOWN = 0; // 0x0

/**
 * Reason code if suggested network connection attempt failed with association failure.
 */

public static final int STATUS_SUGGESTION_CONNECTION_FAILURE_ASSOCIATION = 1; // 0x1

/**
 * Reason code if suggested network connection attempt failed with an authentication failure.
 */

public static final int STATUS_SUGGESTION_CONNECTION_FAILURE_AUTHENTICATION = 2; // 0x2

/**
 * Reason code if suggested network connection attempt failed with an IP provision failure.
 */

public static final int STATUS_SUGGESTION_CONNECTION_FAILURE_IP_PROVISIONING = 3; // 0x3

/**
 * Reason code if suggested network connection attempt failed with an unknown failure.
 */

public static final int STATUS_SUGGESTION_CONNECTION_FAILURE_UNKNOWN = 0; // 0x0

/**
 * Broadcast intent action indicating that a connection to the supplicant has
 * been established (and it is now possible
 * to perform Wi-Fi operations) or the connection to the supplicant has been
 * lost. One extra provides the connection state as a boolean, where {@code true}
 * means CONNECTED.
 * @deprecated This is no longer supported.
 * @see #EXTRA_SUPPLICANT_CONNECTED
 */

@Deprecated @android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String SUPPLICANT_CONNECTION_CHANGE_ACTION = "android.net.wifi.supplicant.CONNECTION_CHANGE";

/**
 * Broadcast intent action indicating that the state of establishing a connection to
 * an access point has changed.One extra provides the new
 * {@link android.net.wifi.SupplicantState SupplicantState}. Note that the supplicant state is Wi-Fi specific, and
 * is not generally the most useful thing to look at if you are just interested in
 * the overall state of connectivity.
 * @see #EXTRA_NEW_STATE
 * @see #EXTRA_SUPPLICANT_ERROR
 * @deprecated This is no longer supported.
 */

@Deprecated @android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String SUPPLICANT_STATE_CHANGED_ACTION = "android.net.wifi.supplicant.STATE_CHANGE";

/** Indicates an invalid SSID. */

public static final java.lang.String UNKNOWN_SSID = "<unknown ssid>";

/**
 * Wi-Fi interface of type AP (access point Wi-Fi infrastructure device).
 */

public static final int WIFI_INTERFACE_TYPE_AP = 1; // 0x1

/**
 * Wi-Fi interface of type Wi-Fi Aware (aka NAN).
 */

public static final int WIFI_INTERFACE_TYPE_AWARE = 2; // 0x2

/**
 * Wi-Fi interface of type Wi-Fi Direct (aka P2P).
 */

public static final int WIFI_INTERFACE_TYPE_DIRECT = 3; // 0x3

/**
 * Wi-Fi interface of type STA (station/client Wi-Fi infrastructure device).
 */

public static final int WIFI_INTERFACE_TYPE_STA = 0; // 0x0

/**
 * In this Wi-Fi lock mode, Wi-Fi will be kept active,
 * and will behave normally, i.e., it will attempt to automatically
 * establish a connection to a remembered access point that is
 * within range, and will do periodic scans if there are remembered
 * access points but none are in range.
 *
 * @deprecated This API is non-functional and will have no impact.
 */

@Deprecated public static final int WIFI_MODE_FULL = 1; // 0x1

/**
 * In this Wi-Fi lock mode, Wi-Fi will not go to power save.
 * This results in operating with low packet latency.
 * The lock is only active when the device is connected to an access point.
 * The lock is active even when the device screen is off or the acquiring application is
 * running in the background.
 * This mode will consume more power and hence should be used only
 * when there is a need for this tradeoff.
 * <p>
 * An example use case is when a voice connection needs to be
 * kept active even after the device screen goes off.
 * Holding a {@link #WIFI_MODE_FULL_HIGH_PERF} lock for the
 * duration of the voice call may improve the call quality.
 * <p>
 * When there is no support from the hardware, the {@link #WIFI_MODE_FULL_HIGH_PERF}
 * lock will have no impact.
 */

public static final int WIFI_MODE_FULL_HIGH_PERF = 3; // 0x3

/**
 * In this Wi-Fi lock mode, Wi-Fi will operate with a priority to achieve low latency.
 * {@link #WIFI_MODE_FULL_LOW_LATENCY} lock has the following limitations:
 * <ol>
 * <li>The lock is only active when the device is connected to an access point.</li>
 * <li>The lock is only active when the screen is on.</li>
 * <li>The lock is only active when the acquiring app is running in the foreground.</li>
 * </ol>
 * Low latency mode optimizes for reduced packet latency,
 * and as a result other performance measures may suffer when there are trade-offs to make:
 * <ol>
 * <li>Battery life may be reduced.</li>
 * <li>Throughput may be reduced.</li>
 * <li>Frequency of Wi-Fi scanning may be reduced. This may result in: </li>
 * <ul>
 * <li>The device may not roam or switch to the AP with highest signal quality.</li>
 * <li>Location accuracy may be reduced.</li>
 * </ul>
 * </ol>
 * <p>
 * Example use cases are real time gaming or virtual reality applications where
 * low latency is a key factor for user experience.
 * <p>
 * Note: For an app which acquires both {@link #WIFI_MODE_FULL_LOW_LATENCY} and
 * {@link #WIFI_MODE_FULL_HIGH_PERF} locks, {@link #WIFI_MODE_FULL_LOW_LATENCY}
 * lock will be effective when app is running in foreground and screen is on,
 * while the {@link #WIFI_MODE_FULL_HIGH_PERF} lock will take effect otherwise.
 */

public static final int WIFI_MODE_FULL_LOW_LATENCY = 4; // 0x4

/**
 * In this Wi-Fi lock mode, Wi-Fi will be kept active,
 * but the only operation that will be supported is initiation of
 * scans, and the subsequent reporting of scan results. No attempts
 * will be made to automatically connect to remembered access points,
 * nor will periodic scans be automatically performed looking for
 * remembered access points. Scans must be explicitly requested by
 * an application in this mode.
 *
 * @deprecated This API is non-functional and will have no impact.
 */

@Deprecated public static final int WIFI_MODE_SCAN_ONLY = 2; // 0x2

/**
 * Wi-Fi simultaneous connection to multiple internet-providing Wi-FI networks (APs) is enabled
 * and restricted to a single network on different bands (e.g.&nbsp;a DBS AP).
 *
 * @see #getStaConcurrencyForMultiInternetMode()
 *
 */

public static final int WIFI_MULTI_INTERNET_MODE_DBS_AP = 1; // 0x1

/**
 * Wi-Fi simultaneous connection to multiple internet-providing Wi-Fi networks (APs) is
 * disabled.
 *
 * @see #getStaConcurrencyForMultiInternetMode()
 *
 */

public static final int WIFI_MULTI_INTERNET_MODE_DISABLED = 0; // 0x0

/**
 * Wi-Fi simultaneous connection to multiple internet-providing Wi-Fi networks (APs) is enabled.
 * The device can connect to any networks/APs - it is just restricted to using different bands
 * for individual connections.
 *
 * @see #getStaConcurrencyForMultiInternetMode()
 *
 */

public static final int WIFI_MULTI_INTERNET_MODE_MULTI_AP = 2; // 0x2

/**
 * Broadcast intent action indicating that Wi-Fi has been enabled, disabled,
 * enabling, disabling, or unknown. One extra provides this state as an int.
 * Another extra provides the previous state, if available.  No network-related
 * permissions are required to subscribe to this broadcast.
 *
 * <p class="note">This broadcast is not delivered to manifest receivers in
 * applications that target API version 26 or later.
 *
 * @see #EXTRA_WIFI_STATE
 * @see #EXTRA_PREVIOUS_WIFI_STATE
 */

@android.annotation.SdkConstant(android.annotation.SdkConstant.SdkConstantType.BROADCAST_INTENT_ACTION) public static final java.lang.String WIFI_STATE_CHANGED_ACTION = "android.net.wifi.WIFI_STATE_CHANGED";

/**
 * Wi-Fi is disabled.
 *
 * @see #WIFI_STATE_CHANGED_ACTION
 * @see #getWifiState()
 */

public static final int WIFI_STATE_DISABLED = 1; // 0x1

/**
 * Wi-Fi is currently being disabled. The state will change to {@link #WIFI_STATE_DISABLED} if
 * it finishes successfully.
 *
 * @see #WIFI_STATE_CHANGED_ACTION
 * @see #getWifiState()
 */

public static final int WIFI_STATE_DISABLING = 0; // 0x0

/**
 * Wi-Fi is enabled.
 *
 * @see #WIFI_STATE_CHANGED_ACTION
 * @see #getWifiState()
 */

public static final int WIFI_STATE_ENABLED = 3; // 0x3

/**
 * Wi-Fi is currently being enabled. The state will change to {@link #WIFI_STATE_ENABLED} if
 * it finishes successfully.
 *
 * @see #WIFI_STATE_CHANGED_ACTION
 * @see #getWifiState()
 */

public static final int WIFI_STATE_ENABLING = 2; // 0x2

/**
 * Wi-Fi is in an unknown state. This state will occur when an error happens while enabling
 * or disabling.
 *
 * @see #WIFI_STATE_CHANGED_ACTION
 * @see #getWifiState()
 */

public static final int WIFI_STATE_UNKNOWN = 4; // 0x4

/** Authentication failure on WPS
 * @deprecated This is deprecated
 */

@Deprecated public static final int WPS_AUTH_FAILURE = 6; // 0x6

/** WPS overlap detected
 * @deprecated This is deprecated
 */

@Deprecated public static final int WPS_OVERLAP_ERROR = 3; // 0x3

/** WPS timed out
 * @deprecated This is deprecated
 */

@Deprecated public static final int WPS_TIMED_OUT = 7; // 0x7

/** TKIP only prohibited
 * @deprecated This is deprecated
 */

@Deprecated public static final int WPS_TKIP_ONLY_PROHIBITED = 5; // 0x5

/** WEP on WPS is prohibited
 * @deprecated This is deprecated
 */

@Deprecated public static final int WPS_WEP_PROHIBITED = 4; // 0x4
/**
 * Provides the results of a call to {@link #addNetworkPrivileged(android.net.wifi.WifiConfiguration)}
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class AddNetworkResult implements android.os.Parcelable {

/**
 * @param statusCode Value is {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_SUCCESS}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_FAILURE_UNKNOWN}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_ADD_PASSPOINT_FAILURE}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_ADD_WIFI_CONFIG_FAILURE}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_INVALID_CONFIGURATION}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION_MODIFY_CONFIG}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION_MODIFY_PROXY_SETTING}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION_MODIFY_MAC_RANDOMIZATION}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_FAILURE_UPDATE_NETWORK_KEYS}, or {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_INVALID_CONFIGURATION_ENTERPRISE}
 */

public AddNetworkResult(int statusCode, int networkId) { throw new RuntimeException("Stub!"); }

public int describeContents() { throw new RuntimeException("Stub!"); }

/**
 * {@inheritDoc}

 * @param dest This value cannot be {@code null}.
 */

public void writeToParcel(@android.annotation.NonNull android.os.Parcel dest, int flags) { throw new RuntimeException("Stub!"); }

/** Implement the Parcelable interface */

@android.annotation.NonNull public static final android.os.Parcelable.Creator<android.net.wifi.WifiManager.AddNetworkResult> CREATOR;
static { CREATOR = null; }

/**
 * Generic failure code for adding a passpoint network.
 */

public static final int STATUS_ADD_PASSPOINT_FAILURE = 3; // 0x3

/**
 * Generic failure code for adding a non-passpoint network.
 */

public static final int STATUS_ADD_WIFI_CONFIG_FAILURE = 4; // 0x4

/**
 * The operation has failed due to an unknown reason.
 */

public static final int STATUS_FAILURE_UNKNOWN = 1; // 0x1

/**
 * Internal failure in updating network keys..
 */

public static final int STATUS_FAILURE_UPDATE_NETWORK_KEYS = 9; // 0x9

/**
 * The network configuration is invalid.
 */

public static final int STATUS_INVALID_CONFIGURATION = 5; // 0x5

/**
 * The enterprise network is missing either the root CA or domain name.
 */

public static final int STATUS_INVALID_CONFIGURATION_ENTERPRISE = 10; // 0xa

/**
 * The calling app does not have permission to call this API.
 */

public static final int STATUS_NO_PERMISSION = 2; // 0x2

/**
 * The calling app has no permission to modify the configuration.
 */

public static final int STATUS_NO_PERMISSION_MODIFY_CONFIG = 6; // 0x6

/**
 * The calling app has no permission to modify the MAC randomization setting.
 */

public static final int STATUS_NO_PERMISSION_MODIFY_MAC_RANDOMIZATION = 8; // 0x8

/**
 * The calling app has no permission to modify the proxy setting.
 */

public static final int STATUS_NO_PERMISSION_MODIFY_PROXY_SETTING = 7; // 0x7

/**
 * The operation has completed successfully.
 */

public static final int STATUS_SUCCESS = 0; // 0x0

/**
 * The identifier of the added network, which could be used in other operations. This field
 * will be set to {@code -1} if the operation failed.
 */

public final int networkId;
{ networkId = 0; }

/**
 * One of the {@code STATUS_} values. If the operation is successful this field
 * will be set to {@code STATUS_SUCCESS}.

 * <br>
 * Value is {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_SUCCESS}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_FAILURE_UNKNOWN}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_ADD_PASSPOINT_FAILURE}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_ADD_WIFI_CONFIG_FAILURE}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_INVALID_CONFIGURATION}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION_MODIFY_CONFIG}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION_MODIFY_PROXY_SETTING}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_NO_PERMISSION_MODIFY_MAC_RANDOMIZATION}, {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_FAILURE_UPDATE_NETWORK_KEYS}, or {@link android.net.wifi.WifiManager.AddNetworkResult#STATUS_INVALID_CONFIGURATION_ENTERPRISE}
 */

public final int statusCode;
{ statusCode = 0; }
}

/**
 * Class describing an impact of interface creation - returned by
 * {@link #reportCreateInterfaceImpact(int,boolean,java.util.concurrent.Executor,java.util.function.BiConsumer)}. Due to Wi-Fi
 * concurrency limitations certain interfaces may have to be torn down. Each of these
 * interfaces was requested by a set of applications who could potentially be impacted.
 *
 * This class contain the information for a single interface: the interface type with
 * {@link android.net.wifi.WifiManager.InterfaceCreationImpact#getInterfaceType() InterfaceCreationImpact#getInterfaceType()} and the set of impacted packages
 * with {@link android.net.wifi.WifiManager.InterfaceCreationImpact#getPackages() InterfaceCreationImpact#getPackages()}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class InterfaceCreationImpact {

/**
 * @param interfaceType Value is {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_STA}, {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_AP}, {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_AWARE}, or {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_DIRECT}

 * @param packages This value cannot be {@code null}.
 */

public InterfaceCreationImpact(int interfaceType, @android.annotation.NonNull java.util.Set<java.lang.String> packages) { throw new RuntimeException("Stub!"); }

/**
 * @return The interface type which will be torn down to make room for the interface
 * requested in {@link #reportCreateInterfaceImpact(int,boolean,java.util.concurrent.Executor,java.util.function.BiConsumer)}.

 * Value is {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_STA}, {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_AP}, {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_AWARE}, or {@link android.net.wifi.WifiManager#WIFI_INTERFACE_TYPE_DIRECT}
 */

public int getInterfaceType() { throw new RuntimeException("Stub!"); }

/**
 * @return The list of potentially impacted packages due to tearing down the interface
 * specified in {@link #getInterfaceType()}.

 * This value cannot be {@code null}.
 */

@android.annotation.NonNull
public java.util.Set<java.lang.String> getPackages() { throw new RuntimeException("Stub!"); }

public int hashCode() { throw new RuntimeException("Stub!"); }

public boolean equals(java.lang.Object that) { throw new RuntimeException("Stub!"); }
}

/**
 * Callback class for applications to receive updates about the LocalOnlyHotspot status.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static class LocalOnlyHotspotCallback {

public LocalOnlyHotspotCallback() { throw new RuntimeException("Stub!"); }

/** LocalOnlyHotspot start succeeded. */

public void onStarted(android.net.wifi.WifiManager.LocalOnlyHotspotReservation reservation) { throw new RuntimeException("Stub!"); }

/**
 * LocalOnlyHotspot stopped.
 * <p>
 * The LocalOnlyHotspot can be disabled at any time by the user.  When this happens,
 * applications will be notified that it was stopped. This will not be invoked when an
 * application calls {@link android.net.wifi.WifiManager.LocalOnlyHotspotReservation#close() LocalOnlyHotspotReservation#close()}.
 */

public void onStopped() { throw new RuntimeException("Stub!"); }

/**
 * LocalOnlyHotspot failed to start.
 * <p>
 * Applications can attempt to call
 * {@link android.net.wifi.WifiManager#startLocalOnlyHotspot(android.net.wifi.WifiManager.LocalOnlyHotspotCallback,android.os.Handler) WifiManager#startLocalOnlyHotspot(LocalOnlyHotspotCallback, Handler)} again at
 * a later time.
 * <p>
 * @param reason The reason for failure could be one of: {@link
 * #ERROR_TETHERING_DISALLOWED}, {@link #ERROR_INCOMPATIBLE_MODE},
 * {@link #ERROR_NO_CHANNEL}, or {@link #ERROR_GENERIC}.
 */

public void onFailed(int reason) { throw new RuntimeException("Stub!"); }

public static final int ERROR_GENERIC = 2; // 0x2

public static final int ERROR_INCOMPATIBLE_MODE = 3; // 0x3

public static final int ERROR_NO_CHANNEL = 1; // 0x1

public static final int ERROR_TETHERING_DISALLOWED = 4; // 0x4
}

/**
 * LocalOnlyHotspotReservation that contains the {@link android.net.wifi.SoftApConfiguration SoftApConfiguration} for the active
 * LocalOnlyHotspot request.
 * <p>
 * Applications requesting LocalOnlyHotspot for sharing will receive an instance of the
 * LocalOnlyHotspotReservation in the
 * {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#onStarted(android.net.wifi.WifiManager.LocalOnlyHotspotReservation) LocalOnlyHotspotCallback#onStarted(LocalOnlyHotspotReservation)} call.  This
 * reservation contains the relevant {@link android.net.wifi.SoftApConfiguration SoftApConfiguration}.
 * When an application is done with the LocalOnlyHotspot, they should call {@link android.net.wifi.WifiManager.LocalOnlyHotspotReservation#close()  }.  Once this happens, the application will not receive
 * any further callbacks. If the LocalOnlyHotspot is stopped due to a
 * user triggered mode change, applications will be notified via the {@link android.net.wifi.WifiManager.LocalOnlyHotspotCallback#onStopped()  } callback.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class LocalOnlyHotspotReservation implements java.lang.AutoCloseable {

LocalOnlyHotspotReservation() { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@link android.net.wifi.WifiConfiguration WifiConfiguration} of the current Local Only Hotspot (LOHS).
 * May be null if hotspot enabled and security type is not
 * {@code WifiConfiguration.KeyMgmt.None} or {@code WifiConfiguration.KeyMgmt.WPA2_PSK}.
 *
 * @deprecated Use {@code WifiManager#getSoftApConfiguration()} to get the
 * LOHS configuration.
 */

@Deprecated
@android.annotation.Nullable
public android.net.wifi.WifiConfiguration getWifiConfiguration() { throw new RuntimeException("Stub!"); }

/**
 * Returns the {@link android.net.wifi.SoftApConfiguration SoftApConfiguration} of the current Local Only Hotspot (LOHS).

 * @return This value cannot be {@code null}.
 */

@android.annotation.NonNull
public android.net.wifi.SoftApConfiguration getSoftApConfiguration() { throw new RuntimeException("Stub!"); }

public void close() { throw new RuntimeException("Stub!"); }

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }
}

/**
 * Allows an application to receive Wifi Multicast packets.
 * Normally the Wifi stack filters out packets not explicitly
 * addressed to this device.  Acquring a MulticastLock will
 * cause the stack to receive packets addressed to multicast
 * addresses.  Processing these extra packets can cause a noticeable
 * battery drain and should be disabled when not needed.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class MulticastLock {

private MulticastLock() { throw new RuntimeException("Stub!"); }

/**
 * Locks Wifi Multicast on until {@link #release} is called.
 *
 * If this MulticastLock is reference-counted each call to
 * {@code acquire} will increment the reference count, and the
 * wifi interface will receive multicast packets as long as the
 * reference count is above zero.
 *
 * If this MulticastLock is not reference-counted, the first call to
 * {@code acquire} will turn on the multicast packets, but subsequent
 * calls will be ignored.  Only one call to {@link #release} will
 * be required, regardless of the number of times that {@code acquire}
 * is called.
 *
 * Note that other applications may also lock Wifi Multicast on.
 * Only they can relinquish their lock.
 *
 * Also note that applications cannot leave Multicast locked on.
 * When an app exits or crashes, any Multicast locks will be released.
 */

public void acquire() { throw new RuntimeException("Stub!"); }

/**
 * Unlocks Wifi Multicast, restoring the filter of packets
 * not addressed specifically to this device and saving power.
 *
 * If this MulticastLock is reference-counted, each call to
 * {@code release} will decrement the reference count, and the
 * multicast packets will only stop being received when the reference
 * count reaches zero.  If the reference count goes below zero (that
 * is, if {@code release} is called a greater number of times than
 * {@link #acquire}), an exception is thrown.
 *
 * If this MulticastLock is not reference-counted, the first call to
 * {@code release} (after the radio was multicast locked using
 * {@link #acquire}) will unlock the multicast, and subsequent calls
 * will be ignored.
 *
 * Note that if any other Wifi Multicast Locks are still outstanding
 * this {@code release} call will not have an immediate effect.  Only
 * when all applications have released all their Multicast Locks will
 * the Multicast filter be turned back on.
 *
 * Also note that when an app exits or crashes all of its Multicast
 * Locks will be automatically released.
 */

public void release() { throw new RuntimeException("Stub!"); }

/**
 * Controls whether this is a reference-counted or non-reference-
 * counted MulticastLock.
 *
 * Reference-counted MulticastLocks keep track of the number of calls
 * to {@link #acquire} and {@link #release}, and only stop the
 * reception of multicast packets when every call to {@link #acquire}
 * has been balanced with a call to {@link #release}.  Non-reference-
 * counted MulticastLocks allow the reception of multicast packets
 * whenever {@link #acquire} is called and stop accepting multicast
 * packets whenever {@link #release} is called.
 *
 * @param refCounted true if this MulticastLock should keep a reference
 * count
 */

public void setReferenceCounted(boolean refCounted) { throw new RuntimeException("Stub!"); }

/**
 * Checks whether this MulticastLock is currently held.
 *
 * @return true if this MulticastLock is held, false otherwise
 */

public boolean isHeld() { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }
}

/**
 * Abstract class for scan results callback. Should be extended by applications and set when
 * calling {@link android.net.wifi.WifiManager#registerScanResultsCallback(java.util.concurrent.Executor,android.net.wifi.WifiManager.ScanResultsCallback) WifiManager#registerScanResultsCallback(Executor, ScanResultsCallback)}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract static class ScanResultsCallback {

public ScanResultsCallback() { throw new RuntimeException("Stub!"); }

/**
 * Called when new scan results are available.
 * Clients should use {@link android.net.wifi.WifiManager#getScanResults() WifiManager#getScanResults()} to get the scan results.
 */

public abstract void onScanResultsAvailable();
}

/**
 * Abstract callback class for applications to receive updates about the Wi-Fi subsystem
 * restarting. The Wi-Fi subsystem can restart due to internal recovery mechanisms or via user
 * action.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public abstract static class SubsystemRestartTrackingCallback {

public SubsystemRestartTrackingCallback() { throw new RuntimeException("Stub!"); }

/**
 * Indicates that the Wi-Fi subsystem is about to restart.
 */

public abstract void onSubsystemRestarting();

/**
 * Indicates that the Wi-Fi subsystem has restarted.
 */

public abstract void onSubsystemRestarted();
}

/**
 * Interface for suggestion connection status listener.
 * Should be implemented by applications and set when calling
 * {@link android.net.wifi.WifiManager#addSuggestionConnectionStatusListener(java.util.concurrent.Executor,android.net.wifi.WifiManager.SuggestionConnectionStatusListener) WifiManager#addSuggestionConnectionStatusListener(
 * Executor, SuggestionConnectionStatusListener)}.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface SuggestionConnectionStatusListener {

/**
 * Called when the framework attempted to connect to a suggestion provided by the
 * registering app, but the connection to the suggestion failed.
 * @param wifiNetworkSuggestion The suggestion which failed to connect.
 * This value cannot be {@code null}.
 * @param failureReason the connection failure reason code. One of
 * {@link #STATUS_SUGGESTION_CONNECTION_FAILURE_ASSOCIATION},
 * {@link #STATUS_SUGGESTION_CONNECTION_FAILURE_AUTHENTICATION},
 * {@link #STATUS_SUGGESTION_CONNECTION_FAILURE_IP_PROVISIONING}
 * {@link #STATUS_SUGGESTION_CONNECTION_FAILURE_UNKNOWN}

 * Value is {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_CONNECTION_FAILURE_UNKNOWN}, {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_CONNECTION_FAILURE_ASSOCIATION}, {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_CONNECTION_FAILURE_AUTHENTICATION}, or {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_CONNECTION_FAILURE_IP_PROVISIONING}
 */

public void onConnectionStatus(@android.annotation.NonNull android.net.wifi.WifiNetworkSuggestion wifiNetworkSuggestion, int failureReason);
}

/**
 * Interface for network suggestion user approval status change listener.
 * Should be implemented by applications and registered using
 * {@link #addSuggestionUserApprovalStatusListener(java.util.concurrent.Executor,android.net.wifi.WifiManager.SuggestionUserApprovalStatusListener)} (
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface SuggestionUserApprovalStatusListener {

/**
 * Called when the user approval status of the App has changed.
 * @param status The current status code for the user approval. One of the
 *               {@code STATUS_SUGGESTION_APPROVAL_} values.

 * Value is {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_APPROVAL_UNKNOWN}, {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_APPROVAL_PENDING}, {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_APPROVAL_APPROVED_BY_USER}, {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_APPROVAL_REJECTED_BY_USER}, or {@link android.net.wifi.WifiManager#STATUS_SUGGESTION_APPROVAL_APPROVED_BY_CARRIER_PRIVILEGE}
 */

public void onUserApprovalStatusChange(int status);
}

/**
 * Allows an application to keep the Wi-Fi radio awake.
 * Normally the Wi-Fi radio may turn off when the user has not used the device in a while.
 * Acquiring a WifiLock will keep the radio on until the lock is released.  Multiple
 * applications may hold WifiLocks, and the radio will only be allowed to turn off when no
 * WifiLocks are held in any application.
 * <p>
 * Before using a WifiLock, consider carefully if your application requires Wi-Fi access, or
 * could function over a mobile network, if available.  A program that needs to download large
 * files should hold a WifiLock to ensure that the download will complete, but a program whose
 * network usage is occasional or low-bandwidth should not hold a WifiLock to avoid adversely
 * affecting battery life.
 * <p>
 * Note that WifiLocks cannot override the user-level "Wi-Fi Enabled" setting, nor Airplane
 * Mode.  They simply keep the radio from turning off when Wi-Fi is already on but the device
 * is idle.
 * <p>
 * Any application using a WifiLock must request the {@code android.permission.WAKE_LOCK}
 * permission in an {@code <uses-permission>} element of the application's manifest.
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public class WifiLock {

private WifiLock() { throw new RuntimeException("Stub!"); }

/**
 * Locks the Wi-Fi radio on until {@link #release} is called.
 *
 * If this WifiLock is reference-counted, each call to {@code acquire} will increment the
 * reference count, and the radio will remain locked as long as the reference count is
 * above zero.
 *
 * If this WifiLock is not reference-counted, the first call to {@code acquire} will lock
 * the radio, but subsequent calls will be ignored.  Only one call to {@link #release}
 * will be required, regardless of the number of times that {@code acquire} is called.
 */

public void acquire() { throw new RuntimeException("Stub!"); }

/**
 * Unlocks the Wi-Fi radio, allowing it to turn off when the device is idle.
 *
 * If this WifiLock is reference-counted, each call to {@code release} will decrement the
 * reference count, and the radio will be unlocked only when the reference count reaches
 * zero.  If the reference count goes below zero (that is, if {@code release} is called
 * a greater number of times than {@link #acquire}), an exception is thrown.
 *
 * If this WifiLock is not reference-counted, the first call to {@code release} (after
 * the radio was locked using {@link #acquire}) will unlock the radio, and subsequent
 * calls will be ignored.
 */

public void release() { throw new RuntimeException("Stub!"); }

/**
 * Controls whether this is a reference-counted or non-reference-counted WifiLock.
 *
 * Reference-counted WifiLocks keep track of the number of calls to {@link #acquire} and
 * {@link #release}, and only allow the radio to sleep when every call to {@link #acquire}
 * has been balanced with a call to {@link #release}.  Non-reference-counted WifiLocks
 * lock the radio whenever {@link #acquire} is called and it is unlocked, and unlock the
 * radio whenever {@link #release} is called and it is locked.
 *
 * @param refCounted true if this WifiLock should keep a reference count
 */

public void setReferenceCounted(boolean refCounted) { throw new RuntimeException("Stub!"); }

/**
 * Checks whether this WifiLock is currently held.
 *
 * @return true if this WifiLock is held, false otherwise
 */

public boolean isHeld() { throw new RuntimeException("Stub!"); }

public void setWorkSource(android.os.WorkSource ws) { throw new RuntimeException("Stub!"); }

public java.lang.String toString() { throw new RuntimeException("Stub!"); }

protected void finalize() throws java.lang.Throwable { throw new RuntimeException("Stub!"); }
}

/** Interface for callback invocation on a start WPS action
 * @deprecated This is deprecated
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
@Deprecated
public abstract static class WpsCallback {

@Deprecated
public WpsCallback() { throw new RuntimeException("Stub!"); }

/** WPS start succeeded
 * @deprecated This API is deprecated
 */

@Deprecated
public abstract void onStarted(java.lang.String pin);

/** WPS operation completed successfully
 * @deprecated This API is deprecated
 */

@Deprecated
public abstract void onSucceeded();

/**
 * WPS operation failed
 * @param reason The reason for failure could be one of
 * {@link #WPS_TKIP_ONLY_PROHIBITED}, {@link #WPS_OVERLAP_ERROR},
 * {@link #WPS_WEP_PROHIBITED}, {@link #WPS_TIMED_OUT} or {@link #WPS_AUTH_FAILURE}
 * and some generic errors.
 * @deprecated This API is deprecated
 */

@Deprecated
public abstract void onFailed(int reason);
}

}

