/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.app.sdksandbox;

import android.content.Context;
import java.util.concurrent.Executor;
import android.os.Bundle;
import android.os.OutcomeReceiver;

/**
 * Provides APIs to load {@link android.content.pm.SharedLibraryInfo#TYPE_SDK_PACKAGE SDKs}
 * into SDK sandbox process, and then interact with them.
 *
 * <p>{@code SdkSandbox} is a java process running in a separate uid range. Each app has its own
 * SDK sandbox process.
 *
 * <p>First app needs to declare {@code SDKs} it depends on in it's {@code AndroidManifest.xml}
 * using {@code <uses-sdk-library>} tag. App can only load {@code SDKs} it depends on into the
 * {@code SdkSandbox}.
 *
 * <p>Note: All APIs defined in this class are not stable and subject to change.
 *
 * @see android.content.pm.SharedLibraryInfo#TYPE_SDK_PACKAGE
 * @see
 * <a href="https://developer.android.com/design-for-safety/ads/sdk-runtime">SDK runtime design proposal</a>
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class SdkSandboxManager {

SdkSandboxManager() { throw new RuntimeException("Stub!"); }

/**
 * Returns current state of the {@code SdkSandbox}.

 * @return Value is {@link android.app.sdksandbox.SdkSandboxManager#SDK_SANDBOX_STATE_DISABLED}, or {@link android.app.sdksandbox.SdkSandboxManager#SDK_SANDBOX_STATE_ENABLED_PROCESS_ISOLATION}
 */

public static int getSdkSandboxState() { throw new RuntimeException("Stub!"); }

/**
 * Add a callback which gets registered for sdk sandbox lifecycle events, such as sdk sandbox
 * death. If the sandbox has not yet been created when this is called, the request will be
 * stored until a sandbox is created, at which point it is activated for that sandbox. Multiple
 * callbacks can be added to detect death.
 *
 * @param callbackExecutor the {@link java.util.concurrent.Executor Executor} on which to invoke the callback
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param callback the {@link android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback SdkSandboxProcessDeathCallback} which will receive sdk sandbox
 *     lifecycle events.

 * This value cannot be {@code null}.
 */

public void addSdkSandboxProcessDeathCallback(@android.annotation.NonNull java.util.concurrent.Executor callbackExecutor, @android.annotation.NonNull android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Remove an {@link android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback SdkSandboxProcessDeathCallback} that was previously added using {@link android.app.sdksandbox.SdkSandboxManager#addSdkSandboxProcessDeathCallback(java.util.concurrent.Executor,android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback)  }
 *
 * @param callback the {@link android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback SdkSandboxProcessDeathCallback} which was previously added using
 *     {@link android.app.sdksandbox.SdkSandboxManager#addSdkSandboxProcessDeathCallback(java.util.concurrent.Executor,android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback) SdkSandboxManager#addSdkSandboxProcessDeathCallback(Executor,
 *     SdkSandboxProcessDeathCallback)}

 * This value cannot be {@code null}.
 */

public void removeSdkSandboxProcessDeathCallback(@android.annotation.NonNull android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback callback) { throw new RuntimeException("Stub!"); }

/**
 * Load SDK in a SDK sandbox java process.
 *
 * <p>It loads SDK library with {@code sdkName} to a sandbox process asynchronously, caller
 * should be notified through {@code receiver}.
 *
 * <p>App should already declare {@code SDKs} it depends on in its {@code AndroidManifest} using
 * {@code <use-sdk-library>} tag. App can only load {@code SDKs} it depends on into the {@code
 * SdkSandbox}.
 *
 * <p>When client application loads the first SDK, a new {@code SdkSandbox} process will be
 * created, otherwise other SDKs will be loaded into the same sandbox which already created for
 * the client application.
 *
 * <p>This API may only be called while the caller is running in the foreground. Calls from the
 * background will result in a {@link java.lang.SecurityException SecurityException} being thrown.
 *
 * @param sdkName name of the SDK to be loaded.
 * This value cannot be {@code null}.
 * @param params additional parameters to be passed to the SDK in the form of a {@link android.os.Bundle Bundle}
 *     as agreed between the client and the SDK.
 * This value cannot be {@code null}.
 * @param executor the {@link java.util.concurrent.Executor Executor} on which to invoke the receiver.
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param receiver This either returns a {@link android.app.sdksandbox.SandboxedSdk SandboxedSdk} on a successful run, or {@link android.app.sdksandbox.LoadSdkException LoadSdkException}.

 * This value cannot be {@code null}.
 */

public void loadSdk(@android.annotation.NonNull java.lang.String sdkName, @android.annotation.NonNull android.os.Bundle params, @android.annotation.NonNull java.util.concurrent.Executor executor, @android.annotation.NonNull android.os.OutcomeReceiver<android.app.sdksandbox.SandboxedSdk,android.app.sdksandbox.LoadSdkException> receiver) { throw new RuntimeException("Stub!"); }

/**
 * Unloads an SDK that has been previously loaded by the caller.
 *
 * <p>It is not guaranteed that the memory allocated for this SDK will be freed immediately. All
 * subsequent calls to {@link #requestSurfacePackage(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)}
 * for the given {@code sdkName} will fail.
 *
 * <p>This API may only be called while the caller is running in the foreground. Calls from the
 * background will result in a {@link java.lang.SecurityException SecurityException} being thrown.
 *
 * @param sdkName name of the SDK to be unloaded.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if the SDK is not loaded.
 */

public void unloadSdk(@android.annotation.NonNull java.lang.String sdkName) { throw new RuntimeException("Stub!"); }

/**
 * Send a request for a surface package to the sdk.
 *
 * <p>After client application receives a signal about a successful SDK loading, and has added a
 * {@link android.view.SurfaceView} to the view hierarchy, it may asynchronously request a
 * {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage} to render a view from the SDK.
 *
 * <p>When the {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage} is ready, {@code onResult} function of the {@code
 * receiver} will be called with Bundle, that bundle will contain the key {@code
 * EXTRA_SURFACE_PACKAGE} with value present the requested {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage}.
 *
 * <p>This API may only be called while the caller is running in the foreground. Calls from the
 * background will result in a {@link java.lang.SecurityException SecurityException} being thrown.
 *
 * @param sdkName name of the SDK loaded into sdk sandbox.
 * This value cannot be {@code null}.
 * @param params the parameters which the client application passes to the SDK, it should
 *     contain the following params: (EXTRA_WIDTH_IN_PIXELS, EXTRA_HEIGHT_IN_PIXELS,
 *     EXTRA_DISPLAY_ID, EXTRA_HOST_TOKEN). If any of these params is missing, an
 *     IllegalArgumentException will be thrown. Any additional parameters may be passed as
 *     agreed between the client and the SDK.
 * This value cannot be {@code null}.
 * @param callbackExecutor the {@link java.util.concurrent.Executor Executor} on which to invoke the callback
 * This value cannot be {@code null}.
 * Callback and listener events are dispatched through this
 * {@link java.util.concurrent.Executor Executor}, providing an easy way to control which thread is
 * used. To dispatch events through the main thread of your
 * application, you can use
 * {@link android.content.Context#getMainExecutor() Context.getMainExecutor()}.
 * Otherwise, provide an {@link java.util.concurrent.Executor Executor} that dispatches to an appropriate thread.
 * @param receiver This either returns a {@link android.os.Bundle Bundle} on success which should contain the key
 *     EXTRA_SURFACE_PACKAGE with value of {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage} response, or {@link android.app.sdksandbox.RequestSurfacePackageException RequestSurfacePackageException} on failure.
 * This value cannot be {@code null}.
 * @throws java.lang.IllegalArgumentException if any of the following params (EXTRA_WIDTH_IN_PIXELS,
 *     EXTRA_HEIGHT_IN_PIXELS, EXTRA_DISPLAY_ID, EXTRA_HOST_TOKEN) are missing from the Bundle
 *     or passed with the wrong value or type.
 * @see android.app.sdksandbox.SdkSandboxManager#EXTRA_WIDTH_IN_PIXELS
 * @see android.app.sdksandbox.SdkSandboxManager#EXTRA_HEIGHT_IN_PIXELS
 * @see android.app.sdksandbox.SdkSandboxManager#EXTRA_DISPLAY_ID
 * @see android.app.sdksandbox.SdkSandboxManager#EXTRA_HOST_TOKEN
 */

public void requestSurfacePackage(@android.annotation.NonNull java.lang.String sdkName, @android.annotation.NonNull android.os.Bundle params, @android.annotation.NonNull java.util.concurrent.Executor callbackExecutor, @android.annotation.NonNull android.os.OutcomeReceiver<android.os.Bundle,android.app.sdksandbox.RequestSurfacePackageException> receiver) { throw new RuntimeException("Stub!"); }

/**
 * The name of key to be used in the Bundle fields of {@link #requestSurfacePackage(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)}, its value should define the integer ID of the logical
 * display to display the {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage}.
 */

public static final java.lang.String EXTRA_DISPLAY_ID = "android.app.sdksandbox.extra.DISPLAY_ID";

/**
 * The name of key to be used in the Bundle fields of {@link #requestSurfacePackage(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)}, its value should define the integer height of the {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage} in pixels.
 */

public static final java.lang.String EXTRA_HEIGHT_IN_PIXELS = "android.app.sdksandbox.extra.HEIGHT_IN_PIXELS";

/**
 * The name of key to be used in the Bundle fields of {@link #requestSurfacePackage(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)}, its value should present the token returned by {@link
 * android.view.SurfaceView#getHostToken()} once the {@link android.view.SurfaceView} has been
 * added to the view hierarchy. Only a non-null value is accepted to enable ANR reporting.
 */

public static final java.lang.String EXTRA_HOST_TOKEN = "android.app.sdksandbox.extra.HOST_TOKEN";

/**
 * The name of key in the Bundle which is passed to the {@code onResult} function of the {@link android.os.OutcomeReceiver OutcomeReceiver} which is field of {@link #requestSurfacePackage(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)}, its value presents the requested {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage}.
 */

public static final java.lang.String EXTRA_SURFACE_PACKAGE = "android.app.sdksandbox.extra.SURFACE_PACKAGE";

/**
 * The name of key to be used in the Bundle fields of {@link #requestSurfacePackage(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)}, its value should define the integer width of the {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage} in pixels.
 */

public static final java.lang.String EXTRA_WIDTH_IN_PIXELS = "android.app.sdksandbox.extra.WIDTH_IN_PIXELS";

/**
 * SDK is already loaded.
 *
 * <p>This indicates that client application tried to reload the same SDk by calling {@link android.app.sdksandbox.SdkSandboxManager#loadSdk(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)  } after being
 * successfully loaded.
 */

public static final int LOAD_SDK_ALREADY_LOADED = 101; // 0x65

/** Internal error while loading SDK.
 *
 * <p>This indicates a generic internal error happened while applying the call from
 * client application.
 */

public static final int LOAD_SDK_INTERNAL_ERROR = 500; // 0x1f4

/**
 * SDK not found.
 *
 * <p>This indicates that client application tried to load a non-existing SDK by calling {@link android.app.sdksandbox.SdkSandboxManager#loadSdk(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver)  }.
 */

public static final int LOAD_SDK_NOT_FOUND = 100; // 0x64

/**
 * SDK error after being loaded.
 *
 * <p>This indicates that the SDK encountered an error during post-load initialization. The
 * details of this can be obtained from the Bundle returned in {@link android.app.sdksandbox.LoadSdkException LoadSdkException} through
 * the {@link android.os.OutcomeReceiver OutcomeReceiver} passed in to {@link android.app.sdksandbox.SdkSandboxManager#loadSdk SdkSandboxManager#loadSdk}.
 */

public static final int LOAD_SDK_SDK_DEFINED_ERROR = 102; // 0x66

/**
 * SDK sandbox is disabled.
 *
 * <p>This indicates that the SDK sandbox is disabled. Any subsequent attempts to load SDKs in
 * this boot will also fail.
 */

public static final int LOAD_SDK_SDK_SANDBOX_DISABLED = 103; // 0x67

/** Internal error while requesting a {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage}.
 *
 * <p>This indicates a generic internal error happened while requesting a
 * {@link android.view.SurfaceControlViewHost.SurfacePackage SurfacePackage}.
 */

public static final int REQUEST_SURFACE_PACKAGE_INTERNAL_ERROR = 700; // 0x2bc

/**
 * Sdk sandbox process is not available.
 *
 * <p>This indicates that the sdk sandbox process is not available, either because it has died,
 * disconnected or was not created in the first place.
 */

public static final int SDK_SANDBOX_PROCESS_NOT_AVAILABLE = 503; // 0x1f7

/**
 * Use with {@link android.content.Context#getSystemService(java.lang.String) Context#getSystemService(String)} to retrieve a {@link android.app.sdksandbox.SdkSandboxManager SdkSandboxManager} for
 * interacting with the SDKs belonging to this client application.
 */

public static final java.lang.String SDK_SANDBOX_SERVICE = "sdk_sandbox";

/**
 * SDK Sandbox is disabled.
 *
 * <p>{@link android.app.sdksandbox.SdkSandboxManager SdkSandboxManager} APIs are hidden. Attempts at calling them will result in
 * {@link java.lang.UnsupportedOperationException UnsupportedOperationException}.
 */

public static final int SDK_SANDBOX_STATE_DISABLED = 0; // 0x0

/**
 * SDK Sandbox is enabled.
 *
 * <p>App can use {@link android.app.sdksandbox.SdkSandboxManager SdkSandboxManager} APIs to load {@code SDKs} it depends on into the
 * corresponding {@code SdkSandbox} process.
 */

public static final int SDK_SANDBOX_STATE_ENABLED_PROCESS_ISOLATION = 2; // 0x2
/**
 * A callback for tracking events SDK sandbox death.
 *
 * <p>The callback can be added using {@link android.app.sdksandbox.SdkSandboxManager#addSdkSandboxProcessDeathCallback(java.util.concurrent.Executor,android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback)  } and removed using {@link android.app.sdksandbox.SdkSandboxManager#removeSdkSandboxProcessDeathCallback(android.app.sdksandbox.SdkSandboxManager.SdkSandboxProcessDeathCallback)  }
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static interface SdkSandboxProcessDeathCallback {

/**
 * Notifies the client application that the SDK sandbox has died. The sandbox could die for
 * various reasons, for example, due to memory pressure on the system, or a crash in the
 * sandbox.
 *
 * The system will automatically restart the sandbox process if it died due to a crash.
 * However, the state of the sandbox will be lost - so any SDKs that were loaded previously
 * would have to be loaded again, using {@link android.app.sdksandbox.SdkSandboxManager#loadSdk(java.lang.String,android.os.Bundle,java.util.concurrent.Executor,android.os.OutcomeReceiver) SdkSandboxManager#loadSdk(String, Bundle,
 * Executor, OutcomeReceiver)} to continue using them.
 */

public void onSdkSandboxDied();
}

}

