/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package android.app.appsearch;

import java.util.Map;

/**
 * Encapsulates a request to update the schema of an {@link android.app.appsearch.AppSearchSession AppSearchSession} database.
 *
 * <p>The schema is composed of a collection of {@link android.app.appsearch.AppSearchSchema AppSearchSchema} objects, each of which
 * defines a unique type of data.
 *
 * <p>The first call to SetSchemaRequest will set the provided schema and store it within the {@link android.app.appsearch.AppSearchSession AppSearchSession} database.
 *
 * <p>Subsequent calls will compare the provided schema to the previously saved schema, to determine
 * how to treat existing documents.
 *
 * <p>The following types of schema modifications are always safe and are made without deleting any
 * existing documents:
 *
 * <ul>
 *   <li>Addition of new {@link android.app.appsearch.AppSearchSchema AppSearchSchema} types
 *   <li>Addition of new properties to an existing {@link android.app.appsearch.AppSearchSchema AppSearchSchema} type
 *   <li>Changing the cardinality of a property to be less restrictive
 * </ul>
 *
 * <p>The following types of schema changes are not backwards compatible:
 *
 * <ul>
 *   <li>Removal of an existing {@link android.app.appsearch.AppSearchSchema AppSearchSchema} type
 *   <li>Removal of a property from an existing {@link android.app.appsearch.AppSearchSchema AppSearchSchema} type
 *   <li>Changing the data type of an existing property
 *   <li>Changing the cardinality of a property to be more restrictive
 * </ul>
 *
 * <p>Providing a schema with incompatible changes, will throw an {@link
 * android.app.appsearch.exceptions.AppSearchException}, with a message describing the
 * incompatibility. As a result, the previously set schema will remain unchanged.
 *
 * <p>Backward incompatible changes can be made by :
 *
 * <ul>
 *   <li>setting {@link android.app.appsearch.SetSchemaRequest.Builder#setForceOverride SetSchemaRequest.Builder#setForceOverride} method to {@code true}. This
 *       deletes all documents that are incompatible with the new schema. The new schema is then
 *       saved and persisted to disk.
 *   <li>Add a {@link android.app.appsearch.Migrator Migrator} for each incompatible type and make no deletion. The migrator will
 *       migrate documents from it's old schema version to the new version. Migrated types will be
 *       set into both {@link android.app.appsearch.SetSchemaResponse#getIncompatibleTypes() SetSchemaResponse#getIncompatibleTypes()} and {@link android.app.appsearch.SetSchemaResponse#getMigratedTypes()        }. See the migration section below.
 * </ul>
 *
 * @see android.app.appsearch.AppSearchSession#setSchema
 * @see android.app.appsearch.Migrator
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public final class SetSchemaRequest {

SetSchemaRequest() { throw new RuntimeException("Stub!"); }

/** Returns the {@link android.app.appsearch.AppSearchSchema AppSearchSchema} types that are part of this request. */

@android.annotation.NonNull
public java.util.Set<android.app.appsearch.AppSearchSchema> getSchemas() { throw new RuntimeException("Stub!"); }

/**
 * Returns all the schema types that are opted out of being displayed and visible on any system
 * UI surface.
 */

@android.annotation.NonNull
public java.util.Set<java.lang.String> getSchemasNotDisplayedBySystem() { throw new RuntimeException("Stub!"); }

/**
 * Returns a mapping of schema types to the set of packages that have access to that schema
 * type.
 *
 * <p>It’s inefficient to call this method repeatedly.
 */

@android.annotation.NonNull
public java.util.Map<java.lang.String,java.util.Set<android.app.appsearch.PackageIdentifier>> getSchemasVisibleToPackages() { throw new RuntimeException("Stub!"); }

/**
 * Returns the map of {@link android.app.appsearch.Migrator Migrator}, the key will be the schema type of the {@link android.app.appsearch.Migrator Migrator}
 * associated with.
 */

@android.annotation.NonNull
public java.util.Map<java.lang.String,android.app.appsearch.Migrator> getMigrators() { throw new RuntimeException("Stub!"); }

/** Returns whether this request will force the schema to be overridden. */

public boolean isForceOverride() { throw new RuntimeException("Stub!"); }

/** Returns the database overall schema version. */

public int getVersion() { throw new RuntimeException("Stub!"); }
/** Builder for {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest} objects. */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public static final class Builder {

public Builder() { throw new RuntimeException("Stub!"); }

/**
 * Adds one or more {@link android.app.appsearch.AppSearchSchema AppSearchSchema} types to the schema.
 *
 * <p>An {@link android.app.appsearch.AppSearchSchema AppSearchSchema} object represents one type of structured data.
 *
 * <p>Any documents of these types will be displayed on system UI surfaces by default.
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder addSchemas(@android.annotation.NonNull android.app.appsearch.AppSearchSchema... schemas) { throw new RuntimeException("Stub!"); }

/**
 * Adds a collection of {@link android.app.appsearch.AppSearchSchema AppSearchSchema} objects to the schema.
 *
 * <p>An {@link android.app.appsearch.AppSearchSchema AppSearchSchema} object represents one type of structured data.
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder addSchemas(@android.annotation.NonNull java.util.Collection<android.app.appsearch.AppSearchSchema> schemas) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether or not documents from the provided {@code schemaType} will be displayed and
 * visible on any system UI surface.
 *
 * <p>This setting applies to the provided {@code schemaType} only, and does not persist
 * across {@link android.app.appsearch.AppSearchSession#setSchema AppSearchSession#setSchema} calls.
 *
 * <p>The default behavior, if this method is not called, is to allow types to be displayed
 * on system UI surfaces.
 *
 * @param schemaType The name of an {@link android.app.appsearch.AppSearchSchema AppSearchSchema} within the same {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest}, which will be configured.
 * @param displayed Whether documents of this type will be displayed on system UI surfaces.
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder setSchemaTypeDisplayedBySystem(@android.annotation.NonNull java.lang.String schemaType, boolean displayed) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether or not documents from the provided {@code schemaType} can be read by the
 * specified package.
 *
 * <p>Each package is represented by a {@link android.app.appsearch.PackageIdentifier PackageIdentifier}, containing a package name
 * and a byte array of type {@link android.content.pm.PackageManager#CERT_INPUT_SHA256}.
 *
 * <p>To opt into one-way data sharing with another application, the developer will need to
 * explicitly grant the other application’s package name and certificate Read access to its
 * data.
 *
 * <p>For two-way data sharing, both applications need to explicitly grant Read access to
 * one another.
 *
 * <p>By default, data sharing between applications is disabled.
 *
 * @param schemaType The schema type to set visibility on.
 * @param visible Whether the {@code schemaType} will be visible or not.
 * @param packageIdentifier Represents the package that will be granted visibility.
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder setSchemaTypeVisibilityForPackage(@android.annotation.NonNull java.lang.String schemaType, boolean visible, @android.annotation.NonNull android.app.appsearch.PackageIdentifier packageIdentifier) { throw new RuntimeException("Stub!"); }

/**
 * Sets the {@link android.app.appsearch.Migrator Migrator} associated with the given SchemaType.
 *
 * <p>The {@link android.app.appsearch.Migrator Migrator} migrates all {@link android.app.appsearch.GenericDocument GenericDocument}s under given schema type
 * from the current version number stored in AppSearch to the final version set via {@link
 * #setVersion}.
 *
 * <p>A {@link android.app.appsearch.Migrator Migrator} will be invoked if the current version number stored in AppSearch
 * is different from the final version set via {@link #setVersion} and {@link android.app.appsearch.Migrator#shouldMigrate  } returns {@code true}.
 *
 * <p>The target schema type of the output {@link android.app.appsearch.GenericDocument GenericDocument} of {@link android.app.appsearch.Migrator#onUpgrade  } or {@link android.app.appsearch.Migrator#onDowngrade Migrator#onDowngrade} must exist in this {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest}.
 *
 * @param schemaType The schema type to set migrator on.
 * @param migrator The migrator translates a document from its current version to the final
 *     version set via {@link #setVersion}.
 * @see android.app.appsearch.SetSchemaRequest.Builder#setVersion
 * @see SetSchemaRequest.Builder#addSchemas
 * @see android.app.appsearch.AppSearchSession#setSchema
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder setMigrator(@android.annotation.NonNull java.lang.String schemaType, @android.annotation.NonNull android.app.appsearch.Migrator migrator) { throw new RuntimeException("Stub!"); }

/**
 * Sets a Map of {@link android.app.appsearch.Migrator Migrator}s.
 *
 * <p>The key of the map is the schema type that the {@link android.app.appsearch.Migrator Migrator} value applies to.
 *
 * <p>The {@link android.app.appsearch.Migrator Migrator} migrates all {@link android.app.appsearch.GenericDocument GenericDocument}s under given schema type
 * from the current version number stored in AppSearch to the final version set via {@link
 * #setVersion}.
 *
 * <p>A {@link android.app.appsearch.Migrator Migrator} will be invoked if the current version number stored in AppSearch
 * is different from the final version set via {@link #setVersion} and {@link android.app.appsearch.Migrator#shouldMigrate  } returns {@code true}.
 *
 * <p>The target schema type of the output {@link android.app.appsearch.GenericDocument GenericDocument} of {@link android.app.appsearch.Migrator#onUpgrade  } or {@link android.app.appsearch.Migrator#onDowngrade Migrator#onDowngrade} must exist in this {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest}.
 *
 * @param migrators A {@link java.util.Map Map} of migrators that translate a document from it's current
 *     version to the final version set via {@link #setVersion}. The key of the map is the
 *     schema type that the {@link android.app.appsearch.Migrator Migrator} value applies to.
 * @see android.app.appsearch.SetSchemaRequest.Builder#setVersion
 * @see SetSchemaRequest.Builder#addSchemas
 * @see android.app.appsearch.AppSearchSession#setSchema
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder setMigrators(@android.annotation.NonNull java.util.Map<java.lang.String,android.app.appsearch.Migrator> migrators) { throw new RuntimeException("Stub!"); }

/**
 * Sets whether or not to override the current schema in the {@link android.app.appsearch.AppSearchSession AppSearchSession}
 * database.
 *
 * <p>Call this method whenever backward incompatible changes need to be made by setting
 * {@code forceOverride} to {@code true}. As a result, during execution of the setSchema
 * operation, all documents that are incompatible with the new schema will be deleted and
 * the new schema will be saved and persisted.
 *
 * <p>By default, this is {@code false}.
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder setForceOverride(boolean forceOverride) { throw new RuntimeException("Stub!"); }

/**
 * Sets the version number of the overall {@link android.app.appsearch.AppSearchSchema AppSearchSchema} in the database.
 *
 * <p>The {@link android.app.appsearch.AppSearchSession AppSearchSession} database can only ever hold documents for one version at
 * a time.
 *
 * <p>Setting a version number that is different from the version number currently stored in
 * AppSearch will result in AppSearch calling the {@link android.app.appsearch.Migrator Migrator}s provided to {@link android.app.appsearch.AppSearchSession#setSchema  } to migrate the documents already in AppSearch from the
 * previous version to the one set in this request. The version number can be updated
 * without any other changes to the set of schemas.
 *
 * <p>The version number can stay the same, increase, or decrease relative to the current
 * version number that is already stored in the {@link android.app.appsearch.AppSearchSession AppSearchSession} database.
 *
 * <p>The version of an empty database will always be 0. You cannot set version to the
 * {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest}, if it doesn't contains any {@link android.app.appsearch.AppSearchSchema AppSearchSchema}.
 *
 * @param version A positive integer representing the version of the entire set of schemas
 *     represents the version of the whole schema in the {@link android.app.appsearch.AppSearchSession AppSearchSession} database,
 *     default version is 1.
 * @throws java.lang.IllegalArgumentException if the version is negative.
 * @see android.app.appsearch.AppSearchSession#setSchema
 * @see android.app.appsearch.Migrator
 * @see android.app.appsearch.SetSchemaRequest.Builder#setMigrator
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest.Builder setVersion(int version) { throw new RuntimeException("Stub!"); }

/**
 * Builds a new {@link android.app.appsearch.SetSchemaRequest SetSchemaRequest} object.
 *
 * @throws java.lang.IllegalArgumentException if schema types were referenced, but the corresponding
 *     {@link android.app.appsearch.AppSearchSchema AppSearchSchema} type was never added.
 */

@android.annotation.NonNull
public android.app.appsearch.SetSchemaRequest build() { throw new RuntimeException("Stub!"); }
}

}

