/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * This file is available under and governed by the GNU General Public
 * License version 2 only, as published by the Free Software Foundation.
 * However, the following notice accompanied the original version of this
 * file:
 *
 * Written by Doug Lea with assistance from members of JCP JSR-166
 * Expert Group and released to the public domain, as explained at
 * http://creativecommons.org/publicdomain/zero/1.0/
 */


package java.util.concurrent;


/**
 * A service that decouples the production of new asynchronous tasks
 * from the consumption of the results of completed tasks.  Producers
 * {@code submit} tasks for execution. Consumers {@code take}
 * completed tasks and process their results in the order they
 * complete.  A {@code CompletionService} can for example be used to
 * manage asynchronous I/O, in which tasks that perform reads are
 * submitted in one part of a program or system, and then acted upon
 * in a different part of the program when the reads complete,
 * possibly in a different order than they were requested.
 *
 * <p>Typically, a {@code CompletionService} relies on a separate
 * {@link Executor} to actually execute the tasks, in which case the
 * {@code CompletionService} only manages an internal completion
 * queue. The {@link ExecutorCompletionService} class provides an
 * implementation of this approach.
 *
 * <p>Memory consistency effects: Actions in a thread prior to
 * submitting a task to a {@code CompletionService}
 * <a href="package-summary.html#MemoryVisibility"><i>happen-before</i></a>
 * actions taken by that task, which in turn <i>happen-before</i>
 * actions following a successful return from the corresponding {@code take()}.
 * @apiSince 1
 */

@SuppressWarnings({"unchecked", "deprecation", "all"})
public interface CompletionService<V> {

/**
 * Submits a value-returning task for execution and returns a Future
 * representing the pending results of the task.  Upon completion,
 * this task may be taken or polled.
 *
 * @param task the task to submit
 * @return a Future representing pending completion of the task
 * @throws RejectedExecutionException if the task cannot be
 *         scheduled for execution
 * @throws NullPointerException if the task is null
 * @apiSince 1
 */

public java.util.concurrent.Future<V> submit(java.util.concurrent.Callable<V> task);

/**
 * Submits a Runnable task for execution and returns a Future
 * representing that task.  Upon completion, this task may be
 * taken or polled.
 *
 * @param task the task to submit
 * @param result the result to return upon successful completion
 * @return a Future representing pending completion of the task,
 *         and whose {@code get()} method will return the given
 *         result value upon completion
 * @throws RejectedExecutionException if the task cannot be
 *         scheduled for execution
 * @throws NullPointerException if the task is null
 * @apiSince 1
 */

public java.util.concurrent.Future<V> submit(java.lang.Runnable task, V result);

/**
 * Retrieves and removes the Future representing the next
 * completed task, waiting if none are yet present.
 *
 * @return the Future representing the next completed task
 * @throws InterruptedException if interrupted while waiting
 * @apiSince 1
 */

public java.util.concurrent.Future<V> take() throws java.lang.InterruptedException;

/**
 * Retrieves and removes the Future representing the next
 * completed task, or {@code null} if none are present.
 *
 * @return the Future representing the next completed task, or
 *         {@code null} if none are present
 * @apiSince 1
 */

public java.util.concurrent.Future<V> poll();

/**
 * Retrieves and removes the Future representing the next
 * completed task, waiting if necessary up to the specified wait
 * time if none are yet present.
 *
 * @param timeout how long to wait before giving up, in units of
 *        {@code unit}
 * @param unit a {@code TimeUnit} determining how to interpret the
 *        {@code timeout} parameter
 * @return the Future representing the next completed task or
 *         {@code null} if the specified waiting time elapses
 *         before one is present
 * @throws InterruptedException if interrupted while waiting
 * @apiSince 1
 */

public java.util.concurrent.Future<V> poll(long timeout, java.util.concurrent.TimeUnit unit) throws java.lang.InterruptedException;
}

