package com.excelliance.open.reflect;

import java.lang.reflect.Method;
import java.util.HashMap;

/**
 * 为反射提供缓存
 */
public class Cache {

    private HashMap<ClassLoader, HashMap<String, Class>> sClassMap;
    private HashMap<Class, HashMap<String, Method>> sClassMethod;

    private static Cache sInstance;

    private Cache() {
        sClassMap = new HashMap();
        sClassMethod = new HashMap();
    }

    /**
     * 查找classloader加载的指定类名的类对象
     *
     * @param classLoader classloader
     * @param classname   类名(类全名)
     * @return class对象
     */
    public static Class findClass(ClassLoader classLoader, String classname) {
        return getInstance().findClass1(classLoader, classname);
    }

    /**
     * 查找类中定义的函数
     *
     * @param clazz   类对象, 必须是 @{{@link #findClass(ClassLoader, String)}} 返回的对象
     * @param method  函数名
     * @param argType 参数类型
     * @return 方法对象
     */
    public static Method findMethod(Class clazz, String method, Class... argType) {
        return getInstance().findMethod1(clazz, method, argType);
    }

    private static Cache getInstance() {
        Cache cache = sInstance;
        if (null != cache) {
            return cache;
        }

        synchronized (Cache.class) {
            cache = sInstance;
            if (null == cache) {
                cache = new Cache();
                sInstance = cache;
            }
        }
        return cache;
    }

    public Class findClass1(ClassLoader classLoader, String classname) {
        HashMap<String, Class> clMap = relocateClassMapByClassLoader(classLoader);
        Class clazz = clMap.get(classname);
        if (null != clazz) {
            return clazz;
        }

        synchronized (Cache.class) {
            try {
                clazz = classLoader.loadClass(classname);
            } catch (ClassNotFoundException ex) {
                throw new Error(ex);
            }
            clMap.put(classname, clazz);
        }
        return clazz;
    }

    private HashMap<String, Class> relocateClassMapByClassLoader(ClassLoader loader) {
        // TODO sClassMap 只增不减, 不加锁应该没有问题吧?
        HashMap<String, Class> item = sClassMap.get(loader);
        if (null != item) {
            return item;
        }
        synchronized (Cache.class) {
            item = new HashMap();
            sClassMap.put(loader, item);
        }
        return item;
    }

    private Method findMethod1(Class clazz, String method, Class... parameterTypes) {
        HashMap<String, Method> clMap = relocateMethodMapByClass(clazz);
        String signature = composeSignature(method, parameterTypes);
        Method m = clMap.get(signature);
        if (null != m) {
            return m;
        }

        synchronized (Cache.class) {
            try {
                m = clazz.getDeclaredMethod(method, parameterTypes);
            } catch (NoSuchMethodException ex) {
                throw new Error(ex);
            } catch (SecurityException ex) {
                throw new Error(ex);
            }
            m.setAccessible(true);
            clMap.put(signature, m);
        }
        return m;
    }

    private String composeSignature(String method, Class... argType) {
        StringBuilder builder = new StringBuilder();
        builder.append(method);
        builder.append('(');

        int count = null != argType ? argType.length : 0;
        for (int i = 0; i < count; i++) {
            Class temp = argType[i];
            builder.append(temp.getName());

            // 不是最后一个添加分隔符','
            if (i < count - 1) {
                builder.append(',');
            }
        }
        builder.append(')');
        return builder.toString();
    }

    private HashMap<String, Method> relocateMethodMapByClass(Class clazz) {
        HashMap<String, Method> item = sClassMethod.get(clazz);
        if (null != item) {
            return item;
        }
        synchronized (Class.class) {
            item = new HashMap();
            sClassMethod.put(clazz, item);
        }
        return item;
    }
}

