package com.excelliance.open.floatwindow;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ValueAnimator;
import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.graphics.PixelFormat;
import android.os.Build;
import android.os.CountDownTimer;
import android.os.Handler;
import android.os.Looper;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.Interpolator;

public abstract class BaseFloatWindow {
    /**
     * 悬浮球 坐落 左 右 标记
     */
    public static final int LEFT = 0;
    public static final int RIGHT = 1;

    /**
     * 记录 logo 停放的位置，以备下次恢复
     */
    private static final String XLOCATION = "XLocation";
    private static final String YLOCATION = "YLocation";

    private static final int TOUCH_SLOP = 8;

    /**
     * 停靠默认位置
     */
    private int mDefaultLocation = RIGHT;

    /**
     * 悬浮窗 坐落 位置
     */
    private int mHintLocation = mDefaultLocation;

    /**
     * 记录当前手指位置在屏幕上的横坐标值
     */
    private float mXInScreen;

    /**
     * 记录当前手指位置在屏幕上的纵坐标值
     */
    private float mYInScreen;

    /**
     * 记录手指按下时在屏幕上的横坐标的值
     */
    private float mXDownInScreen;

    /**
     * 记录手指按下时在屏幕上的纵坐标的值
     */
    private float mYDownInScreen;

    /**
     * 记录屏幕的宽度
     */
    private int mScreenWidth;

    /**
     * 记录屏幕的高度
     */
    private int mScreenHeight;

    /**
     * 来自 activity 的 wManager
     */
    private WindowManager mWindowManager;

    /**
     * 为 wManager 设置 LayoutParams
     */
    private WindowManager.LayoutParams wmParams;

    /**
     * 用于 定时 隐藏 logo的定时器
     */
    private CountDownTimer mHideTimer;

    private Handler mHandler = new Handler(Looper.getMainLooper());


    /**
     * 悬浮窗左右移动到默认位置 动画的 加速器
     */
    private Interpolator mLinearInterpolator = new AccelerateInterpolator();

    /**
     * 标记是否拖动中
     */
    private boolean mIsDraging = false;

    /**
     * 用于恢复悬浮球的location的属性动画值
     */
    private float mResetLocationPercent;

    protected Context mContext;
    protected Activity mActivity;

    /**
     * 悬浮窗中间logo 的 view
     */
    private View mLogoView;

    /**
     *
     */
    private ValueAnimator mValueAnimator = null;
    /**
     * 菜单是否打开
     */
    private boolean mIsOpenMenu = false;

    /**
     * 正在展示
     */
    private boolean mIsShowing = false;

    /**
     * down事件时悬浮x轴的值
     */
    private float mXDownOfLogo;

    /**
     * down事件时悬浮y轴的值
     */
    private float mYDownOfLogo;

    /**
     * 请求展示中, 当不满足展示条件时, {@link #show()} 被调用, 当满足条件后需要展示
     */
    private boolean requestShow;

    public BaseFloatWindow(Context context,Activity activity) {
        mContext = context;
        mActivity = activity;
        initFloatView();
        initTimer();
        initFloatWindow();
    }

    private void initFloatView() {
        LayoutInflater inflater = LayoutInflater.from(mContext);
        mLogoView = getLogoView(inflater);
        mLogoView.setOnTouchListener(mTouchListener);
    }

    /**
     * 初始化 隐藏悬浮球的定时器
     */
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    private void initTimer() {
        mHideTimer = new CountDownTimer(5000, 1000) {        //悬浮窗超过5秒没有操作的话会自动隐藏
            @Override
            public void onTick(long millisUntilFinished) {
                if (mIsOpenMenu) {
                    mHideTimer.cancel();
                }
            }

            @Override
            public void onFinish() {
                if (mIsOpenMenu) {
                    mHideTimer.cancel();
                    return;
                }
                if (!mIsDraging) {
                    if (mHintLocation == LEFT) {
                        mLogoView.setAlpha(0.5f);
                    } else {
                        mLogoView.setAlpha(0.5f);
                    }
                }
            }
        };
    }

    /**
     * 这个事件用于处理移动、自定义点击或者其它事情，return true可以保证onclick事件失效
     */
    private View.OnTouchListener mTouchListener = new View.OnTouchListener() {
        @Override
        public boolean onTouch(View v, MotionEvent event) {
//            Log.d("DEBUG", "onTouch: x: " + event.getX() + ", y: " + event.getY()
//                    + ", rawX: " + event.getRawX() + ", rawY: " + event.getRawY() + ", action: " + event.getAction());
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    floatEventDown(event);
                    break;
                case MotionEvent.ACTION_MOVE:
                    floatEventMove(event);
                    break;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    floatEventUp();
                    break;
            }
            return true;
        }
    };

    @TargetApi(Build.VERSION_CODES.HONEYCOMB_MR2)
    private void initFloatWindow() {
        if (null == wmParams) {
            initWindowManagerAndLayoutParams();
        }

        // 监听布局事件, 用于恢复悬浮标的停靠位置
        mActivity.getWindow().getDecorView().getViewTreeObserver().addOnGlobalLayoutListener(mLayoutListener);
    }

    /**
     * 处理 down 事件
     *
     * @param event
     */
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    private void floatEventDown(MotionEvent event) {
        mIsDraging = false;
        mHideTimer.cancel();
        mXDownInScreen = event.getRawX();
        mYDownInScreen = event.getRawY();
        mXInScreen = event.getRawX();
        mYInScreen = event.getRawY();
        mXDownOfLogo = wmParams.x;
        mYDownOfLogo = wmParams.y;
        mLogoView.setTranslationX(0);// 从缩进屏幕恢复正常
        mLogoView.setAlpha(1f);
    }

    /**
     * 处理 move 事件
     *
     * @param event
     */
    private void floatEventMove(MotionEvent event) {
        mIsDraging = true;
        mXInScreen = event.getRawX();
        mYInScreen = event.getRawY();

        final int deltaX = (int) (mXInScreen - mXDownInScreen);
        final int deltaY = (int) (mYInScreen - mYDownInScreen);
        int distance = (deltaX * deltaX) + (deltaY * deltaY);
        int slopSquare = TOUCH_SLOP * TOUCH_SLOP;

        if (distance > slopSquare) {
            updateScreenSize();
            wmParams.x = keepXInScreen(mXDownOfLogo + deltaX);
            wmParams.y = keepYInScreen(mYDownOfLogo + deltaY);
            updateViewPosition();
        }
    }

    /**
     * 处理 up 事件
     */
    @TargetApi(Build.VERSION_CODES.HONEYCOMB)
    private void floatEventUp() {
        if (mXInScreen <= mScreenWidth / 2) {
            mHintLocation = LEFT;
        } else {
            mHintLocation = RIGHT;
        }
        //手指离开屏幕后 悬浮球自动回到 屏幕两侧
        mValueAnimator = ValueAnimator.ofFloat(100);
        mValueAnimator.setInterpolator(mLinearInterpolator);
        mValueAnimator.setDuration(200);
        mValueAnimator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator animation) {
                float value = (float) animation.getAnimatedValue();
                mResetLocationPercent = value / 100f;
                mHandler.post(mUpdatePositionRunnable);
            }
        });

        mValueAnimator.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationCancel(Animator animation) {
                mIsDraging = false;
                mHideTimer.start();
            }

            @Override
            public void onAnimationEnd(Animator animation) {
                mIsDraging = false;
                saveFloatPosition();
                mHideTimer.start();
            }
        });

        if (!mValueAnimator.isRunning()) {
            mValueAnimator.start();
        }

        //这里需要判断如果如果手指所在位置和logo所在位置在一个宽度内则不移动,
        final int deltaX = (int) (mXInScreen - mXDownInScreen);
        final int deltaY = (int) (mYInScreen - mYDownInScreen);
        int distance = (deltaX * deltaX) + (deltaY * deltaY);
        int slopSquare = TOUCH_SLOP * TOUCH_SLOP;

        if (distance > slopSquare) {
            mIsDraging = false;
        } else {
            doOpen();
        }
    }

    /**
     * 手指离开屏幕后 用于恢复 悬浮球的 logo 的左右位置
     */
    private Runnable mUpdatePositionRunnable = new Runnable() {
        @Override
        public void run() {
            mIsDraging = true;
            reLocateViewPosition();
        }
    };

    /**
     * 手指离开屏幕后的重新定位
     */
    private void reLocateViewPosition() {
        if (wmParams.x > 0 && wmParams.x < mScreenWidth) {
            if (mHintLocation == LEFT) {
                float detalX = mXInScreen * mResetLocationPercent;
                detalX = (mXInScreen - detalX);
                wmParams.x = keepXInScreen(detalX);
            } else {
                float detalX = (mScreenWidth - mLogoView.getWidth() - mXInScreen);
                detalX = detalX * mResetLocationPercent;
                detalX += mXInScreen;
                wmParams.x = keepXInScreen(detalX);
            }
            wmParams.gravity = Gravity.LEFT | Gravity.TOP;
            updateViewPosition();
        }
        mIsDraging = false;
    }

    /**
     * 更新悬浮窗在屏幕中的位置。
     */
    private void updateViewPosition() {
        mIsDraging = true;
        try {
            if (!mIsOpenMenu) {
                mIsDraging = true;
//                Log.d("DEBUG", "onTouch: x: " + wmParams.x + ", y: " + wmParams.y);
                mWindowManager.updateViewLayout(mLogoView, wmParams);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public boolean isShowing() {
        return mIsShowing;
    }

    /**
     * 显示 悬浮窗
     */
    public void show() {
        if (isShowing()){
            return;
        }

        if (null == wmParams) {
            requestShow = true;
            return;
        }

        try {
            if (mWindowManager != null && wmParams != null && mLogoView != null) {
                mWindowManager.addView(mLogoView, wmParams);
                mIsShowing = true;
                restoreFloatPosition();
            }
            if (mHideTimer != null) {
                mHideTimer.start();
            } else {
                initTimer();
                mHideTimer.start();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        requestShow = false;
    }

    /**
     * 移除所有悬浮窗 释放资源
     */
    public void dismiss() {
        if (!isShowing()){
            return;
        }
        // 悬浮窗移动之后，记录位置, dismiss 不要记录了, 可能会记录错的值, 比如0
        // saveFloatPosition();
        mLogoView.clearAnimation();
        try {
            mHideTimer.cancel();
            mWindowManager.removeViewImmediate(mLogoView);
            mIsOpenMenu = false;
            mIsDraging = false;
            mIsShowing = false;
        } catch (Exception e) {
            e.printStackTrace();
        }
        requestShow = false;
    }

    /**
     * 用于保存悬浮球的位置记录
     *
     * @param key          String
     * @param defaultValue int
     * @return int
     */
    private int getSetting(String key, int defaultValue) {
        try {
            SharedPreferences sharedata = mContext.getSharedPreferences("floatLogo", 0);
            return sharedata.getInt(key, defaultValue);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return defaultValue;
    }

    /**
     * 用于保存悬浮球的位置记录
     *
     * @param key   String
     * @param value int
     */
    @TargetApi(Build.VERSION_CODES.GINGERBREAD)
    public void saveSetting(String key, int value) {
        try {
            SharedPreferences.Editor sharedata = mContext.getSharedPreferences("floatLogo", 0).edit();
            sharedata.putInt(key, value);
            sharedata.apply();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static int dp2Px(float dp, Context mContext) {
        return (int) TypedValue.applyDimension(
                TypedValue.COMPLEX_UNIT_DIP,
                dp,
                mContext.getResources().getDisplayMetrics());
    }

    protected abstract View getLogoView(LayoutInflater inflater);

    protected abstract void doOpen();

    /**
     * 保持x的值总是在屏幕(防止偏移出屏幕)
     * @param realX x轴的值
     * @return 限制内的值
     */
    private int keepXInScreen(float realX) {
        int rawX = (int) (realX + 0.5);
        if (rawX < 0) {
            rawX = 0;
        } else if (rawX + mLogoView.getWidth() > mScreenWidth) {
            rawX = mScreenWidth - mLogoView.getWidth();
        }
        return rawX;
    }

    /**
     * 保持y的值总是在屏幕(防止偏移出屏幕)
     * @param realY y轴的值
     * @return 限制内的值
     */
    private int keepYInScreen(float realY) {
        int rawY = (int) (realY + 0.5);
        if (rawY < 0) {
            rawY = 0;
        } else if (rawY + mLogoView.getHeight() > mScreenHeight) {
            rawY = mScreenHeight - mLogoView.getHeight();
        }
        return rawY;
    }

    /**
     * 更新屏幕的尺寸
     */
    private void updateScreenSize() {
        View decor = mActivity.getWindow().getDecorView();
        if (null != decor) {
            mScreenWidth = decor.getWidth();
            mScreenHeight = decor.getHeight();
        }
    }

    /**
     * activity view layout
     */
    private ViewTreeObserver.OnGlobalLayoutListener mLayoutListener = new ViewTreeObserver.OnGlobalLayoutListener() {

        @Override
        public void onGlobalLayout() {
//            Log.d("DEBUG", "onGlobalLayout");
            if (null == wmParams) {
                initWindowManagerAndLayoutParams();
            }

            // LayoutParams 创建成功; 且请求展示中, 需要调用show
            if (null != wmParams && requestShow) {
                show();
            }

            if (isShowing()) {
                restoreFloatPosition();
            }
        }
    };

    private void restoreFloatPosition() {
        // 恢复位置的时候不再从 decorView 获取flag信息, 获取的话可能导致输入法遮挡
        /*
        WindowManager.LayoutParams params = (WindowManager.LayoutParams) mActivity.getWindow().getDecorView().getLayoutParams();
        if (null != params) {
            wmParams.flags = params.flags;
            wmParams.flags |= WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE;
        }
         */
        
        updateScreenSize();
        
        // activity的decorView宽度和高度未布局出来前, 定位没有意义
        if (mScreenHeight <= 0 || mScreenWidth <= 0) {
            return;
        }
        
        mHintLocation = getSetting(XLOCATION, mDefaultLocation);
        if (mHintLocation == LEFT) {
            wmParams.x = 0;
        } else {
            wmParams.x = keepXInScreen(mScreenWidth - mLogoView.getWidth());
        }

        int screenHeight = mScreenHeight;
        int defaultY = ((screenHeight) / 2) / 3;
        int y = getSetting(YLOCATION, defaultY);
        wmParams.y = keepYInScreen(y);
        updateViewPosition();
    }

    /**
     * 当不再需要此对象时调用
     */
    public void destroy() {
        // 监听布局事件, 用于恢复悬浮标的停靠位置
        mActivity.getWindow().getDecorView().getViewTreeObserver().removeOnGlobalLayoutListener(mLayoutListener);
        dismiss();
    }

    /**
     * 保存悬浮窗位置
     */
    private void saveFloatPosition() {
        saveSetting(XLOCATION, mHintLocation);
        saveSetting(YLOCATION, wmParams.y);
    }

    private void initWindowManagerAndLayoutParams() {
        // 从activity的DecorView 复制 LayoutParams
        WindowManager.LayoutParams params = null;
        if (!(mActivity instanceof Activity)) {
            wmParams = new WindowManager.LayoutParams();
            mWindowManager = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                if (Build.VERSION.SDK_INT > 23) {
                    //在android7.1以上系统需要使用TYPE_PHONE类型 配合运行时权限
                    wmParams.type = WindowManager.LayoutParams.TYPE_PHONE;
                } else {
                    wmParams.type = WindowManager.LayoutParams.TYPE_TOAST;
                }
            } else {
                wmParams.type = WindowManager.LayoutParams.TYPE_PHONE;
            }
        } else if (null != (params = (WindowManager.LayoutParams)mActivity.getWindow().getDecorView().getLayoutParams())) {
            Activity activity = (Activity) mActivity;
            mWindowManager = activity.getWindowManager();
            wmParams = new WindowManager.LayoutParams();
            wmParams.copyFrom(params);
            wmParams.flags |= WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE;
            //类似dialog，寄托在activity的windows上,activity关闭时需要关闭当前float
            wmParams.type = WindowManager.LayoutParams.TYPE_APPLICATION;
        }

        if (null != wmParams) {
            wmParams.format = PixelFormat.RGBA_8888;
            wmParams.flags |= WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE;
            wmParams.alpha = 1;
            wmParams.width = WindowManager.LayoutParams.WRAP_CONTENT;
            wmParams.height = WindowManager.LayoutParams.WRAP_CONTENT;
            wmParams.gravity = Gravity.LEFT | Gravity.TOP;
        }
    }
}
