package com.excelliance.open;

import android.app.Activity;
import android.app.Dialog;
import android.util.Log;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * 弹框守卫, 为dialog弹出时机分级别和先后控制
 * 1. 级别高的弹框优先弹出, 存在低级别的先隐藏, 等高级别的被用户处理后再弹出
 * 2. 同级别的弹框按先入先弹出顺序处理
 */
public class PopupGuard {
    private static final String TAG = "PopupGuard";

    /**
     * 弹框优先级级别, 信息/进度/状态(一般都不需要用户确认)
     */
    public static final int PRIORITY_INFO = 1;

    /**
     * 弹框优先级级别, 确认(需要用户主动确认)
     */
    public static final int PRIORITY_CONFIRM = 100;

    /**
     * 弹框优先级级别, 立即(级别较高, 需要用户立即确认)
     */
    public static final int PRIORITY_SEASONABLE = 1000;

    /**
     * 记录“正在”弹出的弹框, 从后向前依次弹出
     */
    private List<PriorityItem> mPopupItems = new ArrayList<>();


    private HashMap<Dialog, Activity> mDialogActivity;

    /**
     * 使用优先级方式弹出对话框<br/>
     * 如果当前有级别更高的弹框, 指定的弹框将在高级别弹框消失后弹出
     * @param dialog 弹框
     * @param priority 优先级
     * @return 指定弹框是否弹出; true, 弹出; false, 稍后弹出
     */
    public boolean show(AliveDialogLayer dialog, int priority, Activity activity) {
        if (hasInQueue(dialog)) {
            return false;
        }

        int size = null != mPopupItems ? mPopupItems.size() : 0;
        int newPos = reposition(priority);

        // 新加入的弹框在最后一个, dismissing 当前弹框
        if (newPos >= size) {
            PriorityItem item = lastDialog();
            if (null != item) {
                dismissRealDialog(item.dialog);
            }
        }

        // 加入到队列中
        PriorityItem newItem = new PriorityItem();
        newItem.dialog = dialog;
        newItem.priority = priority;
        newItem.activity = activity;
        mPopupItems.add(newPos, newItem);

        // 新加入的弹框在最后一个, 需要弹出来
        if (newPos >= size) {
            showRealDialog(dialog, activity);
            return true;
        }
        return false;
    }

    /**
     * 使用优先级方式关闭对话框<br/>
     * 指定的弹框必须是通过 @{@link #show(AliveDialogLayer, int, Activity)} 弹出的.<br/>
     * 如果当前弹框就是指定的弹框将关闭, 不是但在队列里也将视为关闭成功
     * @param dialog 弹框
     * @return 关闭是否成功。 true, 成功; false 不在队列中
     */
    public boolean dismiss(AliveDialogLayer dialog) {
        int size = null != mPopupItems ? mPopupItems.size() : 0;

        boolean isDismiss = false;
        Activity activity = null;
        for (int i = size - 1; i >= 0; i--) {
            PriorityItem item = mPopupItems.get(i);
            if (dialog == item.dialog) {
                dismissRealDialog(dialog);
                mPopupItems.remove(item);
                isDismiss = true;
                activity = item.activity;
                break;
            }
        }
        if (isDismiss) {
            return popupDialogIfRemain(activity);
        }

        // 没有真正的去dismissing dialog, 就不要弹出后续
        return false;
    }

    /**
     * 表示是否当前有对话框是展示的
     * @return true, 有; false, 没有
     */
    public boolean isShowing() {
        return null != mPopupItems ? !mPopupItems.isEmpty() : false;
    }

    private static class PriorityItem {
        private AliveDialogLayer dialog;
        private int priority;
        private Activity activity;
    }

    private PriorityItem lastDialog() {
        int size = null != mPopupItems ? mPopupItems.size() : 0;
        if (size > 0) {
            return mPopupItems.get(size - 1);
        }
        return null;
    }

    /**
     * 定位新入弹框应该投放的位置
     * @param priority 弹框级别
     * @return 位置索引
     */
    public int reposition(int priority) {
        int size = null != mPopupItems ? mPopupItems.size() : 0;
        int newPos = size;
        for (int i = size - 1; i >= 0; i--) {
            PriorityItem item = mPopupItems.get(i);
            int itemPriority = item.priority;
            if (itemPriority < priority) {
                break;
            } else {
                // 等于的情况，按先入先展示和小于的逻辑是一样的
                newPos = i;
            }
        }
        return newPos;
    }

    /**
     * 在当前dialog关闭时调用, 如果还是剩余的dialog, 把最后一个弹出来
     * @return true, 又弹出了一个; false, 没有弹框了
     */
    private boolean popupDialogIfRemain(Activity activity) {
        PriorityItem item = lastDialog();
        if (null != item) {
            showRealDialog(item.dialog, activity);
            return true;
        }
        return false;
    }

    private void dismissRealDialog(AliveDialogLayer dialog) {
        if (dialog != null) {
            try {
                dialog.dismissFromPopupGuard();
            } catch (Exception e){
                e.printStackTrace();
                Log.d(TAG, "hideProgress exception");
            }
        }
    }

    private void showRealDialog(AliveDialogLayer dialog, Activity activity) {
        if (dialog != null) {
            dialog.show(this, activity);
        }
    }

    /**
     * dialog 是否已经存在队列中
     * @param dialog 指定dialog
     * @return true, 已经在队列中
     */
    private boolean hasInQueue(AliveDialogLayer dialog) {
        int size = null != mPopupItems ? mPopupItems.size() : 0;
        for (int i = 0; i < size; i++) {
            PriorityItem item = mPopupItems.get(i);
            if (item.dialog == dialog) {
                return true;
            }
        }
        return false;
    }

    private boolean associateDialogActivity(Dialog dialog, Activity activity) {
        HashMap<Dialog, Activity> map = mDialogActivity;
        if (null == map) {
            map = new HashMap<>();
            mDialogActivity = map;
        }
        map.put(dialog, activity);
        return true;
    }


    private Activity getDialogActivity(Dialog dialog) {
        HashMap<Dialog, Activity> map = mDialogActivity;
        if (null == map) {
            return null;
        }
        return map.get(dialog);
    }

    private boolean disassociateDialogActivity(Dialog dialog, Activity activity) {
        HashMap<Dialog, Activity> map = mDialogActivity;
        if (null != map) {
            return activity == map.remove(dialog);
        }
        return false;
    }
}
