package com.excelliance.open;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;

/**
 * 对Dialog和Dialog.Builder的封装, 主要解决Activity切换时Dialog无法正常弹出或者崩溃问题。
 * 保活的Dialog。
 */
public class AliveDialogLayer implements DialogInterface {

    /**
     * Helper class for creating dialog
     */
    public abstract static class Builder {
        private AliveDialogLayer mAttach;

        public Builder(Context context) {

        }

        /**
         * 表示构建者的内容发生了变化
         */
        public void invalidate() {
            AliveDialogLayer layer = mAttach;
            if (null != layer) {
                layer.invalidate();
            }
        }

        /**
         * 当需要创建一个新的dialog时调用<br/>
         * 派生类去实现 {@link #onCreate(Activity)} 函数
         */
        public final Dialog create(Activity activity) {
            return onCreate(activity);
        }

        /**
         * 依据构造参数创建即将展示在指定activity上的dialog对象<br/>
         * 注意: 不要缓存dialog对象
         * @param activity 展示dialog的activity
         * @return dialog对象
         */
        protected abstract Dialog onCreate(Activity activity);

        /**
         * 绑定到指定AliveDialogLayer时调用
         * @param layer 展示的AliveDialogLayer
         */
        protected void attach(AliveDialogLayer layer) {
            mAttach = layer;
        }
        
        protected AliveDialogLayer getAttachLayer() {
            return mAttach;
        }
    }

    private Builder dataBuilder;
    private Context context;

    private Dialog topDialog;
    private Activity topActivity;

    /**
     * true, 表示从最后一次展示dialog, 数据有更新。
     */
    private boolean isInvalidate;

    /**
     * 当 layer 是通过 {@link PopupGuard} 弹出时, 此对象有值
     */
    private PopupGuard mAttachGuard;

    public AliveDialogLayer(Context context) {
        this(context, null);
    }

    public AliveDialogLayer(Context context, Builder builder) {
        this.context = context;
        if (null == builder) {
            builder = new CustomDialog.Builder(context);
        }
        dataBuilder = builder;
    }

    /**
     * 在指定的Activity之上展示dialog layer.
     * @param activity 界面
     */
    public void show(Activity activity) {
        if (activity.isFinishing()) {
            return;
        }
        boolean recreate = false;
        Dialog cd = this.topDialog;

        // 未创建或者activity变化
        if (null == cd || topActivity != activity) {
            recreate = true;
            isInvalidate = false;
        } else if (isInvalidate) {
            // dialog内容有变化
            recreate = true;
        }

        if (recreate) {
            // 旧的dialog需要隐藏
            if (null != cd && cd.isShowing()) {
                dismiss(cd);
            }
            topActivity = activity;
            Builder builder = getDialogBuilder();
            cd = topDialog = builder.create(activity);
            builder.attach(this);
        }

        // 展示dialog
        cd.show();
    }

    /**
     * @return Whether the dialog is currently showing.
     */
    public boolean isShowing() {
        Dialog cd = this.topDialog;
        return null != cd && cd.isShowing();
    }

    /**
     * 获取dialog对象构造器
     * @return builder对象
     */
    public Builder getDialogBuilder() {
        return dataBuilder;
    }

    /**
     * Invalidate the whole dialog.
     */
    protected void invalidate() {
        if (null != topDialog) {
            isInvalidate = true;
        }
    }

    /**
     * 关闭dialog层
     */
    @Override
    public void dismiss() {
        if (null != mAttachGuard) {
            mAttachGuard.dismiss(this);
        } else {
            dismissFromPopupGuard();
        }
    }

    private void dismiss(Dialog cd) {
        if (null != cd && cd.isShowing()) {
            try {
                cd.dismiss();
            } catch (Exception ex) {
                // do nothing
            }
        }
    }

    @Override
    public void cancel() {
        //TODO 暂时不需要做什么
    }

    /**
     * 通过{@link PopupGuard} 在指定的Activity之上展示dialog layer.
     * @param popupGuard 弹出守护
     * @param activity 界面
     */
    public void show(PopupGuard popupGuard, Activity activity) {
        if (activity.isFinishing()) {
            return;
        }
        show(activity);
        mAttachGuard = popupGuard;
    }
    
    void dismissFromPopupGuard() {
        Dialog cd = this.topDialog;
        if (null != cd) {
            cd.dismiss();
        }
    }
}
