package com.excelliance.open.reflect;

import android.content.Context;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * 反射器。代表类或者类的一个实例<br/>
 * 1. 构造对象时, 传递类名和可选context, context的目的是提供classloader。
 * 2. 为了速度, 查找函数时加了缓存。
 * 3. 表示实例时, 此对象的实例等同于指定类的一个实例, 调用@{{@link #newReceiver(Class[], Object...)}},
 * 之用调用的动态实例函数, 将使用它创建的对象实例作为this。
 */
public class Reflector {
	private ClassLoader classLoader;
	private String className;
	private Class reflectClass;
	private Object receiver;

	/**
	 * 用类名构造反射器对象
	 * @param className 全名(如：java.lang.String)。
	 */
	public Reflector(String className) {
		this(className, Reflector.class.getClassLoader());
	}

	/**
	 * 用类名和classloader对象构造反射器
	 * @param className 类全名(如：java.lang.String)。
	 * @param classLoader 类加载器对象
	 */
	public Reflector(String className, ClassLoader classLoader) {
		this.className = className;
		this.classLoader = classLoader;
	}

	/**
	 * 用类名构造反射器对象, 将使用context的classloader来查找类名对应的类。
	 * @param className 全名(如：java.lang.String)。
	 * @param context 提供classloader
	 */
	public Reflector(String className, Context context) {
		this(className, context.getClassLoader());
	}

	private Class reflectClass() {
		Class clazz = reflectClass;
		if (null != clazz) {
			return clazz;
		}
		clazz = Cache.findClass(classLoader, className);
		reflectClass = clazz;
		return clazz;
	}

	private Method reflectMethod(String method, Object... args) {
		Class[] argsType = null;
		if (null != args && args.length > 0) {
			argsType = new Class[args.length];
			for (int i = 0; i < args.length; i++) {
				argsType[i] = args[i].getClass();
			}
		}
		Method m = reflectMethod(method, argsType);
		return m;
	}

	/**
	 * 调用类声明/定义的静态函数或者非静态函数。
	 * 调用非静态函数this对象是reveiver, 需要调用newReceiver()之后调用; 而调用非静态函数没有限制
	 * @param method 函数名
	 * @param args 参数
	 * @return 函数返回值
	 */
	public Object invoke(String method, Object... args) {
		Method m = reflectMethod(method, args);
		return this.invoke(receiver, m, args);
	}

	/**
	 * 调用类声明/定义的非静态函数。
	 * @param receiver 函数接收者, 非静态函数的this对象
	 * @param method 函数名
	 * @param args 参数
	 * @return 函数返回值
	 */
	public Object invoke(Object receiver, String method, Object... args) {
		Method m = reflectMethod(method, args);
		return invoke(receiver, m, args);
	}

	/**
	 * 调用类声明/定义的静态函数或者非静态函数。
	 * 调用非静态函数this对象是reveiver, 需要调用newReceiver()之后调用; 而调用非静态函数没有限制
	 * @param method 函数名
	 * @param parameterTypes 函数参数的类型
	 * @param args 函数参数
	 * @return 函数返回值
	 */
	public Object invoke(String method, Class[] parameterTypes, Object... args) {
		Method m = reflectMethod(method, parameterTypes);
		return invoke(receiver, m, args);
	}

	private Method reflectMethod(String method, Class... parameterTypes) {
		Class clazz = reflectClass();
		Method m = Cache.findMethod(clazz, method, parameterTypes);
		return m;
	}

	private Object invoke(Object receiver, Method method, Object... args) {
		Class clazz = reflectClass();
		Method m = method;
		try {
			if (null == receiver) {
				return m.invoke(clazz, args);
			} else {
				return m.invoke(receiver, args);
			}
		} catch (IllegalAccessException ex) {
			throw new Error(ex);
		} catch (IllegalArgumentException ex) {
			throw new Error(ex);
		} catch (InvocationTargetException ex) {
			throw new Error(ex);
		}
	}

	/**
	 * 通过构造函数创建类实例。(new 实例)
	 * @param parameterTypes 构造函数参数类型
	 * @param args 构造函数参数
	 * @return 实例
	 */
	public Object newInstance(Class[] parameterTypes, Object... args) {
		Class clazz = reflectClass();
		try {
			Constructor constructor = clazz.getConstructor(parameterTypes);
			return constructor.newInstance(args);
		} catch (NoSuchMethodException ex) {
		        throw new Error(ex);
		} catch (Exception ex) {
		        throw new Error(ex);
		}
	}

	/**
	 * 创建receiver对象, 创建之后此对象将代码类的实例. <br/>
	 * 注意: newReceiver各重载函数只能调用一次, 多次调用将抛出异常
	 * @param parameterTypes 构造函数参数类型
	 * @param args 构造函数参数
	 */
	public void newReceiver(Class[] parameterTypes, Object... args) {
		if (null != receiver) {
			throw new IllegalStateException("must call newReceiver only once");
		}
		receiver = newInstance(parameterTypes, args);
	}

	/**
	 * 调用类声明/定义的非静态函数。
	 * @param receiver 函数接收者, 非静态函数的this对象
	 * @param method 函数名
	 * @param args 参数
	 * @return 函数返回值
	 */
	public Object invoke(Object receiver, String method, Class[] parameterTypes, Object... args) {
		Method m = reflectMethod(method, parameterTypes);
		return invoke(receiver, m, args);
	}

	/**
	 * 创建receiver对象通过调用无参的构造函数, 创建之后此对象将代码类的实例. <br/>
	 * 注意: newReceiver各重载函数只能调用一次, 多次调用将抛出异常
	 */
	public void newReceiver() {
		if (null != receiver) {
			throw new IllegalStateException("must call newReceiver only once");
		}
		receiver = newInstance();
	}

	/**
	 * 通过无参构造函数创建类实例。(new 实例)
	 * @return 实例
	 */
	public Object newInstance() {
		Class clazz = reflectClass();
		try {
			Constructor ctor = clazz.getDeclaredConstructor();
			ctor.setAccessible(true);
			return ctor.newInstance();
//			return clazz.newInstance();
		} catch (NoSuchMethodException e) {
			throw new Error(e);
		} catch (IllegalAccessException e) {
			throw new Error(e);
		} catch (InstantiationException e) {
			throw new Error(e);
		} catch (InvocationTargetException e) {
			throw new Error(e);
		}
	}

	/**
	 * 获取通过 {@link #newReceiver()} 及其重载函数创建的receiver对象. <br/>
	 * return null, 表示未调用 {@link #newReceiver()} 及相关重载函数
	 * @return 本实例代表的类实例对象.
	 */
	public Object getReceiver() {
		return receiver;
	}

	/**
	 * 获取代表类的类对象(Class对象)
	 * @return 类对象
	 */
	public Class getReflectClass() {
		return reflectClass();
	}
}
